/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Outer.java 53 2007-12-14 13:41:27Z tokmensa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStreamWriter;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.LinkedList;
import java.util.List;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * A pipe consists of a pair of channels: a writable channel (called "sink",
 * where information is written to) and a readable channel (called "source",
 * where information is obtained from).
 *
 * The general idea is that once some bytes are available on the source:
 *
 * 1. The pipe reads those bytes. 2. If required, does some action with some or
 * all of these bytes (for example, write them to a file). 3. If a sink is
 * present, writes the exact same bytes in the exact same order on the sink.
 *
 * An Outer instance can retrieve information from (=use as source) a
 * PrintStream or a File and outputs information to (=uses as sink) a
 * PipedOutputStream.
 *
 * The main difference between an Outer and an Iner is that an Outer can also
 * filter out (@see Outer.setRegexp and Outer.setTimeFilter) information and
 * output it on a target. Note that the sink still keeps on receiving unmodified
 * data from the source.
 */
public class Outer implements Runnable {
    /**
     * Creates an Outer using a PipedOutputStream as source and a PrintStream as
     * output. A dummy sink (accessible via {@link Outer#getSink()} and that can
     * be used an input to any pipe) will be created by default.
     *
     * @see Outer#setSink(PipedOutputStream)
     *
     * @param src PipedOutputStream to use as source.
     * @param target PrintStream to use as target (can be null).
     *
     * @throws IOException If construction or pipe connection fails.
     */
    public Outer(final PipedOutputStream src, final PrintStream target) throws IOException {
        // Source
        this.source = new PipedInputStream(src);
        this.rsource = new LineNumberReader(new InputStreamReader(this.source));

        // Target
        if (target != null) {
            this.target = new PrintWriter(target, /* autoflush */true);
        } else {
            this.target = null;
        }

        // Sink
        this.setSink(null);
    }

    /**
     * Creates an Outer using a PipedOutputStream as source and a File as
     * output. A dummy sink (accessible via {@link Outer#getSink()} and that can
     * be used an input to any pipe) will be created by default.
     *
     * @see Outer#setSink(PipedOutputStream)
     *
     * @param src PipedOutputStream to use as source.
     * @param file File to use as output.
     *
     * @throws IOException If construction or pipe connection fails.
     */
    public Outer(final PipedOutputStream src, final File file) throws IOException {
        // Source
        this.source = new PipedInputStream(src);
        this.rsource = new LineNumberReader(new InputStreamReader(this.source));

        // Target
        this.target = new PrintWriter(new FileWriter(file), /* autoflush */true);

        // Sink
        this.setSink(null);
    }

    /**
     * @param sink Sink to set. A dummy sink will be created if null.
     */
    public void setSink(final PipedOutputStream sink) {
        if (sink == null) {
            this.setSink(new PipedOutputStream());
        } else {
            if (this.wsink != null) {
                try {
                    this.wsink.close();
                    this.sink.close();
                } catch (Exception e) {
                    this.logger.warn("Error while closing sink : {0}", e);
                }
            }
            this.sink = sink;
            this.wsink = new PrintWriter(new OutputStreamWriter(sink), /* autoflush */true);
        }
    }

    /**
     * @return The sink currently used.
     */
    public PipedOutputStream getSink() {
        return this.sink;
    }

    /**
     * @return The id set.
     */
    public String getId() {
        return this.id;
    }

    /**
     * @param id The id to set.
     */
    public void setId(final String id) {
        this.id = id;
    }

    /**
     * Tests the Outer implementation.
     *
     * @param args Ignored.
     *
     * @throws IOException When there's a problem with the source or sink.
     */
    public static void main(final String[] args) throws IOException {
        PipedOutputStream src = new PipedOutputStream();
        PrintWriter wsrc = new PrintWriter(src);

        Outer outer = new Outer(src, System.out);
        Outer.setSeparator("\t");
        String[] format = new String[] {"time", "a", "c", "f", "name"};
        outer.setFormatter(format);
        outer.setRegexp(".*:*type=bbb,.*");

        Outer outer2 = new Outer(outer.getSink(), new File("toto.log"));
        outer2.setRegexp(".*aaa.*name=n0.$");

        new Thread(outer2).start();
        new Thread(outer).start();

        String[] names = new String[] {"org.ow2.jasmine.monitoring.mbeancmd:type=aaa,name=n00",
                                       "org.ow2.jasmine.monitoring.mbeancmd:type=bbb,name=n01",
                                       "org.ow2.jasmine.monitoring.mbeancmd:type=aaa,name=n02",
                                       "org.ow2.jasmine.monitoring.mbeancmd:type=aaa,name=n03",
                                       "org.ow2.jasmine.monitoring.mbeancmd:type=bbb,name=n04",
                                       "org.ow2.jasmine.monitoring.mbeancmd:type=bbb,name=n05",
                                       "org.ow2.jasmine.monitoring.mbeancmd:type=aaa,name=n06",
                                       "org.ow2.jasmine.monitoring.mbeancmd:type=aaa,name=n07",
                                       "org.ow2.jasmine.monitoring.mbeancmd:type=aaa,name=n08",
                                       "org.ow2.jasmine.monitoring.mbeancmd:type=bbb,name=n09",
                                       "org.ow2.jasmine.monitoring.mbeancmd:type=bbb,name=n10"};

        wsrc.println("time\ta\tb\tc\td\te\tf\tname");
        for (int i = 1; i <= 10; i++) {
            wsrc.println("" + i + "\t" + (100 + i) + "\t" + (200 + i) + "\t" + (300 + i) + "\t" + (400 + i) + "\t" + (500 + i)
                    + "\t" + (600 + i) + "\t" + names[i - 1]);
            wsrc.flush();
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        src.close();
        try {
            Thread.sleep(5000);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }

        System.out.println("org.ow2.jasmine.monitoring.mbeancmd.Outer.main has completed.");
    }

    /**
     * Starts the activity on the Outer (as described in the class header).
     */
    public void run() {
        boolean goOn = true;
        long count = 0;

        while (goOn) {
            try {
                String ln = this.rsource.readLine();
                if (ln == null) {
                    goOn = false;
                    continue;
                }
                count++;

                if (this.heading == null) {
                    this.setHeading(ln);
                }

                // The formatter applies to target.
                // The full line is pipelined to the sink.
                if ((this.target != null) && this.matches(ln) && this.timeMatches(ln)) {
                    if (this.isFormat) {
                        String[] s = ln.split(separator);
                        for (int i = 0; i < this.colIndexes.length; i++) {
                            this.target.print(s[this.colIndexes[i]]);
                            if (i != (this.colIndexes.length - 1)) {
                                this.target.print(separator);
                            }
                        }
                        this.target.println();
                        this.target.flush();
                    } else {
                        this.target.println(ln);
                        this.target.flush();
                    }
                }
                if (this.wsink != null) {
                    this.wsink.println(ln);
                }
            } catch (IOException e) {
                goOn = false;
            }
        }
        if (this.target != null) {
            this.target.flush();
            this.target.close();
        }
        if (this.wsink != null) {
            try {
                this.wsink.flush();
                this.wsink.close();
                this.sink.close();
            } catch (Exception e) {
                e.printStackTrace(System.err);
            }
        }
    }

    /**
     * @param h Heading to set. Heading defines the output format and is the
     *            first line sent out on any pipe.
     */
    private void setHeading(final String h) {
        this.isFormat = false;
        if (h == null) {
            this.heading = null;
        } else {
            this.heading = h.split(separator);
        }

        if ((this.columns != null) && (this.heading != null)) {
            this.isFormat = this.buildIndex();
        }

        // Compute time filter index.
        if (this.isTimeFilter && (this.heading != null)) {
            int i = 0;
            for (i = 0; i < this.heading.length; i++) {
                if (this.timeField.equals(this.heading[i])) {
                    this.timeIx = i;
                    break;
                }
            }
            if (i == this.heading.length) {
                this.resetTimeFilter();
            }
        }
    }

    /**
     * @param columns Name of the columns to set in the formatter.
     */
    public void setFormatter(final String[] columns) {
        this.isFormat = false;
        if (columns != null) {
            this.columns = columns.clone();
        } else {
            this.columns = null;
        }

        if ((columns != null) && (this.heading != null)) {
            this.isFormat = this.buildIndex();
        }
    }

    /**
     * @param exp Regular expression to use as filter for the target.
     */
    public void setRegexp(final String exp) {
        if (exp == null) {
            this.isFilter = false;
        } else {
            this.isFilter = true;
        }
        this.regexp = exp;
    }

    /**
     * @see Outer#resetTimeFilter()
     *
     * @param field Which field to use for time filtering.
     * @param fieldFormat Format of the time field as described in
     *            {@link SimpleDateFormat}.
     * @param from Time from which to start outputting data.
     * @param to Time at which to stop outputting data.
     */
    public void setTimeFilter(final String field, final String fieldFormat, final String from, final String to) {
        try {
            this.timeField = field;
            if (fieldFormat == null) {
                this.timeFormat = null;
            } else {
                this.timeFormat = new SimpleDateFormat(fieldFormat);
            }

            /*
             * from and to must follow the format : "yyyy/MM/dd HH:mm:ss"
             */
            // this.fromTime = getTime(from);
            // this.toTime = getTime(to);
            this.fromTime = this.simpleDateFormat.parse(from).getTime();
            this.toTime = this.simpleDateFormat.parse(to).getTime();
            this.isTimeFilter = true;
        } catch (ParseException e) {
            this.resetTimeFilter();
            e.printStackTrace(System.err);
        }
    }

    /**
     * Resets the time filter.
     *
     * @see Outer#setTimeFilter(String, String, String, String)
     */
    public void resetTimeFilter() {
        this.isTimeFilter = false;
        this.timeField = "time";
        this.timeFormat = null;
        this.fromTime = 0;
        this.toTime = 0;
        this.timeIx = -1;
    }

    /**
     * @return The current field separator.
     */
    public static String getSeparator() {
        return separator;
    }

    /**
     * @param s New field separator to use.
     */
    public static void setSeparator(final String s) {
        separator = s;
    }

    /**
     * @see Outer#setRegexp(String)
     *
     * @param ln String to try to match.
     *
     * @return true if ln matches the regular expression set using
     *         Outer.setRegexp, false otherwise.
     */
    private boolean matches(final String ln) {
        if (!this.isFilter) {
            return true;
        }
        if (this.rsource.getLineNumber() == 1) {
            return true;
        }
        if (this.heading == null) {
            return true;
        }
        return ln.matches(this.regexp);
    }

    /**
     * @see Outer#setTimeFilter(String, String, String, String)
     *
     * @param ln String to try to match.
     *
     * @return true if ln matches the time interval set using
     *         Outer.setTimeFilter, false otherwise.
     */
    private boolean timeMatches(final String ln) {
        if (!this.isTimeFilter) {
            return true;
        }
        if (this.rsource.getLineNumber() == 1) {
            return true;
        }
        if (this.heading == null) {
            return true;
        }

        // Retrieve sample time, filter
        boolean ret = false;
        String[] s = ln.split(separator);
        try {
            long time = this.getTime(s[this.timeIx]);

            if (this.fromTime <= time && time < this.toTime) {
                ret = true;
            }
        } catch (NumberFormatException e) {
            e.printStackTrace(System.err);
        } catch (ParseException e) {
            e.printStackTrace(System.err);
        }

        return ret;
    }

    /**
     * Gets the long representation of a time based on the time format.
     *
     * @param t String to convert into a time.
     *
     * @return Long representation of t.
     *
     * @throws ParseException If parsing fails.
     */
    private long getTime(final String t) throws ParseException {
        if (this.timeFormat == null) {
            return Long.parseLong(t);
        } else {
            return this.timeFormat.parse(t).getTime();
        }
    }

    /**
     * Builds the index.
     *
     * @return true if successful, false otherwise.
     */
    private boolean buildIndex() {
        boolean format = false;
        List<String> names = new LinkedList<String>();
        List<Integer> indexes = new LinkedList<Integer>();
        for (int i = 0; i < this.columns.length; i++) {
            boolean colFound = false;
            for (int j = 0; j < this.heading.length; j++) {
                if (this.heading[j].equals(this.columns[i])) {
                    names.add(this.heading[j]);
                    indexes.add(new Integer(j));
                    colFound = true;
                    break;
                }
            }
            if (!colFound) {
                throw new ArrayIndexOutOfBoundsException("Column " + this.columns[i] + " not found in data header.");
            }
        }
        if (names.size() > 0) {
            String[] cols = new String[names.size()];
            this.colIndexes = new int[indexes.size()];
            for (int i = 0; i < this.colIndexes.length; i++) {
                cols[i] = names.get(i);
                this.colIndexes[i] = (indexes.get(i)).intValue();
            }
            this.columns = cols;
            format = true;
        } else {
            this.columns = null;
            this.colIndexes = null;
        }
        return format;
    }

    /**
     * Source.
     */
    private PipedInputStream source = null;

    /**
     * Reader for the source.
     */
    private LineNumberReader rsource = null;

    /**
     * Sink.
     */
    private PipedOutputStream sink = null;

    /**
     * Writer on the sink.
     */
    private PrintWriter wsink = null;

    /**
     * Target writer.
     */
    private PrintWriter target = null;

    /**
     * Whether format reading is enabled.
     */
    private boolean isFormat = false;

    /**
     * Message headers.
     */
    private String[] heading = null;

    /**
     * Message columns (data).
     */
    private String[] columns = null;

    /**
     * Column indexes is message.
     */
    private int[] colIndexes = null;

    /**
     * Whether regular expression filtering is enabled.
     */
    private boolean isFilter = false;

    /**
     * Regular expression used as filter.
     */
    private String regexp = null;

    /**
     * Default column separator.
     */
    private static final String COLSEP = ";";

    /**
     * Current column separator.
     */
    private static String separator = COLSEP;

    /**
     * Whether time filtering is enabled.
     */
    private boolean isTimeFilter = false;

    /**
     * Default column name for the time.
     */
    private String timeField = "time";

    /**
     * Time format used to parse the time field.
     *
     * If null, time format is a long (milliseconds from 1/1/1970)
     */
    private SimpleDateFormat timeFormat = null;

    /**
     * Time filter column index.
     */
    private int timeIx = -1;

    /**
     * Time format used to parse the boundaries of the time interval: - fromTime -
     * toTime It is fixed to the following human readable format: "yyyy/MM/dd
     * HH:mm:ss".
     */
    private SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

    /**
     * Time from which to start outputting data.
     */
    private long fromTime = 0;

    /**
     * Time at which to stop outputting data.
     */
    private long toTime = 0;

    /**
     * Outer id.
     */
    private String id = "";

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(Outer.class);

}
