/**
 * JASMINe
 * Copyright (C) 2005-2008 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MonologJMX.java 38 2007-12-12 13:03:04Z tokmensa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.IOException;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.logging.LogRecord;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectName;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAP;
import org.ow2.jasmine.monitoring.mbeancmd.Outer;
import org.ow2.jasmine.monitoring.mbeancmd.context.MonologContext;
import org.ow2.jasmine.monitoring.mbeancmd.jasmine.JasmineConnector;

/**
 * Command that displays all Monolog logs from a given J2EE server.
 */
public class MonologJMX extends AbstractCommand {

    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * JMX contexts
     */
    private MonologContext[] context = null;

    /**
     * Constructor: saves the JMX URL.
     * 
     * @param jmxUrl
     *            Target JMX URL.
     */
    public MonologJMX() {
        setOptions();
    }

    public int exec(CommandDispatcher cmdDispatcher) {

        try {
            parseCommandLine(arguments);
        } catch (Exception e) {
            e.printStackTrace();
            return 1;
        }

        /*
         * out (or pout) is : - the output of the sampler - the input for the
         * Outer to print into System.out or into a file sink is: - the sink of
         * the Outer - the input for the graph, if any
         */
        PipedOutputStream out = new PipedOutputStream();
        PrintStream pout = new PrintStream(out);

        Outer outer = null;
        try {

            if (commandLine.hasOption("jasmine")) {
                String jasmineURI = commandLine.getOptionValue("jasmine");

                if (jasmineURI != null) {
                    if (outer == null) {
                        outer = JasmineConnector.connect(out, jasmineURI);
                    } else {
                        Outer oldOuter = outer;
                        outer = JasmineConnector.connect(outer, jasmineURI);
                        new Thread(oldOuter).start();
                    }
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        new Thread(outer).start();

        /*
         * 
         */
        // retrieve the method
        String[] targets = JmxAP.getJmxTargets(commandLine
                .getOptionValues("target"));

        if ((targets != null) && (targets.length > 0)) {
            context = new MonologContext[targets.length];
            for (int i = 0; i < context.length; i++) {
                context[i] = new MonologContext();
                context[i].setName(targets[i]);
                context[i].setJmxUrl(JmxAP.getJmxUrl(targets[i]));
                context[i].setJmxap(new JmxAP(context[i].getJmxUrl(), null));
                try {
                    ObjectName j2eeinstance = ObjectName
                            .getInstance("*:j2eeType=J2EEServer,*");
                    MBeanServerConnection mbscnx = context[i].getJmxap()
                            .getMBeanServerConnection();
                    Iterator onames = mbscnx.queryNames(j2eeinstance, null)
                            .iterator();
                    context[i].setServer(null);
                    while (context[i].getServer() == null) {
                        ObjectName server = (ObjectName) onames.next();
                        context[i].setServer((String) mbscnx.getAttribute(
                                server, "serverName"));
                        context[i].setDomain(server.getDomain());
                    }
                } catch (Exception e) {
                    context[i].setServer("unknown_server_name");
                    context[i].setDomain("unknown_domain");
                } finally {
                    context[i].getJmxap().releaseMBeanServerConnection();
                }
                System.err.println("Target " + context[i].getName() + " - "
                        + context[i].getJmxUrl());
            }
        }

        /*
         * 
         */
        String[] managee = JmxAP.getJmxTargets(commandLine
                .getOptionValues("target"));
        if ((managee != null) && (managee.length > 0)) {
            for (int i = 0; i < managee.length; i++) {

                try {

                    JmxAP jmxAp = new JmxAP(JmxAP.getJmxUrl(managee[i]),
                            cmdDispatcher);

                    ObjectName on = ObjectName
                            .getInstance("jonas:type=service,name=log,fname=trace");

                    MonologContext context = new MonologContext();
                    context.setJmxap(jmxAp);
                    try {
                        ObjectName j2eeinstance = ObjectName
                                .getInstance("*:j2eeType=J2EEServer,*");
                        MBeanServerConnection mbscnx = jmxAp
                                .getMBeanServerConnection();
                        Iterator onames = mbscnx.queryNames(j2eeinstance, null)
                                .iterator();
                        context.setServer(null);
                        while (context.getServer() == null) {
                            ObjectName server = (ObjectName) onames.next();
                            context.setServer((String) mbscnx.getAttribute(
                                    server, "serverName"));
                            context.setDomain(server.getDomain());
                        }
                    } catch (Exception e) {
                        context.setServer("unknown_server_name");
                        context.setDomain("unknown_domain");
                    } finally {
                        jmxAp.releaseMBeanServerConnection();
                    }
                    context.setJmxUrl(JmxAP.getJmxUrl(managee[i]));
                    context.setName(managee[i]);

                    NotificationListener listener = new MonologNotificationListener(
                            pout, context);

                    jmxAp.getMBeanServerConnection().addNotificationListener(
                            on, listener, null, null);

                } catch (MalformedObjectNameException e) {
                    e.printStackTrace();
                } catch (NullPointerException e) {
                    e.printStackTrace();
                } catch (InstanceNotFoundException e) {
                    e.printStackTrace();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        } else {
            System.err.println("No target specified!");
            return 2;
        }

        while (true) {
            try {
                Thread.sleep(Long.MAX_VALUE);
            } catch (InterruptedException e) {
            }
        }
    }

    public String summary() {
        return "Returns Monolog logs from a given JavaEE server";
    }

    /**
     * Parses the command line arguments into {@link Poll#commandLine}.
     * 
     * @param args
     *            Arguments to parse.
     * @throws ParseException
     *             If parsing fails.
     */
    public void parseCommandLine(final String[] args) throws ParseException {
        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);
    }

    /**
     * 
     */
    public void setOptions() {

        options = new Options();

        // jasmine connector output
        Option jasmine = new Option("jasmine", "jasmine", true,
                "Output logged data to jasmine (will disable stdout)");
        jasmine.setRequired(false);
        jasmine.setArgName("jasmineURI");
        jasmine.setArgs(1);
        options.addOption(jasmine);

        // Optional attribute target
        Option target = new Option("target", "target", true,
                "instances to probe");
        target.setRequired(false);
        target.setOptionalArg(true);
        target.setArgs(Option.UNLIMITED_VALUES);
        target.setArgName("instances");
        options.addOption(target);

    }
}

class MonologNotificationListener implements NotificationListener {

    /**
     * 
     */
    String header = "date;time;sname;server;domain;mbean;timestamp;stamp;sequence;loggername;level;message";

    /**
     * 
     */
    MonologContext context = null;

    /**
     * 
     */
    public static final SimpleDateFormat SIMPLEDATEFORMAT = new SimpleDateFormat(
            "yyyy/MM/dd HH:mm:ss");

    /**
     * @param out
     */
    PrintStream out = null;

    /**
     * @param out
     * @param context
     */
    public MonologNotificationListener(PrintStream out, MonologContext context) {
        this.out = out;
        this.context = context;
    }

    /* (non-Javadoc)
     * @see javax.management.NotificationListener#handleNotification(javax.management.Notification, java.lang.Object)
     */
    public void handleNotification(final Notification nf, final Object arg1) {
        LogRecord record = (LogRecord) nf.getUserData();

        StringBuffer bf = new StringBuffer();

        bf.append(header);
        bf.append("\n");
        bf.append(getHeaderBody());
        bf.append(";");
        bf.append(new Date(record.getMillis()));
        bf.append(";");
        bf.append(record.getMillis());
        bf.append(";");
        bf.append(record.getSequenceNumber());
        bf.append(";");
        bf.append(record.getLoggerName());
        bf.append(";");
        bf.append(record.getLevel().toString());
        bf.append(";");
        bf.append(record.getMessage());
        bf.append("\n");

        // bf.append("source class: " + record.getSourceClassName());
        // bf.append("\n");
        // bf.append("source method: " + record.getSourceMethodName());
        // bf.append("\n");
        // bf.append("resource bundle: " + record.getResourceBundleName());
        // bf.append("\n");
        //
        // bf.append("nf.type: " + nf.getType());
        // bf.append("\n");
        // bf.append("nf.stamp: " + nf.getTimeStamp());
        // bf.append("\n");
        // bf.append("nf.source: " + nf.getSource());
        // bf.append("\n");
        // bf.append("nf.seq: " + nf.getSequenceNumber());
        // bf.append("\n");
        // bf.append("nf.msq: " + nf.getMessage());
        // bf.append("\n");
        // bf.append("End of record");

        // System.out.println(bf.toString());

        out.print(bf.toString());
    }

    /**
     * @return Values of header.
     */
    private String getHeaderBody() {

        StringBuffer buff = new StringBuffer();

        // date;time;sname;server;domain;mbean;

        long time = System.currentTimeMillis();
        buff.append(SIMPLEDATEFORMAT.format(new Date(time)));
        buff.append(";");
        buff.append(time);
        buff.append(";");
        buff.append(context.getName());
        buff.append(";");
        buff.append(context.getServer());
        buff.append(";");
        buff.append(context.getDomain());
        buff.append(";");
        buff.append("jonas:type=service,name=log,fname=trace");

        return buff.toString();
    }
}