/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServerSampler.java 55 2007-12-17 16:03:02Z tokmensa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.util.Iterator;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.context.JmxContext;
import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;

/**
 * Sampler for statistics on various parts of the J2EE server. Includes :
 *
 *      - machine cpu, if available
 *      - JVM cpu, if available
 *      - heap committed
 *      - heap used
 *      - number of threads
 *      - pending requests
 *      - current sessions
 *      - database connections
 *      - database busy cnx
 *      - waiters
 *      - dsleaks
 *      - dsfail
 *      - sfb instances
 *      - ssb instances
 *      - ent instances
 *      - committed tx
 *      - served cnx
 */
public class ServerSampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public ServerSampler(final JmxContext context) {
        super(context);
    }

    /**
     * Implementation of inherited abstract method. Returns null since
     * ServerSampler doesn't have one default object name.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOn() {
        return null;
    }

    /**
     * Override of inherited method: will initialize all object name arrays.
     */
    public void init() {
        // Retrieve J2EE Server
        onServer = queryON("*:j2eeType=J2EEServer,*");

        // Retrieve JVM
        onJvm = queryON("*:j2eeType=JVM,*");

        // Retrieve global request processors
        onGRP = queryON("*:type=GlobalRequestProcessor,*");

        // Retrieve ThreadPool Object Name (jk)
        onThreadPool = queryON("*:type=ThreadPool,*");

        // Retrieve Session managers
        onSessions = queryON("*:type=Manager,*");

        // Retrieve JTM Object Name
        onJtm = queryON("*:j2eeType=JTAResource,*");

        // Retrieve DataSource Object Names
        onDS = queryON("*:j2eeType=JDBCDataSource,*");

        // Retrieve Entity bean Object Names
        onEntity = queryON("*:j2eeType=EntityBean,*");

        // Retrieve StatelessSessionBean Object Names
        onSSB = queryON("*:j2eeType=StatelessSessionBean,*");

        // Retrieve StatefulSessionBean Object Names
        onSFB = queryON("*:j2eeType=StatefulSessionBean,*");
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new ServerData();
    }

    /**
     * Implementation of inherited abstract method, will call all pollers of
     * this class.
     *
     * @see Sampler#poll(SampleData)
     *
     * @see ServerSampler#pollServer(ServerData)
     * @see ServerSampler#pollJVM(ServerData)
     * @see ServerSampler#pollSessions(ServerData)
     * @see ServerSampler#pollThreadPool(ServerData)
     * @see ServerSampler#pollJTM(ServerData)
     * @see ServerSampler#pollDataSources(ServerData)
     * @see ServerSampler#pollGlobalRequestProcessor(ServerData)
     * @see ServerSampler#pollEntity(ServerData)
     * @see ServerSampler#pollSSB(ServerData)
     * @see ServerSampler#pollSFB(ServerData)
     */
    protected SampleData poll(final SampleData data) {
        // Status is ok if all polls are OK
        boolean status = true;

        pollServer((ServerData) data);
        status = status && data.isValid();

        pollJVM((ServerData) data);
        status = status && data.isValid();

        pollSessions((ServerData) data);
        status = status && data.isValid();

        pollThreadPool((ServerData) data);
        status = status && data.isValid();

        pollJTM((ServerData) data);
        status = status && data.isValid();

        pollDataSources((ServerData) data);
        status = status && data.isValid();

        pollGlobalRequestProcessor((ServerData) data);
        status = status && data.isValid();

        pollEntity((ServerData) data);
        status = status && data.isValid();

        pollSSB((ServerData) data);
        status = status && data.isValid();

        pollSFB((ServerData) data);
        status = status && data.isValid();

        data.setServerInfo(getName(), getServer(), getDomain());
        data.setValid(status);

        return data;
    }

    /**
     * Polls memory information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollServer(final ServerData data) {
        long totalMem = 0;
        long usedMem = 0;
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "currentTotalMemory",
                "currentUsedMemory"
        };

        for (int i = 0; i < onServer.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onServer[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();                  // currentTotalMemory
                totalMem += Long.parseLong(att.getValue().toString());
                att = (Attribute) it.next();                            // currentUsedMemory
                usedMem += Long.parseLong(att.getValue().toString());
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                e.getCause().printStackTrace(System.err);
            }
        }
        data.setUsedMemory(usedMem);
        data.setMemory(totalMem);
        data.setServerInfo(getName(), getServer(), getDomain());
        if( onServer.length > 0) {
            data.setObjectName(onServer[0]);
        }

    }

    /**
     * Polls thread count.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollJVM(final ServerData data) {
        int nbThreads = 0;
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "allThreadsCount"
        };

        for (int i = 0; i < onJvm.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onJvm[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();                  // allThreadsCount
                nbThreads += Integer.parseInt(att.getValue().toString());
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                e.getCause().printStackTrace(System.err);
            }
        }
        data.setThreads(nbThreads);
    }

    /**
     * Polls session info.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollSessions(final ServerData data ) {
        long nbSess = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "activeSessions"
        };
        for (int i = 0; i < onSessions.length; i++) {
            if (onSessions[i] != null) {
                try {
                    cnx = getMBeanServerConnection();
                    al = cnx.getAttributes(onSessions[i], attIds);
                    releaseMbeanServerConnection();
                    cnx = null;
                    Iterator it = al.iterator();
                    Attribute att = (Attribute) it.next();
                    nbSess += ((Integer) att.getValue()).intValue();
                    data.setValid(true);
                } catch (Exception e) {
                    e.printStackTrace(System.err);
                    e.getCause().printStackTrace(System.err);
                }
            }
        }
        data.setSessions(nbSess);
    }

    /**
     * Polls global request information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollGlobalRequestProcessor(final ServerData data ) {
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "requestCount",
                "errorCount"
        };

        long reqCount = 0;
        long errCount = 0;

        for (int i = 0; i < onGRP.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onGRP[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();              // requestCount
                reqCount += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // errorCount
                errCount += ((Integer) att.getValue()).intValue();
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                e.getCause().printStackTrace(System.err);
            }
        }

        data.setHttpRequests(reqCount);
        data.setHttpErrors(errCount);
    }

    /**
     * Polls thread information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollThreadPool(final ServerData data ) {
        long nbBusy = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "currentThreadsBusy"
        };
        for (int i = 0; i < onThreadPool.length ;i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onThreadPool[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();
                // Even if there is no pending request
                // there is always a busy thread =>
                // if busy = 1 don't count it
                long busy = ((Integer) att.getValue()).intValue();
                nbBusy += (busy == 1 ? 0 : busy);
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                e.getCause().printStackTrace(System.err);
            }
        }
        data.setPendingHttp(nbBusy);
    }

    /**
     * Polls the transaction manager.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollJTM(final ServerData data) {
        long txCount = 0;
        long currentTxGauge = 0;
        long rolledbackTx = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "totalCurrentTransactions",
                "totalCommittedTransactions",
                "totalRolledbackTransactions"
        };

        for (int i = 0; i < onJtm.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onJtm[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();                          // totalCurrentTransactions
                currentTxGauge += Long.parseLong(att.getValue().toString());
                att = (Attribute) it.next();                                    // totalCommittedTransactions
                txCount += Long.parseLong(att.getValue().toString());
                att = (Attribute) it.next();                                    // totalRolledbackTransactions
                rolledbackTx +=  Long.parseLong(att.getValue().toString());
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                e.getCause().printStackTrace(System.err);
            }
        }
        data.setCommitedTx(txCount);
        data.setRollbackTx(rolledbackTx);
        data.setPendingTx(currentTxGauge);
    }

    /**
     * Polls datasources.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollDataSources(final ServerData data) {
        int cnxGauge = 0;
        int cnxFails = 0;
        int cnxLeaks = 0;
        int cnxWaits = 0;
        int cnxRejects = 0;
        int cnxBusy = 0;
        int cnxServed = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "currentOpened",
                "connectionFailures",
                "connectionLeaks",
                "waiterCount",
                "rejectedOpen",
                "currentBusy",
                "servedOpen"
        };
        for (int i = 0; i < onDS.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onDS[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();              // currentOpened
                cnxGauge += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // connectionFailures
                cnxFails += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // connectionLeaks
                cnxLeaks += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // waiterCount
                cnxWaits += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // rejectedOpen
                cnxRejects += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // currentBusy
                cnxBusy += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // servedopen
                cnxServed += ((Integer) att.getValue()).intValue();
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                e.getCause().printStackTrace(System.err);
            }
        }

        data.setDSCnx(cnxGauge);
        data.setDSFails(cnxFails);
        data.setDSLeaks(cnxLeaks);
        data.setDSWaits(cnxWaits);
        data.setDSRejects(cnxRejects);
        data.setDSCnxBusy(cnxBusy);
        data.setDSServed(cnxServed);
    }

    /**
     * Polls entity management information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollEntity(final ServerData data) {
        int entCacheNb = 0;
        int entPoolNb = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "cacheSize",
                "currentInstancePoolSize"
        };
        if (onEntity != null) {
            for (int i = 0; i < onEntity.length ;i++) {
                try {
                    cnx = getMBeanServerConnection();
                    al = cnx.getAttributes(onEntity[i], attIds);
                    releaseMbeanServerConnection();
                    cnx = null;
                    Iterator it = al.iterator();
                    Attribute att = (Attribute) it.next();                  // cacheSize
                    entCacheNb += ((Integer) att.getValue()).intValue();
                    att = (Attribute) it.next();                            // currentInstancePoolSize
                    entPoolNb += ((Integer) att.getValue()).intValue();
                    data.setValid(true);
                } catch (Exception e) {
                    e.printStackTrace(System.err);
                    e.getCause().printStackTrace(System.err);
                }
            }
        }

        data.setEntityCacheNb(entCacheNb);
        data.setEntityPoolNb(entPoolNb);
    }

    /**
     * Poll stateless session bean information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollSSB(final ServerData data) {
        int ssbCacheNb = 0;
        int ssbPoolNb = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "cacheSize",
                "poolSize"
        };
        for (int i = 0; i < onSSB.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onSSB[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();                  // cacheSize
                ssbCacheNb += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                            // currentInstancePoolSize
                ssbPoolNb += ((Integer) att.getValue()).intValue();
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                e.getCause().printStackTrace(System.err);
            }
        }

        data.setSsbCacheNb(ssbCacheNb);
        data.setSsbPoolNb(ssbPoolNb);
    }

    /**
     * Pools stateful session bean information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollSFB(final ServerData data) {
        int sfbCacheNb = 0;
        int sfbPoolNb = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "cacheSize",
                "poolSize"
        };
        if (onSFB != null) {
            for (int i = 0; i < onSFB.length; i++) {
                try {
                    cnx = getMBeanServerConnection();
                    al = cnx.getAttributes(onSFB[i], attIds);
                    releaseMbeanServerConnection();
                    cnx = null;
                    Iterator it = al.iterator();
                    Attribute att = (Attribute) it.next();                  // currentInstancePoolSize
                    sfbCacheNb += ((Integer) att.getValue()).intValue();
                    att = (Attribute) it.next();                            // currentInstancePoolSize
                    sfbPoolNb += ((Integer) att.getValue()).intValue();
                    data.setValid(true);
                } catch (Exception e) {
                    e.printStackTrace(System.err);
                    e.getCause().printStackTrace(System.err);
                }
            }
        }

        data.setSfbCacheNb(sfbCacheNb);
        data.setSfbPoolNb(sfbPoolNb);
    }

    /**
     * J2EE server.
     */
    private ObjectName[] onServer = null;

    /**
     * JVM.
     */
    private ObjectName[] onJvm = null;

    /**
     * JTM.
     */
    private ObjectName[] onJtm = null;

    /**
     * Global request processors.
     */
    private ObjectName[] onGRP = null;

    /**
     * Thread pools.
     */
    private ObjectName[] onThreadPool = null;

    /**
     * Session managers.
     */
    private ObjectName[] onSessions = null;

    /**
     * DataSource Object Names.
     */
    private ObjectName[] onDS = null;

    /**
     * Entity bean Object Names.
     */
    private ObjectName[] onEntity = null;

    /**
     * StatelessSessionBean Object Names.
     */
    private ObjectName[] onSSB = null;

    /**
     * StatefulSessionBean Object Names.
     */
    private ObjectName[] onSFB = null;
}
