/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MLoad.java 38 2007-12-12 13:03:04Z tokmensa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Iterator;
import java.util.Set;

import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAP;

/**
 * Mbean loading facility.
 */
public class MLoad extends AbstractCommand {
    /**
     * Constructor: calls {@link MLoad#setOptions()}.
     */
    public MLoad() {
        this.setOptions();
    }

    /**
     * Sets all options that are parseable from the command line.
     *
     * @see MLoad#options
     */
    private void setOptions() {
        this.options = new Options();
        // options.addOption("v", "verbose", false, "Verbose mode");

        // group of exclusive options :
        // - q : query the mlet mbean name
        // - a : load mbean from URL

        OptionGroup group = new OptionGroup();
        group.setRequired(true);

        // URL
        Option url = new Option("url", "url", true, "Load mbeans from the URL");
        url.setRequired(false);
        url.setArgName("URL");
        url.setArgs(1);
        group.addOption(url);

        // Query mlet ON
        Option query = new Option("q", "query", false, "Query the MLet");
        query.setRequired(false);
        group.addOption(query);

        // delete option
        Option delete = new Option("d", "delete", false, "Delete the MLet");
        delete.setRequired(false);
        group.addOption(delete);

        this.options.addOptionGroup(group);

        // MLet ObjectName
        Option mlet = new Option("m", "mlet", true, "MLet name");
        mlet.setRequired(false);
        mlet.setArgName("name");
        mlet.setArgs(1);
        this.options.addOption(mlet);
    }

    /**
     * Implementation of inherited abstract method.
     *
     * Executes the command.
     *
     * @see AbstractCommand#exec()
     *
     * @return 0 if succeeded, an error code otherwise.
     */
    @Override
    public int exec(final CommandDispatcher cmdDispatcher) {
        try {
            this.parseCommandLine(this.arguments);
        } catch (Exception e) {
            e.printStackTrace();
            return 1;
        }

        this.jmxap = new JmxAP(cmdDispatcher);

        // Query MLet
        if (this.commandLine.hasOption("q")) {
            try {
                Set son = this.findMletON();
                Iterator it = son.iterator();
                while (it.hasNext()) {
                    System.out.println(it.next());
                }
            } catch (Exception e) {
                e.printStackTrace();
                return 2;
            }
        }

        // Load MBean from URL
        if (this.commandLine.hasOption("url")) {
            try {
                this.getMletON();
                Set ois = this.LoadMBeans();
                Iterator it = ois.iterator();
                while (it.hasNext()) {
                    Object o = it.next();
                    if (o instanceof ObjectInstance) {
                        ObjectInstance oi = (ObjectInstance) o;
                        System.out.println(oi);
                    }
                    if (o instanceof Throwable) {
                        Throwable t = (Throwable) o;
                        System.err.println(t.getMessage());
                        t.printStackTrace(System.err);
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
                return 3;
            }
        }

        // Delete MLet
        if (this.commandLine.hasOption("d")) {
            try {
                this.deleteMLet();
            } catch (Exception e) {
                e.printStackTrace();
                return 4;
            }
        }

        return 0;
    }

    /**
     * @return List of MBeans for the URL given as command line argument.
     *
     * @throws InstanceNotFoundException MBean instance not found.
     * @throws MBeanException MBean invocation exception.
     * @throws ReflectionException MBean reflection failed.
     * @throws IOException Connection problem.
     */
    private Set LoadMBeans() throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        Set ois = null;

        String operation = "getMBeansFromURL";
        Object[] params = new Object[] {this.mbeanUrl};
        String[] sig = new String[] {"java.net.URL"};
        try {
            this.mbscnx = this.jmxap.getMBeanServerConnection();
            ois = (Set) this.mbscnx.invoke(this.mleton, operation, params, sig);
        } finally {
            this.jmxap.releaseMBeanServerConnection();
        }

        return ois;
    }

    /**
     * Deletes the MLet given as command argument.
     *
     * @throws InstanceNotFoundException MBean instance not found.
     * @throws MBeanRegistrationException MLet registration failed.
     * @throws MalformedObjectNameException Object name invalid.
     * @throws IOException Connection problem.
     */
    private void deleteMLet() throws InstanceNotFoundException, MBeanRegistrationException, MalformedObjectNameException,
            IOException {
        try {
            this.mbscnx = this.jmxap.getMBeanServerConnection();
            this.mbscnx.unregisterMBean(ObjectName.getInstance(this.getDomain() + ":" + MLET_OBJECTNAME_RADIX + ",name="
                    + (this.mletName != null ? this.mletName : this.server)));
        } finally {
            this.jmxap.releaseMBeanServerConnection();
        }
    }

    /**
     * @return Newly created MLet instance for the URL given as command line
     *         argument.
     *
     * @throws MalformedObjectNameException Object name invalid.
     * @throws InstanceAlreadyExistsException MLet instance already exists.
     * @throws MBeanRegistrationException MLet registration failed.
     * @throws NotCompliantMBeanException MBean not compliant.
     * @throws ReflectionException MBean reflection failed.
     * @throws MBeanException MBean invocation exception.
     * @throws IOException Connection problem.
     */
    private ObjectName getMletON() throws MalformedObjectNameException, InstanceAlreadyExistsException,
            MBeanRegistrationException, NotCompliantMBeanException, ReflectionException, MBeanException, IOException {

        if (!this.isMletON()) {
            try {
                // create MLet mbean
                this.mbscnx = this.jmxap.getMBeanServerConnection();
                ObjectInstance oi = this.mbscnx.createMBean("javax.management.loading.MLet", ObjectName.getInstance(this.getDomain() + ":"
                        + MLET_OBJECTNAME_RADIX + ",name=" + (this.mletName != null ? this.mletName : this.server)));
                this.mleton = oi.getObjectName();
            } finally {
                this.jmxap.releaseMBeanServerConnection();
            }
        }
        return this.mleton;
    }

    /**
     * @return true if the MLet given as command line argument exists.
     *
     * @throws IOException Connection problem.
     * @throws MalformedObjectNameException Object name invalid.
     */
    private boolean isMletON() throws IOException, MalformedObjectNameException {
        if (this.mleton == null) {
            ObjectName on = ObjectName.getInstance("*:" + MLET_OBJECTNAME_RADIX + ",*");
            Set son;
            try {
                this.mbscnx = this.jmxap.getMBeanServerConnection();
                son = this.mbscnx.queryNames(on, null);
            } finally {
                this.jmxap.releaseMBeanServerConnection();
            }

            if (!son.isEmpty()) {
                // found one
                this.mleton = (ObjectName) son.iterator().next();
            }
        }
        return (this.mleton != null);
    }

    /**
     * @return Existing MLet instance for the URL given as command line
     *         argument.
     *
     * @throws IOException Connection problem.
     * @throws MalformedObjectNameException Object name invalid.
     */
    private Set findMletON() throws IOException, MalformedObjectNameException {
        ObjectName on = ObjectName.getInstance("*:" + MLET_OBJECTNAME_RADIX + ",*");
        try {
            this.mbscnx = this.jmxap.getMBeanServerConnection();
            return this.mbscnx.queryNames(on, null);
        } finally {
            this.jmxap.releaseMBeanServerConnection();
        }
    }

    /**
     * Sets the J2EE server's domain and server names into {@link MLoad#domain}
     * and {@link MLoad#server}.
     *
     * @return The J2EE server's domain name, "unknown_domain" on failure.
     *
     * @throws IOException Connection problem.
     * @throws MalformedObjectNameException Object name invalid.
     */
    private String getDomain() throws IOException, MalformedObjectNameException {
        if (this.domain == null) {
            this.server = null;
            ObjectName j2eeinstance = ObjectName.getInstance("*:j2eeType=J2EEServer,*");
            try {
                this.mbscnx = this.jmxap.getMBeanServerConnection();
                Iterator onames = this.mbscnx.queryNames(j2eeinstance, null).iterator();

                while (this.server == null) {
                    ObjectName j2eeserver = (ObjectName) onames.next();
                    this.server = (String) this.mbscnx.getAttribute(j2eeserver, "serverName");
                    this.domain = j2eeserver.getDomain();
                }
            } catch (Exception e) {
                this.server = "unknown_server_name";
                this.domain = "unknown_domain";
            } finally {
                this.jmxap.releaseMBeanServerConnection();
            }
        }
        return this.domain;
    }

    /**
     * Parses the command line arguments into {@link MLoad#commandLine}.
     *
     * @param args Arguments to parse.
     *
     * @throws ParseException If parsing fails.
     * @throws MalformedObjectNameException Object name given in the command
     *             line is invalid.
     * @throws MalformedURLException URL given in the command line is invalid.
     */
    private void parseCommandLine(final String[] args) throws ParseException, MalformedObjectNameException, MalformedURLException {
        BasicParser bp = new BasicParser();
        this.commandLine = bp.parse(this.options, args);

        if (this.commandLine.hasOption("m")) {
            this.mletName = this.commandLine.getOptionValue("m");
        }

        if (this.commandLine.hasOption("url")) {
            String surl = this.commandLine.getOptionValue("url");
            this.mbeanUrl = new URL(surl);
        }
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see AbstractCommand#summary()
     */
    @Override
    public String summary() {
        return "MBean loading utility";
    }

    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * The object name based on the command line arguments.
     */
    private ObjectName mleton = null;

    /**
     * Default ObjectName MLet radix.
     */
    private static final String MLET_OBJECTNAME_RADIX = "type=jonasmlet";

    /**
     * Current MLet name.
     */
    private String mletName = null;

    /**
     * MBean's URL.
     */
    private URL mbeanUrl = null;

    /**
     * JMX connection to server.
     */
    private MBeanServerConnection mbscnx = null;

    /**
     * Domain name.
     */
    private String domain = null;

    /**
     * Server name.
     */
    private String server = null;

    private JmxAP jmxap = null;
}
