/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JoramQSampler.java 1572 2008-07-01 11:54:58Z renaultgu $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.IOException;
import java.util.Hashtable;
import java.util.Iterator;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import org.ow2.jasmine.monitoring.mbeancmd.context.JmxContext;

/**
 * Sampler for JORAM Queues.
 */
public class JoramQSampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public JoramQSampler(final JmxContext context) {
        super(context);
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOn() {
        return "joramClient:type=queue,*";
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new XSampleData();
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#poll(SampleData)
     * @see EntitySampler#pollEntities(XSampleData)
     */
    protected SampleData poll(final SampleData data) {
        pollJoramQueues((XSampleData) data);
        data.setValid(true);
        return data;
    }

    /**
     * Polls JCA Connection Factory providers for data, will call
     * {@link JoramQSampler#pollJoramQueue(ObjectName, JoramQData)}
     * for each object name.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollJoramQueues(final XSampleData data) {

        ObjectName[] oNames = (ObjectName[]) context.getOnames().toArray(new ObjectName[context.getOnames().size()]);

        for (int i = 0; i < oNames.length; i++) {
            JoramQData qdata = new JoramQData();
            qdata.setObjectName(oNames[i]);
            qdata.setSampleTime(System.currentTimeMillis());
            pollJoramQueue(oNames[i], qdata);

            if (qdata.isValid()) {
                data.put(oNames[i], qdata);
            }
        }
    }

    /**
     * Polls one JCA Connection Factory provider for data.
     *
     * @param on    Name of the provider.
     * @param data  ServerData to add polled information to.
     */
    private void pollJoramQueue(final ObjectName on, final JoramQData data) {
        AttributeList al = null;
        MBeanServerConnection cnx = null;

            Attribute att;
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(on, attIds);
                releaseMbeanServerConnection();
                cnx = null;
                Iterator it = al.iterator();
                while (it.hasNext()) {
                    att = (Attribute) it.next();
                    if ("Statistic".equals(att.getName())) {
                        /*
                         * this attribute is a HashTable with the following entries:
                         * - creationDate,
                         * - nbMsgsReceiveSinceCreation,
                         * - nbMsgsSendToDMQSinceCreation,
                         * - nbMsgsDeliverSinceCreation
                         */
                        Hashtable ht = (Hashtable) att.getValue();

                        // retrieve creationDate
                        Attribute a1 = new Attribute("creationDate", ht.get("creationDate"));
                        data.setAttribute(a1);

                        // retrieve nbMsgsReceiveSinceCreation (Long)
                        a1 = new Attribute("nbMsgsReceiveSinceCreation", ht.get("nbMsgsReceiveSinceCreation"));
                        data.setAttribute(a1);

                        // retrieve nbMsgsSendToDMQSinceCreation
                        a1 = new Attribute("nbMsgsSendToDMQSinceCreation", ht.get("nbMsgsSendToDMQSinceCreation"));
                        data.setAttribute(a1);

                        // retrieve nbMsgsDeliverSinceCreation
                        a1 = new Attribute("nbMsgsDeliverSinceCreation", ht.get("nbMsgsDeliverSinceCreation"));
                        data.setAttribute(a1);

                    } else {
                        data.setAttribute(att);
                    }
                }
                data.setServerInfo(getName(), getServer(), getDomain());
                data.setValid(true);
            } catch (InstanceNotFoundException e) {
                e.printStackTrace(System.err);
            } catch (ReflectionException e) {
                e.printStackTrace(System.err);
            } catch (IOException e) {
                e.printStackTrace(System.err);
            }
    }

    /**
     * List of polled attributes.
     * Note that 'Statistic' is a Hastable which is expected to hold the following entries:
     * - creationDate
     * - nbMsgsReceiveSinceCreation
     * - nbMsgsSendToDMQSinceCreation
     * - nbMsgsDeliverSinceCreation
     */
    private static final String[] attIds = {
        "AdminName", "Type", "PendingMessages",
        "NbMaxMsg", "PendingRequests",
        "Statistic", "Threshold"};
}
