/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: SFBSampler.java 55 2007-12-17 16:03:02Z tokmensa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.IOException;
import java.util.Iterator;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import org.ow2.jasmine.monitoring.mbeancmd.context.JmxContext;

/**
 * Sampler for retrieving data from stateful session beans.
 */
public class SFBSampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public SFBSampler(final JmxContext context) {
        super(context);
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOn() {
        return "*:j2eeType=StatefulSessionBean,*";
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new XSampleData();
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#poll(SampleData)
     * @see SFBSampler#pollSFBs(XSampleData)
     */
    protected SampleData poll(final SampleData data) {
        pollSFBs((XSampleData) data);
        data.setValid(true);
        return data;
    }

    /**
     * Polls stateful bean providers for data, will call
     * {@link SFBSampler#pollSFB(ObjectName, SFBData)}
     * for each object name.
     *
     * @param data  XSampleData to add polled information to.
     */
    private void pollSFBs(final XSampleData data) {

        ObjectName[] oNames = (ObjectName[]) context.getOnames().toArray(new ObjectName[context.getOnames().size()]);

        for (int i = 0; i < oNames.length; i++) {
            SFBData sfbdata = new SFBData();
            sfbdata.setObjectName(oNames[i]);
            sfbdata.setSampleTime(System.currentTimeMillis());
            pollSFB(oNames[i], sfbdata);

            if (sfbdata.isValid()) {
                data.put(oNames[i], sfbdata);
            }
        }
    }

    /**
     * Polls one stateful bean provider for data.
     *
     * @param on    Name of the provider.
     * @param data  SFBData to add polled information to.
     */
    private void pollSFB(final ObjectName on, final SFBData data) {
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        Attribute att;
        try {
            cnx = getMBeanServerConnection();
            al = cnx.getAttributes(on, attIds);
            releaseMbeanServerConnection();
            cnx = null;
            Iterator it = al.iterator();
            while (it.hasNext()) {
                att = (Attribute) it.next();
                data.setAttribute(att);
            }
            data.setServerInfo(getName(), getServer(), getDomain());
            data.setValid(true);
        } catch (InstanceNotFoundException e) {
            e.printStackTrace(System.err);
        } catch (ReflectionException e) {
            e.printStackTrace(System.err);
        } catch (IOException e) {
            e.printStackTrace(System.err);
        }
    }

    /**
     * Polled attribute identifiers.
     */
    private static final String[] attIds = {
             "name",
             "cacheSize",
             "poolSize",
             "minPoolSize",
             "maxCacheSize",
             "sessionTimeOut"
    };
}
