/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: SLBSampler.java 55 2007-12-17 16:03:02Z tokmensa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.IOException;
import java.util.Iterator;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import org.ow2.jasmine.monitoring.mbeancmd.context.JmxContext;

/**
 * Sampler for retrieving data from stateless session beans.
 */
public class SLBSampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public SLBSampler(final JmxContext context) {
        super(context);
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOn() {
        return "*:j2eeType=StatelessSessionBean,*";
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new XSampleData();
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#poll(SampleData)
     * @see SLBSampler#pollSLBs(XSampleData)
     */
    protected SampleData poll(final SampleData data) {
        pollSLBs((XSampleData) data);
        data.setValid(true);
        return data;
    }

    /**
     * Polls stateless bean providers for data, will call
     * {@link SLBSampler#pollSLB(ObjectName, SLBData)}
     * for each object name.
     *
     * @param data  XSampleData to add polled information to.
     */
    private void pollSLBs(final XSampleData data) {

        ObjectName[] oNames = (ObjectName[]) context.getOnames().toArray(new ObjectName[context.getOnames().size()]);

        for (int i = 0; i < oNames.length; i++) {
            SLBData slbdata = new SLBData();
            slbdata.setObjectName(oNames[i]);
            slbdata.setSampleTime(System.currentTimeMillis());
            pollSLB(oNames[i], slbdata);

            if (slbdata.isValid()) {
                data.put(oNames[i], slbdata);
            }
        }
    }

    /**
     * Polls one stateless bean provider for data.
     *
     * @param on    Name of the provider.
     * @param data  SLBData to add polled information to.
     */
    private void pollSLB(final ObjectName on, final SLBData data) {
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        Attribute att;
        try {
            cnx = getMBeanServerConnection();
            al = cnx.getAttributes(on, attIds);
            releaseMbeanServerConnection();
            cnx = null;
            Iterator it = al.iterator();
            while (it.hasNext()) {
                att = (Attribute) it.next();
                data.setAttribute(att);
            }
            data.setServerInfo(getName(), getServer(), getDomain());
            data.setValid(true);
        } catch (InstanceNotFoundException e) {
            e.printStackTrace(System.err);
        } catch (ReflectionException e) {
            e.printStackTrace(System.err);
        } catch (IOException e) {
            e.printStackTrace(System.err);
        }
    }

    /**
     * Polled attribute identifiers.
     */
    private static final String[] attIds = {
             "name",
             "cacheSize",
             "poolSize",
             "minPoolSize",
             "maxCacheSize",
             "sessionTimeOut"
    };
}
