/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: SampleData.java 55 2007-12-17 16:03:02Z tokmensa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.PrintStream;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import java.util.TreeMap;

import javax.management.Attribute;
import javax.management.ObjectName;

/**
 * Base class for all data classes.
 */
public abstract class SampleData {
    /**
     * Compute based on a previous measurement, will fill all fields that are
     * marked as being "computed" and don't have a public setter.
     *
     * @param prev  Previous measurement.
     */
    public abstract void compute(final SampleData prev);

    /**
     * Returns the full printout header for this data. The child class should
     * not implement this method, it should only implement
     * {@link SampleData#getInnerPrintHeader()} instead.
     *
     * @return  The printout header format for this data.
     */
    public final String getPrintHeader() {
        String header = "time" + getSeparator() + "date" + getSeparator() 
                      + "sname" + getSeparator() + "server" + getSeparator()
                      + "domain" + getSeparator() + "mbean" + getSeparator()
                      + getInnerPrintHeader();

        return header;
    }

    /**
     * @return  The inner (specialized) part of the header for this data type.
     */
    protected abstract String getInnerPrintHeader();

    /**
     * Prints out all data in attributes based on the print header.
     *
     * @see SampleData#getPrintHeader()
     * @see SampleData#getInnerPrintData()
     *
     * @param out  Stream to print data values into.
     */
    public void printData(final PrintStream out) {
        Date d = new Date(getSampleTime());
        String result = getSampleTime() + getSeparator() + simpleDateFormat.format(d) + getSeparator()
                      + getName() + getSeparator() + getServer() + getSeparator()
                      + getDomain() + getSeparator() + objectName + getSeparator()
                      + getInnerPrintData();

        out.println(result);
    }

    /**
     * @return  The inner (specialized) part of the data for this data type.
     */
    protected abstract String getInnerPrintData();

    /**
     * @param on  Object name to set.
     */
    protected final void setObjectName(final ObjectName on) {
        //this.objectName = on.getKeyProperty("name");
        this.objectName = on.getCanonicalName();
    }

    /**
     * @return  Time of sampling, origin is 01/01/1970.
     */
    protected final long getSampleTime() {
        return sampleTime;
    }

    /**
     * @param t  Time of sampling, origin is 01/01/1970.
     */
    protected final void setSampleTime(final long t) {
        sampleTime = t;
    }

    /**
     * @param att  Attribute to add in the list of data attributes.
     */
    protected final void setAttribute(final Attribute att) {
        atts.put(att.getName(), att.getValue());
    }

    /**
     * @param name  Name of the attribute to get.
     *
     * @return  Attribute corresponding to name, null if none match.
     */
    protected final Object getAttribute(final String name) {
        return atts.get(name);
    }

    /**
     * @param name  Name of the attribute to get.
     *
     * @return  Attribute corresponding to name, transformed into an integer.
     */
    protected final int getAttributeAsInt(final String name) {
        return ((Integer) atts.get(name)).intValue();
    }

    /**
     * @param name  Name of the attribute to get.
     *
     * @return  Attribute corresponding to name, transformed into a long.
     */
    protected final long getAttributeAsLong(final String name) {
        return ((Long) atts.get(name)).longValue();
    }

    /**
     * @param name  Name of the attribute to get.
     *
     * @return  Attribute corresponding to name, transformed into a double.
     */
    protected final double getAttributeAsDouble(final String name) {
        return ((Double) atts.get(name)).doubleValue();
    }

    /**
     * @return  Current separator.
     */
    protected static final String getSeparator() {
        return separator;
    }

    /**
     * @param s  Separator to set.
     */
    public static final void setSeparator(final String s) {
        separator = s;
    }

    /**
     * @return  Name set for the server in the jmxurls.properties file.
     */
    protected final String getName() {
        return this.name;
    }

    /**
     * @return  Name of the server.
     */
    protected final String getServer() {
        return this.server;
    }

    /**
     * @return  Name of the server's domain.
     */
    protected final String getDomain() {
        return this.domain;
    }

    /**
     * @param name    Name set for the server in the jmxurls.properties file.
     * @param server  Name of the server.
     * @param domain  Domain name of the server.
     */
    protected final void setServerInfo(final String name, final String server, final String domain) {
        this.name = name;
        this.server = server;
        this.domain = domain;
    }

    /**
     * @param status  true to mark the data as valid, false otherwise.
     */
    protected final void setValid(final boolean status) {
        this.status = status;
    }

    /**
     * @return true if data valid, false otherwise.
     */
    protected final boolean isValid() {
        return status;
    }

    /**
     * Time at which data has been sampled.
     */
    private long sampleTime = 0;

    /**
     * Name set for the server in the jmxurls.properties file.
     */
    private String name = null;

    /**
     * Name of the server.
     */
    private String server = null;

    /**
     * Domain name of the server.
     */
    private String domain = null;

    /**
     * Object name.
     */
    private String objectName = null;

    /**
     * Is data valid ?
     */
    private boolean status = false;

    /**
     * Decimal formatter.
     */
    protected static final DecimalFormat decimalFormat = (DecimalFormat) DecimalFormat.getInstance();

    /**
     * Date formatter.
     */
    private static SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

    /**
     * Separator.
     */
    private static String separator = ";";

    /**
     * Attributes.
     */
    private Map atts = new TreeMap();

    /**
     * Format decimal parsing.
     */
    static {
        DecimalFormatSymbols sym = new DecimalFormatSymbols();
        sym.setDecimalSeparator('.');
        decimalFormat.setDecimalFormatSymbols(sym);
        decimalFormat.applyPattern("#.#");
    }
}
