/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: TxData.java 55 2007-12-17 16:03:02Z tokmensa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;


/**
 * Transaction statistics.
 */
public class TxData extends SampleData {
    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#compute(SampleData)
     */
    public void compute(final SampleData needs_cast) {
        TxData prev = (TxData) needs_cast;
        if (prev != null) {
            long deltaCommited = this.commitedCount - prev.commitedCount;
            // Tx rate (tx / second)
            rateCommited = 1000.0 * deltaCommited / (this.getSampleTime() - prev.getSampleTime());
            // processing time per request (millis)
            long deltaRolledback = this.rolledbackCount - prev.rolledbackCount;
            long deltaTx = deltaCommited + deltaRolledback;
            if (deltaTx > 0) {
                commitRatio = 100.0 * deltaCommited / deltaTx;
            } else {
                commitRatio = 100.0;
            }
            long tx = commitedCount + rolledbackCount;
            if (tx > 0) {
                this.meanCommitRatio = 100.0 * commitedCount / tx;
            } else {
                meanCommitRatio = 100.0;
            }
        } else {
            rateCommited = 0.0;
            commitRatio = 100.0;
            meanCommitRatio = 100.0;
        }
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintHeader()
     */
    protected String getInnerPrintHeader() {
        return header.replaceAll(";", getSeparator());
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintData()
     */
    protected String getInnerPrintData() {
        return  decimalFormat.format(getCommitedTxRate()) + getSeparator()
              + getPendingTxGauge() + getSeparator()
              + getCommittedTxCnt() + getSeparator()
              + getRolledbackTxCnt() + getSeparator()
              + this.getTimedOutTxCnt() + getSeparator()
              + decimalFormat.format(getMeanCommitRatio()) + getSeparator()
              + decimalFormat.format(getCommitTxRatio());
    }

    /**
     * @param cnt  Committed transactions.
     */
    public void setCommittedTxCnt(final long cnt) {
        commitedCount = cnt;
    }

    /**
     * @return  Committed transactions.
     */
    public long getCommittedTxCnt() {
        return commitedCount;
    }

    /**
     * @param gauge  Pending transactions.
     */
    public void setPendingTxGauge(final long gauge) {
        pendingTx = gauge;
    }

    /**
     * @return  Pending transactions.
     */
    public long getPendingTxGauge() {
        return pendingTx;
    }

    /**
     * @param cnt  Rollbacked transactions.
     */
    public void setRolledbackTxCnt(final long cnt) {
        rolledbackCount = cnt;
    }

    /**
     * @return  Rollbacked transactions.
     */
    public long getRolledbackTxCnt() {
        return rolledbackCount;
    }

    /**
     * @param cnt  Timed out transactions.
     */
    public void setTimedOutTxCnt(final long cnt) {
        timedOutTx = cnt;
    }

    /**
     * @return  Timed out transactions.
     */
    public long getTimedOutTxCnt() {
        return timedOutTx;
    }

    /**
     * @return  Commited transaction rate (computed).
     */
    public double getCommitedTxRate() {
        return rateCommited;
    }

    /**
     * @return  Commit ratio (computed).
     */
    public double getCommitTxRatio() {
        return commitRatio;
    }

    /**
     * @return  Mean commit ratio (computed).
     */
    public double getMeanCommitRatio() {
        return meanCommitRatio;
    }

    /**
     * Committed transactions.
     */
    private long commitedCount = 0;

    /**
     * Pending transactions.
     */
    private long pendingTx = 0;

    /**
     * Rollbacked transactions.
     */
    private long rolledbackCount = 0;

    /**
     * Timed out transactions.
     */
    private long timedOutTx = 0;

    /**
     * Commited transaction rate (computed).
     */
    private double rateCommited = 0.0;

    /**
     * Commit ratio (computed).
     */
    private double commitRatio = 0.0;

    /**
     * Mean commit ratio (computed).
     */
    private double meanCommitRatio = 100.0;

    /**
     * Default output header.
     */
    private static final String header = "txRate;pending;commit;rollback;timedOut;commitRatio;currentCommitRatio";
}
