/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Iner.java 1928 2008-07-30 08:45:19Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStreamWriter;
import java.io.PipedOutputStream;
import java.io.PrintWriter;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * A pipe consists of a pair of channels: a writable channel (called "sink",
 * where information is written to) and a readable channel (called "source",
 * where information is obtained from).
 *
 * The general idea is that once some bytes are available on the source:
 *
 * 1. The pipe reads those bytes. 2. If required, does some action with some or
 * all of these bytes (for example, write them to a file). 3. If a sink is
 * present, writes the exact same bytes in the exact same order on the sink.
 *
 * An Iner instance can retrieve information from (=use as source) an
 * InputStream or a File.
 */
public class Iner implements Runnable {
    /**
     * Creates an Iner using an InputStream as source. A dummy sink (that can be
     * used an input to any pipe,
     *
     * @see Iner.getSink) will be created by default.
     *
     * @see Iner#setSink(PipedOutputStream)
     *
     * @param in InputStream to use as source.
     */
    public Iner(final InputStream in) {
        this.source = new LineNumberReader(new InputStreamReader(in));
        this.setSink(null);
    }

    /**
     * Creates an Iner using a File as source. A dummy sink (that can be used an
     * input to any pipe,
     *
     * @see Iner.getSink) will be created by default.
     *
     * @see Iner#setSink(PipedOutputStream)
     *
     * @param file File to use as source.
     *
     * @throws FileNotFoundException If file not found.
     */
    public Iner(final File file) throws FileNotFoundException {
        this.source = new LineNumberReader(new FileReader(file));
        this.setSink(null);
    }

    /**
     * @param sink Sink to set. A dummy sink will be created if null.
     */
    public void setSink(final PipedOutputStream sink) {
        if (sink == null) {
            this.setSink(new PipedOutputStream());
        } else {
            if (this.wsink != null) {
                try {
                    this.wsink.close();
                    this.sink.close();
                } catch (Exception e) {
                    this.logger.warn("Error while closing sink : {0}", e);
                }
            }
            this.sink = sink;
            this.wsink = new PrintWriter(new OutputStreamWriter(sink), /* autoflush */true);
        }
    }

    /**
     * @return The sink currently used.
     */
    public PipedOutputStream getSink() {
        return this.sink;
    }

    /**
     * Starts the activity on the Iner (as described in the class header).
     */
    public void run() {
        boolean goOn = true;

        while (goOn) {
            try {
                String ln = this.source.readLine();
                if (ln == null) {
                    goOn = false;
                    continue;
                }
                this.wsink.println(ln);
                this.wsink.flush();
            } catch (IOException e) {
                e.printStackTrace(System.err);
                goOn = false;
            }
        }
        this.wsink.flush();
        this.wsink.close();
    }

    /**
     * Lines input line per line.
     */
    private LineNumberReader source = null;

    /**
     * Sink. A dummy sink will be created if the user doesn't specify any.
     */
    private PipedOutputStream sink = null;

    /**
     * Writer that writes on the sink.
     */
    private PrintWriter wsink = null;

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(Iner.class);
}
