/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxAP.java 5426 2009-09-25 07:10:07Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;
//import java.util.logging.Level;
//import java.util.logging.Logger;

import javax.management.MBeanServerConnection;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.resource.ResourceException;

import org.ow2.jasmine.adapter.jmx.pool.api.IJMXConnectionFactory;
import org.ow2.jasmine.adapter.jmx.pool.api.IJMXConnectionServer;
import org.ow2.jasmine.adapter.jmx.pool.api.JMXConnectionParam;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * A JMX Access Point.
 */
public class JmxAP {

    /**
     * Context that contains this JMX Access Point.
     */
    private static Context ctx = null;

    /**
     * Command dispatcher instance.
     */
    private CommandDispatcher cmdDispatcher = null;

    /**
     * Creates a JmxAP that connects to a URL.
     *
     * @param jmxUrl URL to connect to.
     */
    public JmxAP(final String jmxUrl, final CommandDispatcher cmdDispatcher) {
        this.jmxUrl = jmxUrl;
        if (cmdDispatcher != null) {
            this.cmdDispatcher = cmdDispatcher;
            if (this.cmdDispatcher.isEmbedded()) {
                this.connectJMX();
            }
        }
    }

    /**
     * Creates a JmxAP that connects to the default URL.
     */
    public JmxAP(final CommandDispatcher cmdDispatcher) {
        this(jmxTargets.get(defaultTarget), cmdDispatcher);
    }

    /**
     * @return The String version of the JMX URL connected to.
     */
    public String getJmxUrl() {
        return this.jmxUrl;
    }

    /**
     * Establishes a JMX connection. This method is called only if MBeanCmd is
     * embedded. First, the naming context is get. If an error occurs during
     * this first step, the context will stay null. The second step is a look up
     * on the naming context to get the factory which will provide a JMX
     * connection. The JNDI name looked up is the url of the JMX url to get
     * information from where all ":" and "/" are replaced by "_". This means
     * that the RA has to be registered using this way.
     */
    private void connectJMX() {
        if (logger.isDebugEnabled()) {
            logger.debug("Connecting to the jmxFactory ...");
        }
        try {
            ctx = new InitialContext();
        } catch (NamingException e) {
            logger.error("Error while creating initialContext : {0}. An unmanaged JMX connection will be used instead.", e.getMessage());
        }

        String lookup = this.jmxUrl.replaceAll(":|/", "_");
        try {
            this.jmxFactory = (IJMXConnectionFactory) ctx.lookup(lookup);
        } catch (NamingException e) {
            logger.error("Error while looking up JMX Connection factory {0} : {1}.  An unmanaged JMX connection will be used instead. ",lookup, e.getMessage()); 
        }

    }

    /**
     * Get the JMX connection. If MBeanCmd is embedded, the connection will be
     * provide by a factory previously get. If MBeanCMd runs in standalone mode
     * or if the factory is null, a non managed JMX connection will be set.
     *
     * @return The MBeanServerConnection instance used to connect to the J2EE
     *         server via JMX.
     */
    public MBeanServerConnection getMBeanServerConnection() {
        if(logger.isDebugEnabled()) {
            logger.debug("getting MBeanserverConnection");                    
        }
        if (this.jmxConnection == null) {
            if (this.cmdDispatcher != null && this.cmdDispatcher.isEmbedded() && this.jmxFactory != null) {
                try {
                    if(logger.isDebugEnabled()) {
                        logger.debug("getting MBeanserverConnection using the pool");                    
                    }
                    this.jmxConnectionEmbedded = (IJMXConnectionServer) this.jmxFactory.getConnection(new JMXConnectionParam(
                            this.jmxUrl, jmxUsers.get(this.jmxUrl), jmxPasswords.get(this.jmxUrl), jmxProtocolProviderPackages.get(this.jmxUrl)));
                    this.jmxConnection = this.jmxConnectionEmbedded;
                } catch (ResourceException e) {
                    // TODO this.releaseMBeanServerConnection();
                    logger.error("Error while getting a connection : {0}",e.getMessage());
                }
            } else {
                if(logger.isDebugEnabled()) {
                    logger.debug("Creating a simple JMX Connection without using the pool, URL {0}", this.jmxUrl);                    
                }
                this.jmxConnection = new JmxCnxWrapper(this.jmxUrl, jmxUsers.get(this.jmxUrl), jmxPasswords.get(this.jmxUrl), jmxProtocolProviderPackages.get(this.jmxUrl));
            }
        }
        return this.jmxConnection;
    }
    
    /**
     * Get the JMX connection. If MBeanCmd is embedded, the connection will be
     * provide by a factory previously get. If MBeanCMd runs in standalone mode
     * or if the factory is null, a non managed JMX connection will be set.
     *
     * @return The MBeanServerConnection instance used to connect to the J2EE
     *         server via JMX.
     */
    public MBeanServerConnection getMBeanServerConnection(boolean usePool) {
        if(logger.isDebugEnabled()) {
            logger.debug("getting MBeanserverConnection");                    
        }
        if (this.jmxConnection == null) {
            if (usePool && this.cmdDispatcher != null && this.cmdDispatcher.isEmbedded() && this.jmxFactory != null) {
                try {
                    if(logger.isDebugEnabled()) {
                        logger.debug("getting MBeanserverConnection using the pool");                    
                    }
                    this.jmxConnectionEmbedded = (IJMXConnectionServer) this.jmxFactory.getConnection(new JMXConnectionParam(
                            this.jmxUrl, jmxUsers.get(this.jmxUrl), jmxPasswords.get(this.jmxUrl), jmxProtocolProviderPackages.get(this.jmxUrl)));
                    this.jmxConnection = this.jmxConnectionEmbedded;
                } catch (ResourceException e) {
                    // TODO this.releaseMBeanServerConnection();
                    logger.error("Error while getting a connection for URL {0} : {1}", this.jmxUrl, e.getMessage());
                }
            } else {
                if(logger.isDebugEnabled()) {
                    logger.debug("Opening a simple JMX Connection without using the pool, URL {0}", this.jmxUrl);                    
                }
                this.jmxConnection = new JmxCnxWrapper(this.jmxUrl, jmxUsers.get(this.jmxUrl), jmxPasswords.get(this.jmxUrl), jmxProtocolProviderPackages.get(this.jmxUrl));
            }
        }
        return this.jmxConnection;
    }

    /**
     * In embedded mode only.
     *
     * Release the connection in the pool.
     */
    public void releaseMBeanServerConnection() {
        if (this.cmdDispatcher != null && this.cmdDispatcher.isEmbedded() && this.jmxFactory != null) {
            this.jmxConnectionEmbedded.release();
            this.jmxConnection = null;
        }
    }

    /**
     * In embedded mode only.
     *
     * Explicitly close the JMX connection in the pool, physically.
     */
    public void closeMBeanServerConnection() {
        if (this.cmdDispatcher != null && this.cmdDispatcher.isEmbedded() && this.jmxFactory != null) {
            try {
                this.jmxConnectionEmbedded.close();
                this.jmxConnectionEmbedded = null;
            } catch (ResourceException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
    }

    /**
     * @param propertyFile File name of the properties file.
     *
     * @return Parsed version of the propertyFile file.
     *
     * @throws IOException If propertyFile cannot be read.
     */
    private static Properties getProperties(final File propertyFile) throws IOException {
        Properties properties = new Properties();

        FileInputStream in = new FileInputStream(propertyFile);
        properties.load(in);
        in.close();

        return properties;
    }

    /**
     * Gets the JMX URL for a given server name.
     *
     * @param name Name of the server.
     *
     * @return JMX URL corresponding to that server name.
     */
    public static String getJmxUrl(final String name) {
        if (name == null) {
            return jmxTargets.get(defaultTarget);
        } else if (name.startsWith("service:jmx:rmi")) {
            return name;
        } else {
            return jmxTargets.get(name);
        }
    }
    
    /**
     * @return An array containing the names of all current JMX targets.
     */
    public static String[] getJmxTargets() {

        String[] targets = null;
        if (!jmxTargets.isEmpty()) {
            targets = new String[jmxTargets.size()];
            targets = jmxTargets.keySet().toArray(targets);
        }
        return targets;
    }

    /**
     * Gets the names of all JMX targets matching a certain criteria.
     *
     * @param proposed Criteria.
     *
     * @return An array containing the names of all current JMX targets.
     */
    public static String[] getJmxTargets(final String[] proposed) {
        boolean all = false;
        String[] targetsArray = null;
        List<String> targetsList = new LinkedList<String>();

        // Look for defined target
        if (proposed != null) {
            for (int i = 0; i < proposed.length; i++) {
                if ("all".equalsIgnoreCase(proposed[i])) {
                    all = true;
                    break;
                }
                String url = getJmxUrl(proposed[i]);
                if (url != null) {
                    targetsList.add(proposed[i]);
                }
            }
        } else {
            if (defaultTarget != null) {
                targetsList.add(defaultTarget);
            }
        }

        if (all) {
            targetsArray = getJmxTargets();
        } else {
            if (targetsList.size() > 0) {
                targetsArray = new String[targetsList.size()];
                for (int i = 0; i < targetsList.size(); i++) {
                    targetsArray[i] = targetsList.get(i);
                }
            }
        }
        return targetsArray;
    }

    /**
     * @return Default target.
     */
    public static String getDefaultTarget() {
        return defaultTarget;
    }

    /**
     * Loads all JMX jmxConnection properties from a file.
     */
    private static void load() {
        
        Log logger = LogFactory.getLog(JmxAP.class);
        
        // Default target is derived from the jasmine.jmx.url. It may also be
        // defined in the property file using the jasmine.jmx.default.target
        // property. If both are set, jasmine.jmx.url takes precedence.
        String dftTarget = null;
        String url = System.getProperty("jasmine.jmx.url");
        String user = System.getProperty("jasmine.jmx.user");
        String password = System.getProperty("jasmine.jmx.password");
        String filename = System.getProperty("jasmine.jmx.file");
        if (filename == null) {
            filename = DEFAULT_FILE;
        }
        String name = null;
        if (url != null) {
            name = "jonas";
            if (url.matches(".*connector_.*")) {
                int ix = url.lastIndexOf("connector_");
                name = url.substring(ix + "connector_".length());
            }
            jmxTargets.put(name, url);
            if (user != null && password != null) {
                jmxUsers.put(url, user);
                jmxPasswords.put(url, password);
            }
            defaultTarget = name;
        }

        // Load property file
        Properties props = null;
        try {
            props = getProperties(new File(filename));
        } catch (IOException e) {
            // If the file properties is not found in the current directory, try to
            // find it in the JONAS_BASE/conf one.
            try {
                Class.forName("org.ow2.jonas.lib.bootstrap.JProp");
                props = getProperties(new File(org.ow2.jonas.lib.bootstrap.JProp.getConfDir(), filename));
                if (logger.isDebugEnabled()) {
                    logger.debug("class loaded : org.ow2.jonas.lib.bootstrap.JProp");
                }
                
            } catch (Exception e3) {
                logger.warn("Cannot read default JMX configuration file {0} : {1}", filename,e3.getMessage());
            }
        }

        if ((props != null) && (props.size() != 0)) {
            /*
             * Parse the property file CAUTION: enumeration is alphabetically
             * ordered, so proceed in two passes - first pass collects the URLs,
             * because URLs must be set before user/passwords - second pass
             * collects the user and passwords
             */
            // first pass
            Enumeration<?> en = props.propertyNames();
            while (en.hasMoreElements()) {
                String key = (String) en.nextElement();
                if (key.startsWith("jasmine.jmx.url.")) {
                    name = key.substring("jasmine.jmx.url.".length());
                    url = props.getProperty(key);
                    jmxTargets.put(name, url);
                }
            }

            // second pass
            en = props.propertyNames();
            while (en.hasMoreElements()) {
                String key = (String) en.nextElement();
                if (key.startsWith("jasmine.jmx.user.")) {
                    name = key.substring("jasmine.jmx.user.".length());
                    url = jmxTargets.get(name);
                    if (url == null) {
                        logger.warn("Cannot set user for target {0}", name);
                        continue;
                    }
                    jmxUsers.put(url, props.getProperty(key));
                } else if (key.startsWith("jasmine.jmx.password.")) {
                    name = key.substring("jasmine.jmx.password.".length());
                    url = jmxTargets.get(name);
                    if (url == null) {
                        logger.warn("Cannot set password for target {0}", name);
                        continue;
                    }
                    jmxPasswords.put(url, props.getProperty(key));
                } else if (key.startsWith("jasmine.jmx.protocol.providers.")) {
                    name = key.substring("jasmine.jmx.protocol.providers.".length());
                    url = jmxTargets.get(name);
                    if (url == null) {
                        logger.warn("Cannot set protocol providers for target {0}", name);
                        continue;
                    }
                    jmxProtocolProviderPackages.put(url, props.getProperty(key));
                } else if ("jasmine.jmx.default.target".equals(key)) {
                    if (defaultTarget == null) {
                        dftTarget = props.getProperty(key);
                    }
                }
            }
        }

        // Check that the default target belongs to the targets
        if (defaultTarget == null) {
            if (dftTarget != null) {
                if (jmxTargets.get(dftTarget) != null) {
                    defaultTarget = dftTarget;
                } else {
                    logger.warn("jasmine.jmx.default.target property specifies an invalid default target {0}. No default target set.", dftTarget);
                }
            } else {
                // use default URL as default target
                defaultTarget = "jonas";
                jmxTargets.put(defaultTarget, DEFAULT_URL);
                logger.warn("jasmine.jmx.default.target property not set. Setting default target to: {0}; setting jmxUrl to {1}.", defaultTarget,DEFAULT_URL);
            }
        }
    }

    /**
     * Current JMX jmxConnection.
     */
    private MBeanServerConnection jmxConnection = null;

    /**
     * Current JMX jmxConnection in embedded mode.
     */
    private IJMXConnectionServer jmxConnectionEmbedded = null;

    /**
     * Current JMX URL.
     */
    private String jmxUrl = DEFAULT_URL;

    /**
     * Map of JMX targets.
     */
    private static Map<String, String> jmxTargets = new TreeMap<String, String>();

    /**
     * Map of JMX user names.
     */
    private static Map<String, String> jmxUsers = new TreeMap<String, String>();

    /**
     * Map of JMX passwords.
     */
    private static Map<String, String> jmxPasswords = new TreeMap<String, String>();

    /**
     * Map of protocol provider package list. The provider packages are read
     * from the jasmine.jmx.protocol.providers.<target> property Useful for non
     * standard protocols such as t3 Example: if the JMX URL is:
     * service:jmx:t3:/
     * /frecb000650:7001/jndi/weblogic.management.mbeanservers.runtime then the
     * protocol provider is: weblogic.management.remote see also:
     * jmx.remote.protocol.provider.pkgs property in JMXConnectorFactory
     * definition
     * 
     */
    private static Map<String, String> jmxProtocolProviderPackages = new TreeMap<String, String>();

    /**
     * Default JMX target.
     */
    private static String defaultTarget = null;

    /**
     * Default JMX URL.
     */
    public static final String DEFAULT_URL = "service:jmx:rmi:///jndi/rmi://localhost:1099/jrmpconnector_jonas";

    /**
     * Default name of the properties file.
     */
    public static final String DEFAULT_FILE = "jmxurls.properties";

    /**
     * The JMX factory to get connection in embedded mode.
     */
    private IJMXConnectionFactory jmxFactory = null;

    
    /**
     * Logger
     */
    protected Log logger = LogFactory.getLog(JmxAP.class);
        
    static {
        load();
    }
}
