/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Poll.java 3140 2009-03-02 09:15:12Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.File;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.PipedOutputStream;
import java.io.PrintStream;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAP;
import org.ow2.jasmine.monitoring.mbeancmd.Outer;
import org.ow2.jasmine.monitoring.mbeancmd.graph.Grapher;
import org.ow2.jasmine.monitoring.mbeancmd.graph.conf.Configurator;
import org.ow2.jasmine.monitoring.mbeancmd.jasmine.JasmineConnector;
import org.ow2.jasmine.monitoring.mbeancmd.sampling.SampleData;
import org.ow2.jasmine.monitoring.mbeancmd.sampling.XServerSampler;

/**
 * Polls multiple J2EE servers at the given period.
 */
public class Poll extends AbstractCommand {
    /**
     * Constructor: calls {@link Poll#setOptions()}.
     */
    public Poll() {
        this.setOptions();
    }

    /**
     * Implementation of inherited abstract method.
     *
     * Will never return except in the case of failure.
     *
     * @see AbstractCommand#exec()
     */
    public int exec(final CommandDispatcher cmdDispatcher) {
        try {
            parseCommandLine(arguments);
        } catch (Exception e) {
            e.printStackTrace();
            return 1;
        }

        if (this.isDtdOption) {
            printDtd();
            return 0;
        }

        XServerSampler xss = null;
        if (this.isHttpOption) {
            xss = new XServerSampler("web");
        } else if (this.isTxOption) {
            xss = new XServerSampler("tx");
        } else if (this.isCpuSunOption) {
            xss = new XServerSampler("cpusun");
        } else if (this.isServerOption) {
            xss = new XServerSampler("server");
        } else if (this.isJCACFSourceOption) {
            xss = new XServerSampler("jcacf");
            xss.setOnPattern(onPattern);
        } else if (this.isJoramQueueOption) {
            xss = new XServerSampler("joramQueue");
            xss.setOnPattern(onPattern);
        } else if (this.isDataSourceOption) {
            xss = new XServerSampler("datasource");
            xss.setOnPattern(onPattern);
        } else if (this.isServletOption) {
            xss = new XServerSampler("servlet");
            xss.setOnPattern(onPattern);
        } else if (this.isStatelessOption) {
            xss = new XServerSampler("stateless");
            xss.setOnPattern(onPattern);
        } else if (this.isStatefulOption) {
            xss = new XServerSampler("stateful");
            xss.setOnPattern(onPattern);
        } else if (this.isEntityOption) {
            xss = new XServerSampler("entity");
            xss.setOnPattern(onPattern);
        }

        if (xss == null) {
            System.err.println("Option missing!");
            return 2;
        }

        String[] managee = JmxAP.getJmxTargets(commandLine.getOptionValues("target"));

        if ((managee != null) && (managee.length > 0)) {
            for (int i = 0; i < managee.length; i++) {
                xss.addManagee(managee[i], JmxAP.getJmxUrl(managee[i]));
            }
        } else {
            System.err.println("No target specified!");
            return 2;
        }

        /*
         * out (or pout) is : - the output of the sampler - the input for the
         * Outer to print into System.out or into a file
         *
         * sink is: - the sink of the Outer - the input for the graph, if any
         */
        PipedOutputStream out = new PipedOutputStream();
        PrintStream pout = new PrintStream(out);

        Outer outer = null;
        try {
            if (this.outputFilePath != null) {
                outer = new Outer(out, new File(outputFilePath));
            } else if (this.jasmineURI == null) {
                outer = new Outer(out, System.out);
            }

            if (this.jasmineURI != null) {
                if (outer == null) {
                    outer = JasmineConnector.connect(out, this.jasmineURI);
                } else {
                    Outer oldOuter = outer;
                    outer = JasmineConnector.connect(outer, this.jasmineURI);
                    new Thread(oldOuter).start();
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        if (this.graphDef != null) {
            try {
                PipedOutputStream sink = new PipedOutputStream();
                outer.setSink(sink);
                Grapher gr = new Grapher(sink, this.graphDef);
                gr.start();
            } catch (IOException e) {
                e.printStackTrace(System.err);
            }
        }
        new Thread(outer).start();

        xss.process(period, pout, cmdDispatcher);
        return 0;
    }

    /**
     * Calls {@link AbstractCommand#help()} followed by {@link Poll#printDtd()}.
     */
    public void help() {
        super.help();
        System.out.println("DTD for graph definitions:\n");
        printDtd();
    }

    /**
     * Prints out the DTD for graph XML.
     */
    private void printDtd() {
        LineNumberReader ln = Configurator.getDTD();
        if (ln != null) {
            try {
                String line = null;
                while ((line = ln.readLine()) != null) {
                    System.out.println(line);
                }
            } catch (IOException e) {
                e.printStackTrace(System.err);
            }
        }
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see AbstractCommand#summary()
     */
    public String summary() {
        return "Probes one or more J2EE servers and prints statistics";
    }

    /**
     * Tests the Poll implementation.
     *
     * @param args
     *            Arguments to pass to Poll#setArgs.
     */
    public static void main(final String[] args) {
        Poll p = new Poll();
        p.setArgs("poll", args);
        p.exec(null);
    }

    /**
     * Parses the command line arguments into {@link Poll#commandLine}.
     *
     * @param args
     *            Arguments to parse.
     *
     * @throws ParseException
     *             If parsing fails.
     */
    public void parseCommandLine(final String[] args) throws ParseException {
        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);
        isHttpOption = commandLine.hasOption("http");
        isTxOption = commandLine.hasOption("tx");
        isCpuSunOption = commandLine.hasOption("cpusun");
        isServerOption = commandLine.hasOption("server");
        isDtdOption = commandLine.hasOption("dtd");
        isDataSourceOption = commandLine.hasOption("ds");
        if (isDataSourceOption) {
            onPattern = commandLine.getOptionValue("ds");
        }
        isJCACFSourceOption = commandLine.hasOption("jcacf");
        if (isJCACFSourceOption) {
            onPattern = commandLine.getOptionValue("jcacf");
        }
        isJoramQueueOption = commandLine.hasOption("joramq");
        if (isJoramQueueOption) {
            onPattern = commandLine.getOptionValue("joramq");
        }
        isServletOption = commandLine.hasOption("servlet");
        if (isServletOption) {
            onPattern = commandLine.getOptionValue("servlet");
        }
        isStatelessOption = commandLine.hasOption("slb");
        if (isStatelessOption) {
            onPattern = commandLine.getOptionValue("slb");
        }
        isStatefulOption = commandLine.hasOption("sfb");
        if (isStatefulOption) {
            onPattern = commandLine.getOptionValue("sfb");
        }
        isEntityOption = commandLine.hasOption("ent");
        if (isEntityOption) {
            onPattern = commandLine.getOptionValue("ent");
        }
        String s = commandLine.getOptionValue("p");
        if (s != null) {
            period = Long.parseLong(s);
        }
        if (commandLine.hasOption("graph")) {
            this.graphDef = commandLine.getOptionValue("graph");
        }
        if (commandLine.hasOption("f")) {
            this.outputFilePath = commandLine.getOptionValue("f");
        }
        if (commandLine.hasOption("jasmine")) {
            this.jasmineURI = commandLine.getOptionValue("jasmine");
        }
        if (commandLine.hasOption("s")) {
            this.separator = commandLine.getOptionValue("s");
            Outer.setSeparator(separator);
            SampleData.setSeparator(separator);
        }
    }

    /**
     * Sets all options that are parseable from the command line.
     *
     * @see Poll#options
     */
    private void setOptions() {
        options = new Options();
        // options.addOption("v", "verbose",false,"Verbose mode");

        // Period
        Option period = new Option("p", "period", true, "Polling period");
        period.setRequired(false);
        period.setArgName("seconds");
        period.setArgs(1);
        options.addOption(period);

        // Group of Exclusive methods
        OptionGroup polls = new OptionGroup();
        polls.setRequired(false);

        // http
        Option http = new Option("http", "http", false, "HTTP Statistics");
        polls.addOption(http);

        // tx
        Option tx = new Option("tx", "tx", false, "Transaction Statistics");
        polls.addOption(tx);

        // CpuSun
        Option cpuSun = new Option("cpusun", "cpusun", false, "Sun CPU Statistics");
        polls.addOption(cpuSun);

        // server
        Option svr = new Option("server", "server", false, "Server Statistics");
        polls.addOption(svr);

        // datasource
        Option ds = new Option("ds", "ds", false, "DataSource Statistics");
        ds.setArgs(1);
        ds.setArgName("mbean pattern");
        polls.addOption(ds);

        // JCA CF
        Option jcacf = new Option("jcacf", "jcacf", false, "JCA Connection Factory Statistics");
        jcacf.setArgs(1);
        jcacf.setArgName("mbean pattern");
        polls.addOption(jcacf);

        // Joram Queues
        Option joramq = new Option("joramq", "joramq", false, "Joram Queues Statistics");
        joramq.setArgs(1);
        joramq.setArgName("mbean pattern");
        polls.addOption(joramq);

        // Servlet
        Option servlet = new Option("servlet", "servlet", true, "Servlet Statistics");
        servlet.setArgs(1);
        servlet.setArgName("mbean pattern");
        polls.addOption(servlet);

        // Stateless Session beans
        Option stateless = new Option("slb", "stateless", true, "Stateless Session bean Statistics");
        stateless.setArgs(1);
        stateless.setArgName("mbean pattern");
        polls.addOption(stateless);

        // Stateful Session beans
        Option stateful = new Option("sfb", "stateful", true, "Stateful Session bean Statistics");
        stateful.setArgs(1);
        stateful.setArgName("mbean pattern");
        polls.addOption(stateful);

        // Entity beans
        Option entity = new Option("ent", "entity", true, "Stateful Session bean Statistics");
        entity.setArgs(1);
        entity.setArgName("mbean pattern");
        polls.addOption(entity);

        // mbean
        Option mbean = new Option("mbean", "mbean", true, "Any MBean Statistics");
        mbean.setArgs(128);
        mbean.setArgName("mbean pattern and attributes");
        polls.addOption(mbean);

        // dtd
        Option dtd = new Option("dtd", "dtd", false, "DTD for graph definitions");
        polls.addOption(dtd);

        // Field separator
        Option separator = new Option("s", "separator", true, "Set the column separator");
        separator.setRequired(false);
        separator.setArgName("separator");
        separator.setArgs(1);
        options.addOption(separator);

        options.addOptionGroup(polls);

        // graph output
        Option graph = new Option("graph", "graph", true, "Enable graphical output");
        graph.setRequired(false);
        graph.setArgName("graphDefinition");
        graph.setArgs(1);
        options.addOption(graph);

        // file output
        Option file = new Option("f", "file", true, "Send output to file instead of stdout");
        file.setRequired(false);
        file.setArgName("path");
        file.setArgs(1);
        options.addOption(file);

        // jasmine connector output
        Option jasmine = new Option("jasmine", "jasmine", true, "Output logged data to jasmine (will disable stdout)");
        jasmine.setRequired(false);
        jasmine.setArgName("jasmineURI");
        jasmine.setArgs(1);
        options.addOption(jasmine);

        // Optional attribute target
        Option target = new Option("target", "target", true, "instances to poll");
        target.setRequired(false);
        target.setOptionalArg(true);
        target.setArgs(Option.UNLIMITED_VALUES);
        target.setArgName("instances");
        options.addOption(target);

    }

    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * Poll period in seconds.
     */
    private long period = 10;

    /**
     * Print statistics on Http/AJP connectors ?
     */
    private boolean isHttpOption = false;

    /**
     * Print overall statistics ?
     */
    private boolean isServerOption = false;

    /**
     * Print statistics on transactions ?
     */
    private boolean isTxOption = false;

    /**
     * Print statistics on the CPU usage ?
     */
    private boolean isCpuSunOption = false;

    /**
     * Prints statistics on datasources ?
     */
    private boolean isDataSourceOption = false;

    /**
     * Print statistics on JCA connection factories ?
     */
    private boolean isJCACFSourceOption = false;

    /**
     * Print statistics on Joram Queues ?
     */
    private boolean isJoramQueueOption = false;

    /**
     * Print statistics on servlets ?
     */
    private boolean isServletOption = false;

    /**
     * Print statistics on stateless session EJBs ?
     */
    private boolean isStatelessOption = false;

    /**
     * Print statistics on stateful session EJBs ?
     */
    private boolean isStatefulOption = false;

    /**
     * Print statistics on entity EJBs ?
     */
    private boolean isEntityOption = false;

    /**
     * Graph definitions, null for no graph output.
     */
    private String graphDef = null;

    /**
     * Output file path, null for no file output.
     */
    private String outputFilePath = null;

    /**
     * URI of the JASMINe event switch, null for no JASMINe output.
     */
    private String jasmineURI = null;

    /**
     * Print DTD for the graph XML format ?
     */
    private boolean isDtdOption = false;

    /**
     * Object name filtering pattern.
     */
    private String onPattern = null;

    /**
     * Separator to use.
     */
    private String separator = ";";
}
