/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.IOException;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.util.LinkedList;
import java.util.TreeSet;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAP;
import org.ow2.jasmine.monitoring.mbeancmd.Outer;
import org.ow2.jasmine.monitoring.mbeancmd.audit.AuditContext;
import org.ow2.jasmine.monitoring.mbeancmd.audit.Collector;
import org.ow2.jasmine.monitoring.mbeancmd.audit.Metric;
import org.ow2.jasmine.monitoring.mbeancmd.audit.MetricComparator;
import org.ow2.jasmine.monitoring.mbeancmd.audit.MetricSorter;
import org.ow2.jasmine.monitoring.mbeancmd.audit.PollID;
import org.ow2.jasmine.monitoring.mbeancmd.audit.util.MetricMath;

/**
 * Polls a set of target and prints a sorted output.
 * Useful to rapidly detect what is moving ! 
 */
public class Snap extends AbstractCommand {
    /**
     * Constructor: calls {@link Snap#setOptions()}.
     */
    public Snap() {
        setOptions();
    }

    /**
     * Implementation of inherited abstract method.
     * 
     * Will never return except in the case of failure.
     * 
     * @see AbstractCommand#exec()
     */
    public int exec(final CommandDispatcher cmdDispatcher) {

        this.cmdDispatcher = cmdDispatcher;

        /*
         * Set output and error streams
         */
        PipedOutputStream out = new PipedOutputStream();
        pout = new PrintStream(out);

        PipedOutputStream err = new PipedOutputStream();
        perr = new PrintStream(err);

        Outer outer = null;
        Outer outer_err = null;
        try {
            outer = new Outer(out, System.out);
            outer_err = new Outer(err, System.err);
        } catch (IOException e) {
            // Nothing to do, cannot append
        }
        new Thread(outer_err).start();
        new Thread(outer).start();

        /*
         * Parse command
         */
        try {
            parseCommandLine(arguments);
        } catch (Exception e) {
            e.printStackTrace(perr);
            pout.close();
            perr.close();
            return 1;
        }

        /*
         * Process command
         */

        if (context != null) {
            Collector collector = new Collector();
            collector.setContext(context);
            collector.setOnPattern(on);
            collector.setAttributes(attributes);
            collector.setPerr(perr);

            PollID baseline = null;
            PollID id = null;            
            MetricSorter sorter = new MetricSorter();
            
            if (!isDelta) {
                id = collector.poll("snap");
            } else {

                // baseline poll
                baseline = collector.poll("baseline");

                // waiting for a while
                sleep(timeInterval);

                // second poll
                id = collector.poll("snap");

                /*
                for (Metric m : collector.getPoll(baseline)) {
                    bmap.put(m.getRadical(), m);
                }
                replace by*/
                sorter.setBaseline(collector.getPoll(baseline));                
            }

            for (int i = 0; i < sort_attr.length; i++) {

                TreeSet<Metric> tset = sorter.sort(collector.getPoll(id), sort_attr[i], sortMode);

                LinkedList<Metric> mlist = new LinkedList<Metric>();  
                for (Metric m : tset) {
                    mlist.add(m);
                }
                perr.println("\nPoll size : " + tset.size() + " ordered by " + sort_attr[i] + " - sum: "
                        + MetricMath.sum(sort_attr[i], mlist));
                perr.flush();

                // Print results
                MetricSorter.printMetrics(pout, tset, sort_attr[i],printZeroValue);
            }

        } else {
            perr.println("No target.");
        }

        pout.close();
        perr.close();
        return 0;
    }

    /**
     * Sleep for a while. 
     * @param delai
     *            is the time to sleep, in milliseconds.
     */
    public static void sleep(long delai) {

        try {
            Thread.sleep(delai);
        } catch (InterruptedException e) {
            // Nothing to do
        }
    }

    /**
     * Implementation of inherited abstract method.
     * 
     * @see AbstractCommand#summary()
     */
    public String summary() {
        return "Audit MBean in multiple serveurs and prints a sorted output";
    }

    /**
     * Parses the command line arguments into {@link Snap#commandLine}.
     * 
     * @param args
     *            Arguments to parse.
     * 
     * @throws ParseException
     *             If parsing fails.
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    public void parseCommandLine(final String[] args) throws ParseException, MalformedObjectNameException {
        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);

        setObjectName();

        this.attributes = this.commandLine.getOptionValues("a");
        if (attributes == null) {
            throw new ParseException("Missing argument : -a option requires an attribute list");
        }

        /*
         * If -s is not specified, then use the first attribute of the list to
         * sort the mbeans. Check the sort attributes are in the list of
         * collected attributes.
         */
        if (commandLine.hasOption("s")) {
            this.sort_attr = commandLine.getOptionValues("s");

            for (int i = 0; i < sort_attr.length; i++) {
                boolean found = false;
                for (int j = 0; j < attributes.length; j++) {
                    if (sort_attr[i].equals(attributes[j])) {
                        found = true;
                    }
                }
                if (!found) {
                    throw new ParseException("The sort attribute " + sort_attr[i]
                            + " is not in the list of collected attributes.");
                }
            }
        } else {
            this.sort_attr = new String[] { attributes[0] };
        }

        if (commandLine.hasOption("m")) {
            String mode = commandLine.getOptionValue("m");
            if ("up".equalsIgnoreCase(mode)) {
                this.sortMode = MetricComparator.ASCENDING_ORDER;
            } else if ("down".equalsIgnoreCase(mode)) {
                this.sortMode = MetricComparator.DESCENDING_ORDER;
            }
        }

        if (commandLine.hasOption("z")) {
            printZeroValue = true;
        }

        /*
         * Process target option
         */
        String[] targets = JmxAP.getJmxTargets(commandLine.getOptionValues("target"));
        if ((targets != null) && (targets.length > 0)) {
            context = new AuditContext[targets.length];
            for (int i = 0; i < context.length; i++) {
                context[i] = new AuditContext();
                context[i].setName(targets[i]);
                context[i].setJmxUrl(JmxAP.getJmxUrl(targets[i]));
                context[i].setJmxap(new JmxAP(context[i].getJmxUrl(), this.cmdDispatcher));
            }
        }

        /*
         * process delta option
         */
        if (commandLine.hasOption("delta")) {
            String interval = commandLine.getOptionValue("delta", Long.toString(DEFAULT_TIME_INTERVAL/1000));
            try {
                timeInterval = Long.parseLong(interval) * 1000;
            } catch (NumberFormatException e) {
                throw new ParseException("Invalid time interval argument in the '-delta' option : " + interval);
            }
            isDelta = true;
        }
    }

    /**
     * Sets the object name based on the "name" argument in the command line.
     * 
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    private void setObjectName() throws MalformedObjectNameException {
        on = ObjectName.getInstance(commandLine.getOptionValue("name"));
    }

    /**
     * Sets all options that are parseable from the command line.
     * 
     * @see Snap#options
     */
    private void setOptions() {
        options = new Options();
        options.addOption("z", "zero", false, "print zero values (not printed per default)");

        // ObjectName
        Option name = new Option("name", "objectName", true, "An ObjectName passed to every method");
        name.setRequired(true);
        name.setArgName("mbean name");
        name.setArgs(1);
        options.addOption(name);

        // Optional attribute list
        Option atts = new Option("a", "atts", true, "attributes to poll");
        atts.setRequired(true);
        atts.setOptionalArg(false);
        atts.setArgs(Option.UNLIMITED_VALUES);
        atts.setArgName("attributes");
        options.addOption(atts);

        // Sort attribute
        Option sortAtt = new Option("s", "sort", true, "Sort according to an attribute");
        sortAtt.setRequired(false);
        sortAtt.setOptionalArg(false);
        sortAtt.setArgName("sort_attribute");
        // sortAtt.setArgs(1);
        sortAtt.setArgs(Option.UNLIMITED_VALUES);
        options.addOption(sortAtt);

        // Sort mode
        Option sortMode = new Option("m", "mode", true, "Sort mode (ascending or descending)");
        sortMode.setRequired(false);
        sortMode.setOptionalArg(false);
        sortMode.setArgName("mode");
        sortMode.setArgs(1);
        options.addOption(sortMode);

        // baseline
        Option delta = new Option("delta", "delta", true, "computes delta from a baseline");
        delta.setRequired(false);
        delta.setOptionalArg(true);
        delta.setArgName("interval");
        delta.setArgs(1);
        options.addOption(delta);

        // Optional attribute target
        Option target = new Option("target", "target", true, "instances to poll");
        target.setRequired(false);
        target.setOptionalArg(true);
        target.setArgs(Option.UNLIMITED_VALUES);
        target.setArgName("instances");
        options.addOption(target);
    }

    /**
     * @return The ObjectName.
     */
    public String toString() {
        return on.toString();
    }

    /**
     * default time interval (in milliseconds) when processing a baseline
     * comparison
     */
    public static final long DEFAULT_TIME_INTERVAL = 60000;

    /**
     * Statistics contexts.
     */
    private AuditContext[] context = null;

    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * ObjectName to use.
     */
    private ObjectName on = null;

    /**
     * Sort attributes to use.
     */
    private String[] sort_attr = null;

    /**
     * Attributes to retrieve.
     */
    private String[] attributes = null;

    /**
     * Attributes to retrieve.
     */
    private int sortMode = MetricComparator.DESCENDING_ORDER;

    /**
     * Output stream.
     */
    private PrintStream pout = null;

    /**
     * Error stream.
     */
    private PrintStream perr = null;

    /**
     * Command dispatcher instance.
     */
    private CommandDispatcher cmdDispatcher = null;

    /**
     * Ordered TreeSet
     */
    LinkedList<Metric> metrics = new LinkedList<Metric>();

    /**
     * Flag to print "zero" values
     */
    private boolean printZeroValue = false;

    /**
     * Flag to enable baseline comparison
     */
    private boolean isDelta = false;

    /*
     * Time interval for baseline comparison: default is 1 minute
     */

    private long timeInterval = DEFAULT_TIME_INTERVAL;
}
