/**
 * JASMINe
 * Copyright (C) 2005-2008 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XServerSampler.java 2969 2009-01-12 16:17:19Z waeselynck $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.PrintStream;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;

import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAP;
import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;

/**
 * Server sampler.
 */
public class XServerSampler {
    /**
     * Constructor: saves the sampler type.
     *
     * @param type
     *            Sampler type
     */
    public XServerSampler(final String type) {
        sampleType = type;
    }

    /**
     * Adds a manangee.
     *
     * @param name
     *            Managee name.
     * @param jmxUrl
     *            JMX URL for managee.
     */
    public void addManagee(final String name, final String jmxUrl) {
        manageeURLs.put(name, jmxUrl);
    }

    /**
     * @param objectNameAsString
     *            Object name filter pattern.
     */
    public void setOnPattern(final String objectNameAsString) {
        onPattern = objectNameAsString;
    }

    /**
     * Polls the MBean.
     *
     * @param period
     *            Period of polling
     * @param out
     *            Stream to output results into.
     */
    public void mbeanPoll(final long period, final PrintStream out) {
        // Infinite sampling loop
        boolean printHeader = true;
        while (true) {
            SamplerContext ctx = null;
            SampleData data = null;

            for (int i = 0; i < context.length; i++) {
                ctx = context[i];
                data = null;
                try {
                    data = ctx.getSampler().sample(ctx.getPreviousData());
                    if (data != null) {
                        if (printHeader) {
                            out.println(data.getPrintHeader());
                            printHeader = false;
                        }
                        data.printData(out);
                    }

                    if (!ctx.isStarted()) {
                        // first line contains data header
                        ctx.setStarted(true);
                    }
                } catch (Exception e) {
                    e.printStackTrace(System.err);
                    if (ctx.isStarted()) {
                        System.err.println("Error sampling managee " + ctx.getName());
                    }
                }
                ctx.setPreviousData(data);
            }

            try {
                Thread.sleep(period * SECOND);
            } catch (InterruptedException e) {
                break;
            }
        }

    }

    /**
     * Start processing. Will never return, except on failure.
     *
     * @param period
     *            Poll period.
     * @param out
     *            Stream to output results into.
     */
    public final void process(final long period, final PrintStream out, final CommandDispatcher cmdDispatcher) {
        List<SamplerContext> ss = new LinkedList<SamplerContext>();
        Iterator<Entry<String, String>> it = manageeURLs.entrySet().iterator();
        while (it.hasNext()) {
            Entry<String, String> e = it.next();
            SamplerContext ctx = new SamplerContext();
            ctx.setName(e.getKey());

            ctx.setJmxUrl(JmxAP.getJmxUrl(ctx.getName()));
            ctx.setJmxap(new JmxAP(ctx.getJmxUrl(),cmdDispatcher));
            try {
                ObjectName jvminstance = ObjectName.getInstance("*:j2eeType=JVM,*");
                MBeanServerConnection mbscnx = ctx.getJmxap().getMBeanServerConnection();
                Iterator<?> onames = mbscnx.queryNames(jvminstance, null).iterator();

                // ObjectName is like : domain:j2eeType=JVM,name=xxx,J2EEServer=server
                while (ctx.getServer() == null) {
                    ObjectName oname = (ObjectName) onames.next();
                    ctx.setServer(oname.getKeyProperty("J2EEServer"));
                    ctx.setDomain(oname.getDomain());
                }
            } catch (Exception ex) {
                ctx.setServer("unknown_server_name");
                ctx.setDomain("unknown_domain");
            } finally {
                ctx.getJmxap().releaseMBeanServerConnection();
            }
/* OLD
            try {
                ObjectName j2eeinstance = ObjectName.getInstance("*:j2eeType=J2EEServer,*");
                MBeanServerConnection mbscnx = ctx.getJmxap().getMBeanServerConnection();
                Iterator<?> onames = mbscnx.queryNames(j2eeinstance, null).iterator();

                while (ctx.getServer() == null) {
                    ObjectName oname = (ObjectName) onames.next();
                    ctx.setServer((String) mbscnx.getAttribute(oname, "serverName"));
                    ctx.setDomain(oname.getDomain());
                }
            } catch (Exception ex) {
                ctx.setServer("unknown_server_name");
                ctx.setDomain("unknown_domain");
            } finally {
                ctx.getJmxap().releaseMBeanServerConnection();
            }

 */
            Sampler s = SamplerFactory.newSampler(sampleType, ctx);
            ctx.setSampler(s);

            if (onPattern != null) {
                ctx.getSampler().setOn(onPattern);
            }

            ctx.getSampler().init();

            ss.add(ctx);
        }
        context = ss.toArray(new SamplerContext[ss.size()]);

        /*
         * // Initialize samplers for (int i = 0; i < context.length; i++) {
         * context[i].getSampler().init(context[i].getJmxap().getMBeanServerConnection()); }
         */

        mbeanPoll(period, out);
    }

    /**
     * URLs of managees.
     */
    private Map<String, String> manageeURLs = new TreeMap<String, String>();

    /**
     * Sample type.
     */
    private String sampleType = null;

    /**
     * Object name pattern.
     */
    private String onPattern = null;

    /**
     * Array of contexts.
     */
    private SamplerContext[] context = null;

    /**
     * A second in millisecond.
     */
    private static final long SECOND = 1000;
}
