/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: AbstractCommand.java 4451 2009-07-21 12:18:24Z waeselynck $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.text.SimpleDateFormat;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * A Command which has some methods implemented.
 */
public abstract class AbstractCommand implements Command {
    /**
     * Abstract redefinition of inherited method.
     *
     * Executes the command.
     *
     * @see org.ow2.jasmine.monitoring.mbeancmd.Command#exec()
     *
     * @return 0 if succeeded, an error code otherwise.
     */
    public abstract int exec(final CommandDispatcher cmdDispatcher);

    /**
     * Implementation of inherited method.
     *
     * Retrieves the name of the command.
     *
     * @see org.ow2.jasmine.monitoring.mbeancmd.Command#getName()
     *
     * @return Name of the command.
     */
    public final String getName() {
        return this.name;
    }

    /**
     * Gets the help document corresponding to this command.
     *
     * @return Name of the help document corresponding to this command.
     */
    protected String getHelpDoc() {
        String classRadix = this.getClass().getName().substring(this.getClass().getName().lastIndexOf(".") + 1);
        String packName = this.getClass().getPackage().getName();
        String helpdoc = packName.substring(0, packName.lastIndexOf("."));
        helpdoc = helpdoc.replaceAll("[.]", "/") + "/helpdoc/" + classRadix + ".help";
        return helpdoc;
    }

    /**
     * Implementation of inherited method.
     *
     * Displays the help for this command.
     *
     * @see org.ow2.jasmine.monitoring.mbeancmd.Command#help()
     * @see AbstractCommand#getHelpDoc()
     */
    public void help() {
        InputStream in0 = Thread.currentThread().getContextClassLoader().getResourceAsStream(this.getHelpDoc());

        if (in0 != null) {
            BufferedReader in = new BufferedReader(new InputStreamReader(in0));
            OutputStreamWriter out = new OutputStreamWriter(System.out);
            try {

                boolean goOn = true;
                char[] buf = new char[2048];
                while (goOn) {
                    int read;

                    read = in.read(buf);
                    if (read > 0) {
                        out.write(buf, 0, read);
                        out.flush();
                    } else {
                        goOn = false;
                        System.out.println();
                    }
                }

            } catch (IOException e) {
                e.printStackTrace();
            } finally {

                if (in != null) {
                    try {
                        in.close();
                    } catch (IOException e) {
                        this.logger.error("Error while closing the BufferedReader : {0}", e);
                    }
                }
            }
        } else {
            System.out.println("No help available.");
        }

    }

    /**
     * Implementation of inherited method.
     *
     * Sets command name and arguments.
     *
     * @see org.ow2.jasmine.monitoring.mbeancmd.Command#setArgs(java.lang.String,
     *      java.lang.String[])
     *
     * @param cmd Command name.
     * @param args Command arguments.
     */
    public final void setArgs(final String cmd, final String[] args) {
        this.name = cmd.toLowerCase();
        this.arguments = args;
    }

    /**
     * Abstract redefinition of inherited method.
     *
     * Retrieves the command's summary.
     *
     * @see org.ow2.jasmine.monitoring.mbeancmd.Command#summary()
     *
     * @return Command's summary.
     */
    public abstract String summary();

    /**
     * Name of the command.
     */
    protected String name = null;

    /**
     * Command arguments.
     */
    protected String[] arguments = null;

    /**
     * Date format.
     */
    public static final SimpleDateFormat SIMPLEDATEFORMAT = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

    /**
     * Logger.
     */
    protected Log logger = LogFactory.getLog(AbstractCommand.class);
}
