/**
 * JASMINe
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Dump.java 2977 2009-01-14 15:30:24Z loverad $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAP;

/**
 * Command that dumps JDBC Connections information on a given J2EE server.
 */
public class JDBCConnections extends AbstractCommand {

    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * Command dispatcher instance.
     */
    private CommandDispatcher cmdDispatcher = null;

    /**
     * Output file path, null for no file output.
     */
    private String outputFilePath = null;

    /**
     * Connection open time duration filter value.
     */
    private Integer duration;

    /**
     * ObjectName of a JDBCDataSource MBean (as a String).
     */
    private String dataSourceName;

    /**
     * Constructor: calls {@link JDBCConnections#setOptions()}.
     */
    public JDBCConnections() {
        this.setOptions();
    }

    /**
     * Implementation of inherited abstract method.
     * Executes the command.
     * @see AbstractCommand#exec()
     * @param cmdDispatcher the command dispatcher
     * @return 0 if succeeded, an error code otherwise.
     */
    @Override
    @SuppressWarnings("unchecked")
    public int exec(final CommandDispatcher cmdDispatcher) {

        this.cmdDispatcher = cmdDispatcher;

        try {
            if (arguments == null) {
                arguments = new String[1];
                arguments[0] = "";

            }
            parseCommandLine(arguments);


        } catch (Exception e) {
            e.printStackTrace();
            return 1;
        }

        // retrieve the targets
        String[] targets = JmxAP.getJmxTargets(commandLine.getOptionValues("target"));
        if ((targets != null) && (targets.length > 0)) {

            // Get the OutputStream
            // by default, print on the console
            PrintStream os = System.out;
            if (this.outputFilePath != null) {
                // the user decided to print the command result in a file
                File file = new File(this.outputFilePath);
                if (file.isDirectory()) {
                    System.err.println("'" + this.outputFilePath + "' is a directory, remove it first.");
                    return 3;
                }

                try {
                    os = new PrintStream(new FileOutputStream(file));
                } catch (FileNotFoundException e) {
                    System.err.println("Cannot open '" + this.outputFilePath + "' for write: " + e.getMessage());
                    return 3;
                }

            }

            // Write some headers
            writeHeaders(os);

            JmxAP accessPoint = null;
            try {
                ObjectName dataSourceObjectName = ObjectName.getInstance(this.dataSourceName);

                for (int i = 0; i < targets.length; i++) {

                    // Create a JMX AccessPoint
                    String serviceUrl = JmxAP.getJmxUrl(targets[i]);
                    accessPoint = new JmxAP(serviceUrl, this.cmdDispatcher);

                    // Get the server Connection
                    MBeanServerConnection connection = null;
                    connection = accessPoint.getMBeanServerConnection();

                    // Get opened (& filtered) connection IDs
                    Object[] params = new Object[] {this.duration};
                    String[] signature = new String[] {int.class.getName()};
                    int[] ids = (int[]) connection.invoke(dataSourceObjectName,
                                                          "getOpenedConnections",
                                                          params,
                                                          signature);

                    // For each, get detailed informations
                    for (int j = 0; j < ids.length; j++) {
                        Object[] parameters = new Object[] {ids[j]};
                        String[] signature2 = new String[] {int.class.getName()};
                        Map<String, Object> details = (Map<String, Object>) connection.invoke(dataSourceObjectName,
                                                              "getConnectionDetails",
                                                              parameters,
                                                              signature2);

                        // Maybe the connection has been closed (and reopen) in
                        // the meantime
                        Long openDuration = (Long) details.get("duration");
                        if ((openDuration == -1 || (openDuration.longValue() >= duration.longValue() * 1000))) {
                            writeConnectionDetails(os, details);
                            if (j != (ids.length - 1)) {
                                os
                                        .println("-------------- next connection -----------------");
                            }
                        }
                    }
                }
            } catch (Exception e) {
                System.err.println(e.getMessage());
                return 2;
            } finally {
                if (accessPoint != null) {
                    accessPoint.releaseMBeanServerConnection();
                }
                if (os != null) {
                    os.close();
                }
            }
        } else {
            System.err.println("Target not found.");
            return 1;
        }
        return 0;
    }

    /**
     * Headers of the dump result.
     * @param stream the stream for write the result
     */
    private static void writeHeaders(PrintStream stream) {
        stream.println("# ---------------------------------------------------");
        stream.println("# Generated by MBeanCmd (" + new Date() + ")");
        stream.println("# ---------------------------------------------------");
    }

    /**
     * Write the given details of the connection on the given stream.
     * @param stream the stream for write the output
     * @param details the connection's detail
     */
    @SuppressWarnings("unchecked")
    private void writeConnectionDetails(PrintStream stream, Map<String, Object> details) {

        Integer id = (Integer) details.get("id");
        stream.println("Connection ID: " + id);

        Integer openCount = (Integer) details.get("open-count");
        stream.println("Usage count: " + openCount);

        Boolean status = (Boolean) details.get("inactive");
        stream.println("Is inactive: " + status);

        long duration = ((Long) details.get("duration")).longValue();
        if (duration != -1) {
            long durationDays = getNumberOfDays(duration);
            long durationHours = getNumberOfHours(duration, durationDays);
            long durationMinutes = getNumberOfMinutes(duration, durationDays,
                    durationHours);
            long durationSeconds = getNumberOfSeconds(duration, durationDays,
                    durationHours, durationMinutes);
            stream
                    .format(
                            "Elapsed time since 'getConnection': %,d ms (%2$d days %3$d hours %4$d minutes %5$d seconds)%n",
                            duration, durationDays, // days
                            durationHours, // hours
                            durationMinutes, // minutes
                            durationSeconds); // seconds
        } else {
            stream.println("Elapsed time since 'getConnection': unavailable");
        }
        String xid = (String) details.get("transaction-id");
        stream.println("Associated Transaction Xid: " + xid);

        long age = ((Long) details.get("age")).longValue();
        long ageDays = getNumberOfDays(age);
        long ageHours = getNumberOfHours(age, ageDays);
        long ageMinutes = getNumberOfMinutes(age, ageDays, ageHours);
        long ageSeconds = getNumberOfSeconds(age, ageDays, ageHours, ageMinutes);
        stream.format("Connection age (since creation): %,d ms (%2$d days %3$d hours %4$d minutes %5$d seconds)%n",
                      age,
                      ageDays,      // days
                      ageHours,     // hours
                      ageMinutes,   // minutes
                      ageSeconds);  // seconds

        String txTimeout = (String) details.get("transaction-timeout");
        stream.println("Transaction timeout (if under Tx): " + txTimeout + " sec");

        List<Map<String, Object>> openers = (List<Map<String, Object>>) details.get("openers");
        stream.println("Opener(s)'s info:");
        writeThreadInfos(stream, openers);

        List<Map<String, Object>> closers = (List<Map<String, Object>>) details.get("closers");
        if (!closers.isEmpty()) {
            stream.println("Closer(s)'s info:");
            writeThreadInfos(stream, closers);
        }
    }

    /**
     * Extract number of seconds from given time
     * @param duration the duration
     * @param fullDays days
     * @param fullHours hours
     * @param fullMinutes minutes
     * @return number of seconds
     */
    private static long getNumberOfSeconds(long duration, long fullDays,
            long fullHours, long fullMinutes) {
        return (duration - (fullDays * 86400000) - (fullHours * 3600000) - (fullMinutes * 60000)) / 1000;
    }

    /**
     * Extract number of minutes from given time
     * @param duration the duration
     * @param fullDays days
     * @param fullHours hours
     * @return number of minutes
     */
    private static long getNumberOfMinutes(long duration, long fullDays, long fullHours) {
        return (duration - (fullDays * 86400000) - (fullHours * 3600000)) / 60000;
    }

    /**
     * Extract number of hours from given time
     * @param duration the duration
     * @param fullDays days
     * @return number of hours
     */
    private static long getNumberOfHours(long duration, long fullDays) {
        return (duration - (fullDays * 86400000)) / 3600000;
    }

    /**
     * Extract number of days from given time
     * @param duration the duration
     * @return number of minutes
     */
    private static long getNumberOfDays(long duration) {
        return duration / 86400000;
    }

    /**
     * Write the given thread information on the given stream.
     * @param stream the stream for writing the output
     * @param infos the thread info
     */
    private static void writeThreadInfos(PrintStream stream, List<Map<String, Object>> infos) {

        for (Iterator<Map<String, Object>> i = infos.iterator(); i.hasNext();) {
        	Map<String, Object> info = i.next();

            // Print general infos about the Thread
            String name = (String) info.get("thread.name");
            Long time = (Long) info.get("thread.time");
            stream.format("Thread (name:%s), at %2$td/%2$tm/%2$tY %2$tH:%2$tM:%2$tS:%2$tL%n", name, new Date(time));

            String stack = (String) info.get("thread.stack");
            stream.println("Thread StackTrace:");
            stream.print(stack);

            if (i.hasNext()) {
                stream.println(" - - - - - - - - - - -");
            }
        }

    }

    /**
     * Parses the command line arguments into {@link JDBCConnections#commandLine}.
     * @param args
     *            Arguments to parse.
     * @throws ParseException
     *             If parsing fails.
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    public void parseCommandLine(final String[] args) throws ParseException,
                                                             MalformedObjectNameException {

        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);

        this.outputFilePath = commandLine.getOptionValue("output-file", null);

           String durationFilter = commandLine.getOptionValue("duration-filter", "0");
           this.duration = Integer.valueOf(durationFilter);

           this.dataSourceName = commandLine.getOptionValue("datasource-name");
    }

    /**
     * Sets all options that can be parsed from the command line.
     * @see JDBCConnections#options
     */
    private void setOptions() {
        options = new Options();

        // file printing operation
        Option file = new Option("o",
                                 "output-file",
                                 true,
                                 "Print JDBC coonection infos in the given file");
        file.setRequired(false);
        file.setOptionalArg(false); // Sets whether this Option can have an
        // optional argument
        file.setArgName("file name"); // Sets the display name for the
        // argument value
        file.setArgs(1); // Sets the number of argument values this Option
        // can take
        options.addOption(file);

        // DataSource name
        Option name = new Option("n",
                                "datasource-name",
                                true,
                                "ObjectName of the DataSource to be observed");
        name.setRequired(true);
        name.setOptionalArg(false);
        options.addOption(name);

        // Open duration filter
        Option filter = new Option("d",
                                "duration-filter",
                                true,
                                "Filter Connections opened for more than <duration> (unit: seconds)");
        filter.setRequired(false);
        filter.setOptionalArg(false);
        options.addOption(filter);

        // target
        Option target = new Option("target", "target", true, "instances to get connection dump from");
        target.setRequired(false);
        target.setOptionalArg(true);
        target.setArgs(Option.UNLIMITED_VALUES);
        target.setArgName("instances");
        options.addOption(target);
    }

    /**
     * Abstract redefinition of inherited method.
     * Retrieves the command's summary.
     * @see AbstractCommand#summary()
     * @return Command's summary.
     */
    @Override
    public String summary() {
        return "Dumps JDBC Connections infos from a DataSource.";
    }

}
