/**
 * 
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.TreeSet;

import javax.management.MalformedObjectNameException;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.audit.FileCollector;
import org.ow2.jasmine.monitoring.mbeancmd.audit.Metric;
import org.ow2.jasmine.monitoring.mbeancmd.audit.MetricComparator;
import org.ow2.jasmine.monitoring.mbeancmd.audit.MetricSorter;

/**
 * @author waeselyf
 * 
 */
public class Scan extends AbstractCommand {

    /**
     * Constructor: calls {@link Scan#setOptions()}.
     */
    public Scan() {
        setOptions();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand#exec(org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher)
     */
    @Override
    public int exec(CommandDispatcher cmdDispatcher) {

        this.cmdDispatcher = cmdDispatcher;

        /*
         * Parse command
         */
        try {
            parseCommandLine(arguments);
        } catch (Exception e) {
            e.printStackTrace(System.err);
            return 1;
        }

        /*
         * Process command
         */
        
        // set the input stream
        FileCollector fc = null;
        if (this.inputFile != null) {
            try {
                fc = new FileCollector(new File(this.inputFile));
            } catch (FileNotFoundException e) {
                e.printStackTrace(System.err);
                return 1;
            }
        } else {
            fc = new FileCollector(System.in);
        }

        if (isSeparator) {
            fc.setSeparator(separator);
        }
        fc.setAttributes(sort_attr);
        if (isStats) {
            execStatistics(fc);
        } else {
            execNoStatistics(fc);
        }
        return 0;
    }

    /**
     * 
     * @param fc
     */
    private void execStatistics(FileCollector fc) {
        fc.enableStatistics();
        fc.scan();
        MetricSorter sorter = new MetricSorter();
        for (int i = 0; i < sort_attr.length; i++) {
            TreeSet<Metric> tset = sorter.sort(fc.getStats(sort_attr[i]), sort_attr[i],sortMode);
            MetricSorter.printMetrics(System.out, tset, sort_attr[i],printZeroValue,true);
            System.out.println();
        }        
    }

    /**
     * 
     * @param fc
     */
    private void execNoStatistics(FileCollector fc) {
        fc.scan();
        MetricSorter sorter = new MetricSorter();
        if (isDelta) {
            sorter.setBaseline(fc.getBaseline());
        }

        for (int i = 0; i < sort_attr.length; i++) {
            TreeSet<Metric> tset = sorter.sort(fc.getPoll(), sort_attr[i],sortMode);
            MetricSorter.printMetrics(System.out, tset, sort_attr[i],printZeroValue);
            System.out.println();
        }        
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand#summary()
     */
    @Override
    public String summary() {
        return "Scan a file or the standard input and prints a sorted output";
    }

    /**
     * @param args
     */
    public static void main(String[] args) {
        // TODO Auto-generated method stub

    }

    /**
     * Sets all options that are parseable from the command line.
     * 
     * @see Snap#options
     */
    private void setOptions() {
        options = new Options();
        options.addOption("z", "zero", false, "print zero values (not printed per default)");

        // Optional attribute list
        Option atts = new Option("a", "atts", true, "attributes to poll");
        atts.setRequired(true);
        atts.setOptionalArg(false);
        atts.setArgs(Option.UNLIMITED_VALUES);
        atts.setArgName("attributes");
        options.addOption(atts);

        // Sort mode
        Option sortMode = new Option("m", "mode", true, "Sort mode (ascending or descending)");
        sortMode.setRequired(false);
        sortMode.setOptionalArg(false);
        sortMode.setArgName("mode");
        sortMode.setArgs(1);
        options.addOption(sortMode);

        // Input file
        Option ifile = new Option("i", "input", true, "Input file");
        ifile.setRequired(false);
        ifile.setArgName("file");
        ifile.setArgs(1);
        options.addOption(ifile);
        
        // Field separator
        Option separator = new Option("s", "separator", true, "Set the column separator");
        separator.setRequired(false);
        separator.setArgName("separator");
        separator.setArgs(1);
        options.addOption(separator);

        // Group of Exclusive methods
        OptionGroup oper = new OptionGroup();
        oper.setRequired(false);

        // raw (default)
        Option current = new Option("current", "current", false, "retains the most recent metrics.");
        oper.addOption(current);
        
        // baseline
        Option delta = new Option("delta", "delta", false, "computes differences between the end and the beginning.");
        oper.addOption(delta);

        // statistics
        Option stats = new Option("stat", "stat", false, "computes statistics.");
        oper.addOption(stats);

        this.options.addOptionGroup(oper);
    }

    /**
     * Parses the command line arguments into {@link Snap#commandLine}.
     * 
     * @param args
     *            Arguments to parse.
     * 
     * @throws ParseException
     *             If parsing fails.
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    public void parseCommandLine(final String[] args) throws ParseException, MalformedObjectNameException {
        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);

        this.sort_attr = this.commandLine.getOptionValues("a");
        if (sort_attr == null) {
            throw new ParseException("Missing argument : -a option requires an attribute list");
        }

        if (commandLine.hasOption("m")) {
            String mode = commandLine.getOptionValue("m");
            if ("up".equalsIgnoreCase(mode)) {
                this.sortMode = MetricComparator.ASCENDING_ORDER;
            } else if ("down".equalsIgnoreCase(mode)) {
                this.sortMode = MetricComparator.DESCENDING_ORDER;
            }
        }

        if (commandLine.hasOption("z")) {
            printZeroValue = true;
        }

        /*
         * process delta option
         */
        Option opt = this.options.getOption("current");
        String oper = options.getOptionGroup(opt).getSelected();
        if ("delta".equals(oper)) {
            isDelta = true;
        } else if ("stat".equals(oper)) {
            isStats = true;
        } else {
            // nothing to do
        }

        if (commandLine.hasOption("i")) {
            this.inputFile = commandLine.getOptionValue("i");
        }

        this.isSeparator = commandLine.hasOption("s");
        if (isSeparator) {
            this.separator = commandLine.getOptionValue("s");
        }

    }

    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * Sort attributes to use.
     */
    private String[] sort_attr = null;

    /**
     * Attributes to retrieve.
     */
    private int sortMode = MetricComparator.DESCENDING_ORDER;

      /**
     * Command dispatcher instance.
     */
    private CommandDispatcher cmdDispatcher = null;

    /**
     * Flag to print "zero" values
     */
    private boolean printZeroValue = false;

    /**
     * Flag to enable baseline comparison
     */
    private boolean isDelta = false;

    /**
     * Input file name, null for no file input.
     */
    private String inputFile = null;

    /**
     * Is separator set ?
     */
    private boolean isSeparator = true;

    /**
     * Separator to use.
     */
    private String separator = ";";

    /**
     * compute statistics ?
     */
    private boolean isStats = false;

}
