/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Stat.java 4451 2009-07-21 12:18:24Z waeselynck $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.File;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.lang.reflect.Array;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.management.openmbean.CompositeData;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAP;
import org.ow2.jasmine.monitoring.mbeancmd.Outer;
import org.ow2.jasmine.monitoring.mbeancmd.context.StatContext;
import org.ow2.jasmine.monitoring.mbeancmd.graph.Grapher;
import org.ow2.jasmine.monitoring.mbeancmd.graph.conf.Configurator;
import org.ow2.jasmine.monitoring.mbeancmd.jasmine.JasmineConnector;

/**
 * Periodically polls attributes of one or more mbeans on one or more J2EE
 * servers.
 */
public class Stat extends AbstractCommand {
    /**
     * Constructor: calls {@link Stat#setOptions()}.
     */
    public Stat() {
        setOptions();
    }

    /**
     * Tests the Stat implementation.
     * 
     * @param args
     *            Arguments to pass to Stat#setArgs.
     */
    public static void main(final String[] args) {
        Stat p = new Stat();
        p.setArgs("stat", args);
        p.exec(null);
    }

    /**
     * Implementation of inherited abstract method.
     * 
     * Will never return except in the case of failure.
     * 
     * @see AbstractCommand#exec()
     */
    public int exec(final CommandDispatcher cmdDispatcher) {

        this.cmdDispatcher = cmdDispatcher;

        try {
            parseCommandLine(arguments);
        } catch (Exception e) {
            e.printStackTrace();
            return 1;
        }

        /*
         * out (or pout) is : - the output of the sampler - the input for the
         * Outer to print into System.out or into a file
         * 
         * sink is: - the sink of the Outer - the input for the graph, if any
         */
        PipedOutputStream out = new PipedOutputStream();
        pout = new PrintStream(out);

        Outer outer = null;
        try {
            if (this.outputFilePath != null) {
                outer = new Outer(out, new File(outputFilePath));
            } else if (this.jasmineURI == null) {
                outer = new Outer(out, System.out);
            }

            if (this.jasmineURI != null) {
                if (outer == null) {
                    outer = JasmineConnector.connect(out, this.jasmineURI);
                } else {
                    Outer oldOuter = outer;
                    outer = JasmineConnector.connect(outer, this.jasmineURI);
                    new Thread(oldOuter).start();
                }
            }
        } catch (IOException e) {
            logger.error("Cannot set up file or jasmine output :  {0}", e.getMessage(),e);
        }

        if (this.graphDef != null) {
            try {
                PipedOutputStream sink = new PipedOutputStream();
                outer.setSink(sink);
                Grapher gr = new Grapher(sink, this.graphDef);
                gr.start();
            } catch (IOException e) {
                logger.error("Cannot set up graph output :  {0}", e.getMessage(),e);
            }
        }
        new Thread(outer).start();

        process();
        return 0;
    }

    /**
     * Calls {@link AbstractCommand#help()} and prints the DTD for the graph.
     */
    public void help() {
        super.help();

        System.out.println("\n" + "DTD for graph definitions:\n" + "--------------------------\n\n");
        LineNumberReader ln = Configurator.getDTD();
        if (ln != null) {
            try {
                String line = null;
                while ((line = ln.readLine()) != null) {
                    System.out.println(line);
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            logger.error("Error: DTD not found !");
        }
    }

    /**
     * Implementation of inherited abstract method.
     * 
     * @see AbstractCommand#summary()
     */
    public String summary() {
        return "Periodically probes MBeans in one or multiple J2EE servers";
    }

    /**
     * Parses the command line arguments into {@link Stat#commandLine}.
     * 
     * @param args
     *            Arguments to parse.
     * 
     * @throws ParseException
     *             If parsing fails.
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    public void parseCommandLine(final String[] args) throws ParseException, MalformedObjectNameException {
        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);

        setObjectName();

        if (commandLine.hasOption("p")) {
            setPeriod();
        }
        if (commandLine.hasOption("r")) {
            setRefreshPeriod();
        }

        if (commandLine.hasOption("graph")) {
            this.graphDef = commandLine.getOptionValue("graph");
        }
        if (commandLine.hasOption("f")) {
            this.outputFilePath = commandLine.getOptionValue("f");
        }
        if (commandLine.hasOption("jasmine")) {
            this.jasmineURI = commandLine.getOptionValue("jasmine");
        }

        if (commandLine.hasOption("s")) {
            this.separator = commandLine.getOptionValue("s");
            Outer.setSeparator(separator);
        }
    }

    /**
     * Sets the object name based on the "name" argument in the command line.
     * 
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    private void setObjectName() throws MalformedObjectNameException {
        on = new ObjectName(commandLine.getOptionValue("name"));
    }

    /**
     * Sets the period if the command line has the "p" option.
     * 
     * @throws NumberFormatException
     *             Number after the "p" option is invalid.
     */
    private void setPeriod() throws NumberFormatException {
        period = Long.parseLong(commandLine.getOptionValue("p"));
    }

    /**
     * Sets the refresh period if the command line has the "r" option.
     * 
     * @throws NumberFormatException
     *             Number after the "r" option is invalid, note that "never" is
     *             a valid number.
     */
    private void setRefreshPeriod() throws NumberFormatException {
        String val = commandLine.getOptionValue("r");
        if ("never".equalsIgnoreCase(val)) {
            isRefreshable = false;
        } else {
            refreshPeriod = Long.parseLong(val);
        }
    }

    /**
     * Polls the MBeans.
     */
    private void mbeanPoll() {
        String[] atts = getAttToPoll();
        MBeanServerConnection cnx = null;

        // Poll
        boolean goOn = (context != null ? true : false);
        boolean refreshNow = false;
        while (goOn) {
            // TODO: create one thread per polled server
            for (int i = 0; i < context.length; i++) {
                StatContext ctx = context[i];
                try {
                    cnx = ctx.getJmxap().getMBeanServerConnection();
                    long now = System.currentTimeMillis();
                    refreshNow = isRefreshable && (now >= ctx.getRefreshDeadLine());
                    if (refreshNow) {
                        try {
                            ctx.setOnames(cnx.queryNames(on, null));
                            ctx.setRefreshDeadLine(now + this.refreshPeriod * 1000);                            
                            logger.info("MBean discovery: succesful probe, {0}", ctx.getName());
                        } catch (IOException e) {
                            //String cause = (e.getCause() != null ? e.getCause().getMessage() : "unknown cause");                            
                            logger.error("MBean discovery: failed probe, server {0} , caught exception , message: {1}",ctx.getName(),e.getMessage());                           
                            throw e;
                        }
                    }
                    Iterator<?> it = ctx.getOnames().iterator();
                    while (it.hasNext()) {
                        long t = System.currentTimeMillis();
                        ObjectName onGot = (ObjectName) it.next();
                        AttributeList attl;
                        try {
                            /*
                             * Workarround for WLS
                             * attl = cnx.getAttributes(onGot, atts);
                             */
                            attl = getAttributes(cnx,onGot, atts,ctx.getName());
                            printMBeanPoll(t, ctx, attl, onGot);
                        } catch (Exception e) {
                            logger.error("Error on mbean {0} for {1} - {2}", onGot, ctx.getName(), e.getMessage());
                        }
                    }
                } catch (IOException e) {
                        // nothing to do
                } finally {
                    ctx.getJmxap().releaseMBeanServerConnection();
                }
            }

            // Sleep for a while
            try {
                Thread.sleep(period * 1000);
            } catch (InterruptedException e) {
                goOn = false;
            }
        }
    }

    private AttributeList getAttributes(MBeanServerConnection cnx,ObjectName on, String[] atts, String target) throws AttributeNotFoundException, InstanceNotFoundException, MBeanException, ReflectionException, IOException {
    
        AttributeList attl = null;
        boolean tryOneByOne = false;
        
        /*
         * Try to get all attributes in one shot
         */
        try {
            attl = cnx.getAttributes(on, atts);
        } catch (ArrayIndexOutOfBoundsException e) {
            /*
             * workarround for WLS, if attributes cannot be read in one shot
             * due to IOP trouble which raise an ArrayIndexOutOfBoundsException exception.
             * A warning is printed every day 
             */
            if ((warningTrigger % (86400/period))== 0){
                logger.warn("MBeanCmd stat : cannot get attributes in one shot, trying one by one ({0}) for {1}",on,target);
                warningTrigger++;
            }
            tryOneByOne = true;
        }
        if (tryOneByOne) {
            /*
             * get attributes one by one 
             */
            attl = new AttributeList(atts.length);
            for (int i = 0; i < atts.length; i++) {
                Attribute att = new Attribute(atts[i],cnx.getAttribute(on, atts[i]));
                attl.add(att);                
            }
        }
        
        return attl;
    }
    
    /**
     * @return Attributes to poll.
     */
    private String[] getAttToPoll() {
        String[] attsRequested = commandLine.getOptionValues("a");
        String[] atts = attsRequested;

        // Find an accessible instance
        MBeanServerConnection cnx = null;
        JmxAP jmxap = null;
        for (int i = 0; i < context.length; i++) {
            jmxap = context[i].getJmxap();
            cnx = jmxap.getMBeanServerConnection();
            jmxap.releaseMBeanServerConnection();
            if (cnx != null) {
                break;
            }
        }

        // Retrieve attribute names
        try {
            ObjectName onGot = null;
            cnx = jmxap.getMBeanServerConnection();
            Iterator<?> it = cnx.queryNames(on, null).iterator();
            if (it.hasNext()) {
                onGot = (ObjectName) it.next();
            }

            // TODO : manage if onGot is null

            if (attsRequested == null) {
                MBeanAttributeInfo[] info;
                if (onGot != null) {
                    info = cnx.getMBeanInfo(onGot).getAttributes();

                    atts = new String[info.length];
                    for (int i = 0; i < atts.length; i++) {
                        atts[i] = info[i].getName();
                    }
                }
            }

        } catch (Exception e) {
            //Nothing to do
        } finally {
            jmxap.releaseMBeanServerConnection();
        }
        return atts;
    }

    /**
     * Prints a given header.
     * 
     * @param attl
     *            Header attributes.
     * 
     * @return Printed header.
     */
    private String getMBeanPollHeader(final AttributeList attl) {
        if (header == null) {
            StringBuffer buf = new StringBuffer(512);
            buf.append("date");
            buf.append(separator);
            buf.append("time");
            buf.append(separator);
            buf.append("sname");
            buf.append(separator);
            buf.append("server");
            buf.append(separator);
            buf.append("domain");
            buf.append(separator);
            buf.append("mbean");
            Iterator<?> it = attl.iterator();
            while (it.hasNext()) {
                buf.append(separator);
                Attribute att = (Attribute) it.next();
                if ((att.getValue() != null) && att.getValue().getClass().isArray()) {
                    putArrayIntoHeader(buf, att);
                } else if ((att.getValue() != null) && (att.getValue() instanceof Map)) {
                    putMapIntoHeader(buf, att);
                } else if ((att.getValue() != null) && (att.getValue() instanceof CompositeData)) {
                    putCompositeDataIntoHeader(buf, att);
                } else {
                    buf.append(att.getName());
                }
            }

            header = buf.toString();
        }
        return header;
    }

    /**
     * Puts a list of attributes into a buffer.
     * 
     * @param buf
     *            String buffer to print into.
     * @param att
     *            Attribute list.
     */
    private void putArrayIntoHeader(final StringBuffer buf, final Attribute att) {
        Object obj = att.getValue();
        int length = Array.getLength(obj);
        for (int i = 0; i < length; i++) {
            buf.append((i == 0 ? "" : separator));
            buf.append(att.getName());
            buf.append("_");
            buf.append(i);
        }
    }

    /**
     * Puts a map of attributes into a buffer.
     * 
     * @param buf
     *            String buffer to print into.
     * @param att
     *            Attribute map.
     */
    private void putMapIntoHeader(final StringBuffer buf, final Attribute att) {
        Map<?, ?> map = (Map<?, ?>) att.getValue();
        TreeSet<?> ts = new TreeSet<Object>(map.keySet());
        Iterator<?> it = ts.iterator();
        String sep = "";
        while (it.hasNext()) {
            buf.append(sep);
            sep = separator;
            buf.append(att.getName());
            buf.append("_");
            buf.append(it.next());
        }
    }

    /**
     * Puts composite data into a buffer.
     * 
     * @param buf
     *            String buffer to print into.
     * @param att
     *            Attribute data.
     */
    private void putCompositeDataIntoHeader(final StringBuffer buf, final Attribute att) {
        CompositeData cdata = (CompositeData) att.getValue();
        Set<?> itemNames = cdata.getCompositeType().keySet();
        Iterator<?> it = itemNames.iterator();
        String sep = "";
        while (it.hasNext()) {
            buf.append(sep);
            sep = separator;
            buf.append(att.getName());
            buf.append("_");
            buf.append(it.next());
        }
    }

    /**
     * Prints out the poll data based on a header.
     * 
     * @param t
     *            Time.
     * @param ctx
     *            Statistics context.
     * @param attl
     *            Attribute list.
     * @param oname
     *            Object to print as data.
     */
    private void printMBeanPoll(final long t, final StatContext ctx, final AttributeList attl, final ObjectName oname) {
        if (header == null) {
            pout.println(this.getMBeanPollHeader(attl));
        }

        Date d = new Date(t);
        pout.print(SIMPLEDATEFORMAT.format(d));
        pout.print(separator);
        pout.print(t);
        pout.print(separator);
        pout.print(ctx.getJmxUrl());
        pout.print(separator);
        pout.print(ctx.getServer());
        pout.print(separator);
        pout.print(ctx.getDomain());
        pout.print(separator);
        pout.print(oname);
        Iterator<?> it = attl.iterator();
        while (it.hasNext()) {
            pout.print(separator);
            Attribute att = (Attribute) it.next();
            if ((att.getValue() != null) && att.getValue().getClass().isArray()) {
                Object obj = att.getValue();
                int length = Array.getLength(obj);
                for (int i = 0; i < length; i++) {
                    pout.print((i == 0 ? "" : separator) + Array.get(obj, i));
                }
            } else if ((att.getValue() != null) && (att.getValue() instanceof Map)) {
                Map<?, ?> map = (Map<?, ?>) att.getValue();
                TreeSet<?> ts = new TreeSet<Object>(map.keySet());
                String sep = "";
                Iterator<?> it2 = ts.iterator();
                while (it2.hasNext()) {
                    pout.print(sep);
                    sep = separator;
                    pout.print(map.get(it2.next()));
                }
            } else if ((att.getValue() != null) && (att.getValue() instanceof CompositeData)) {
                CompositeData cdata = (CompositeData) att.getValue();
                Set<?> itemNames = cdata.getCompositeType().keySet();
                Iterator<?> it2 = itemNames.iterator();
                String sep = "";
                while (it2.hasNext()) {
                    pout.print(sep);
                    sep = separator;
                    pout.print(cdata.get((String) it2.next()));
                }
            } else {
                pout.print(att.getValue());
            }
        }
        pout.println();
    }

    /**
     * Sets all options that are parseable from the command line.
     * 
     * @see Stat#options
     */
    private void setOptions() {
        options = new Options();
        options.addOption("v", "verbose", false, "Verbose mode");

        // ObjectName
        Option name = new Option("name", "objectName", true, "An ObjectName passed to every method");
        name.setRequired(true);
        name.setArgName("mbean name");
        name.setArgs(1);
        options.addOption(name);

        // Optional attribute list
        Option atts = new Option("a", "atts", true, "attributes to poll");
        atts.setRequired(false);
        atts.setOptionalArg(true);
        atts.setArgs(Option.UNLIMITED_VALUES);
        atts.setArgName("attributes");
        options.addOption(atts);

        // Period
        Option period = new Option("p", "period", true, "Polling period");
        period.setRequired(false);
        period.setArgName("period");
        period.setArgs(1);
        options.addOption(period);

        // MBean list refresh period
        Option refresh = new Option("r", "refresh", true, "Period for rebuilding the mbean list");
        refresh.setRequired(false);
        refresh.setArgName("refreshPeriod");
        refresh.setArgs(1);
        options.addOption(refresh);

        // graph output
        Option graph = new Option("graph", "graph", true, "Enable graphical output");
        graph.setRequired(false);
        graph.setArgName("graphDefinition");
        graph.setArgs(1);
        options.addOption(graph);

        // file output
        Option file = new Option("f", "file", true, "Send output to file instead of stdout");
        file.setRequired(false);
        file.setArgName("path");
        file.setArgs(1);
        options.addOption(file);

        // jasmine connector output
        Option jasmine = new Option("jasmine", "jasmine", true, "Output logged data to jasmine (will disable stdout)");
        jasmine.setRequired(false);
        jasmine.setArgName("jasmineURI");
        jasmine.setArgs(1);
        options.addOption(jasmine);

        // Field separator
        Option separator = new Option("s", "separator", true, "Set the column separator");
        separator.setRequired(false);
        separator.setArgName("separator");
        separator.setArgs(1);
        options.addOption(separator);

        // Optional attribute target
        Option target = new Option("target", "target", true, "instances to poll");
        target.setRequired(false);
        target.setOptionalArg(true);
        target.setArgs(Option.UNLIMITED_VALUES);
        target.setArgName("instances");
        options.addOption(target);
    }

    /**
     * @return The ObjectName.
     */
    public String toString() {
        return on.toString();
    }

    /**
     * Initializes data polling and calls {@link Stat#mbeanPoll()}.
     */
    private void process() {
        // retrieve the method
        String[] targets = JmxAP.getJmxTargets(commandLine.getOptionValues("target"));

        if ((targets != null) && (targets.length > 0)) {
            context = new StatContext[targets.length];
            for (int i = 0; i < context.length; i++) {
                context[i] = new StatContext();
                context[i].setName(targets[i]);
                context[i].setJmxUrl(JmxAP.getJmxUrl(targets[i]));
                context[i].setJmxap(new JmxAP(context[i].getJmxUrl(), this.cmdDispatcher));
                try {
                    ObjectName j2eeinstance = ObjectName.getInstance("*:j2eeType=J2EEServer,*");
                    MBeanServerConnection mbscnx = context[i].getJmxap().getMBeanServerConnection();
                    Iterator<?> onames = mbscnx.queryNames(j2eeinstance, null).iterator();
                    context[i].setServer(null);
                    while (context[i].getServer() == null) {
                        ObjectName server = (ObjectName) onames.next();
                        context[i].setServer((String) mbscnx.getAttribute(server, "serverName"));
                        context[i].setDomain(server.getDomain());
                    }
                } catch (Exception e) {
                    context[i].setServer("unknown_server_name");
                    context[i].setDomain("unknown_domain");
                } finally {
                    context[i].getJmxap().releaseMBeanServerConnection();
                }
                logger.info("Target {0} - {1}", context[i].getName(), context[i].getJmxUrl());
            }
            mbeanPoll();
        } else {
            logger.error("No target.");
        }
    }

    /**
     * Statistics contexts.
     */
    private StatContext[] context = null;

    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * ObjectName to use.
     */
    private ObjectName on = null;

    /**
     * Polling period in seconds.
     */
    private long period = 10;

    /**
     * Refresh period, in seconds. A value of 0 means refresh at each poll
     */
    private long refreshPeriod = 300;

    /**
     * Is content refresheable ?
     */
    private boolean isRefreshable = true;

    /**
     * Graph definitions, null for no graph output.
     */
    private String graphDef = null;

    /**
     * Output file path, null for no file output.
     */
    private String outputFilePath = null;

    /**
     * URI of the JASMINe event switch, null for no JASMINe output.
     */
    private String jasmineURI = null;

    /**
     * Output stream.
     */
    private PrintStream pout = null;

    /**
     * Separator to use.
     */
    private String separator = ";";

    /**
     * Header to use.
     */
    private String header = null;

    /**
     * Command dispatcher instance.
     */
    private CommandDispatcher cmdDispatcher = null;
    
    /**
     * IOP trouble workarround : Warning counter 
     */
    private long warningTrigger = 0;
    
}
