/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Sampler.java 4222 2009-07-01 14:32:56Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.PrintStream;

import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.context.JmxContext;

/**
 * Base class for all sampler classes.
 */
public abstract class Sampler {
    /**
     * Creates the Sampler using a JMX URL.
     *
     * @param urlString
     *            JMX URL.
     */
    public Sampler(final JmxContext context) {
        this.context = context;
    }

    /**
     * @return Connection to MBean server. Will return existing instance if
     *         already connected, and try to connect if no connection exists.
     *         Returns null on failure.
     */
    protected final MBeanServerConnection getMBeanServerConnection() {
        return this.context.getJmxap().getMBeanServerConnection();
    }

    protected void releaseMbeanServerConnection() {
        this.context.getJmxap().releaseMBeanServerConnection();
    }

    /**
     * Gets the list of object names that match a given object name prototype.
     *
     * @param onProto
     *            Object name filtering pattern to use.
     *
     * @return All objects that match onProto.
     */
    protected final ObjectName[] queryON(final String onProto) {
        ObjectName[] ons = null;
        try {
            //Set onSet = null;
            MBeanServerConnection cnx = getMBeanServerConnection();

            System.err.println("MBean discovery: starting search in server " + context.getName() + " (" + context.getServer() + ") pattern " + onProto);
            this.context.setOnames(cnx.queryNames(new ObjectName(onProto), null));
            if (!this.context.getOnames().isEmpty()) {
                System.err.println("MBean discovery: succesful probe, " + context.getName());
                ons = (ObjectName[]) this.context.getOnames().toArray(new ObjectName[this.context.getOnames().size()]);
                for (int i = 0; i < ons.length; i++) {
                    System.err.println("MBean discovery: Found ObjectName in server " + context.getName() + " (" + context.getServer() + " on domain "
                            + context.getDomain() + "): " + ons[i]);
                }
            }
            System.err.println("MBean discovery: search complete in server " + context.getName() + " (" + context.getServer() + ") pattern " + onProto);
        } catch (Exception e) {
            //String cause = (e.getCause() != null ? e.getCause().getMessage() : "unknown cause");
            System.err.println("MBean discovery: failed probe, server " + context.getName() + " , caught exception , message: " + e.getMessage());
            //e.printStackTrace(System.err);
        }
        return ons;
    }

    /**
     * @return Default Object name filtering pattern for this sampler. If null,
     *         we'll consider that the Sampler doesn't have one default object
     *         name filter.
     */
    protected abstract String getDefaultOn();

    /**
     * @return Object name filtering pattern.
     */
    protected String getOn() {
        return this.objectName;
    }

    /**
     * @param onProto
     *            Object name filtering pattern to set.
     */
    public void setOn(final String onProto) {
        this.objectName = onProto;
    }

    /**
     * Initialize the sampler, itself called by
     * {@link Sampler#process(long, PrintStream)}.
     *
     * This implementation will fill {@link Sampler#oNames} based on the object
     * name filter set using {@link Sampler#setOn(String)}.
     *
     * If no filter was set before calling this method, will call set object
     * name to {@link Sampler#getDefaultOn()}.
     *
     * If {@link Sampler#getDefaultOn()} has returned null, then
     * {@link Sampler#oNames} will remain null.
     */
    public void init() {

        if(getOn()==null) {
            setOn(getDefaultOn());
        }

        this.queryON(getOn());
    }

    /**
     * Samples data based on a previous sample.
     *
     * @param previous
     *            Previous sample.
     *
     * @return Sampled data,
     * @see SampleData#compute(SampleData)
     */
    public final SampleData sample(final SampleData previous) {
        SampleData data = newSampleData();

        try {
            data.setSampleTime(System.currentTimeMillis());
            poll(data);
            if (data.isValid()) {
                data.compute(previous);
            }
        } catch (Exception e) {
            e.printStackTrace(System.err);
        }

        if (data.isValid()) {
            return data;
        } else {
            return null;
        }
    }

    /**
     * @return Name set for the server in the jmxurls.properties file.
     */
    public String getName() {
        return this.context.getName();
    }
    
    /**
     * @return
     */
    public String getJmxUrl() {
        return this.context.getJmxUrl();
    }

    /**
     * @return Server name.
     */
    public String getServer() {
        return this.context.getServer();
    }

    /**
     * @return Server's domain name.
     */
    public String getDomain() {
        return this.context.getDomain();
    }

    /**
     * @return New sample data.
     */
    protected abstract SampleData newSampleData();

    /**
     * Poll based on a previous sample.
     *
     * @param prev
     *            Previous sample.
     *
     * @return Resulting data.
     */
    protected abstract SampleData poll(final SampleData prev);

    /**
     * Object name.
     */
    private String objectName = null;

    /**
     * context of the sampler
     */
    protected JmxContext context;

}
