/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServerData.java 2971 2009-01-12 22:13:53Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.util.HashMap;
import java.util.Map;


/**
 * Data retrieved from various parts of the J2EE server. Includes :
 *
 *      - machine cpu, if available
 *      - JVM cpu, if available
 *      - heap committed
 *      - heap used
 *      - number of threads
 *      - pending requests
 *      - current sessions
 *      - database connections
 *      - database busy cnx
 *      - waiters
 *      - dsleaks
 *      - dsfail
 *      - sfb instances
 *      - ssb instances
 *      - ent instances
 *      - committed tx
 *      - served cnx
 */
public class ServerData extends SampleData {
    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#compute(SampleData)
     */
    public void compute(final SampleData needs_cast) {
        ServerData prev = (ServerData) needs_cast;
        // Compute:
        //      - http request rate
        //      - committed transaction rate
        //      - getConnection() rate
        if (prev != null) {
            double dt = this.getSampleTime() - prev.getSampleTime();
            this.httpRate = 1000.0 * (this.getHttpRequests() - prev.getHttpRequests())/dt;
            this.txRate = 1000.0 * (this.getCommitedTx() - prev.getCommitedTx())/dt;
            this.cnxRate = 1000.0 * (this.dsServed - prev.dsServed) / dt;
        } else {
            this.httpRate = 0.0;
            this.txRate = 0.0;
            this.cnxRate = 0.0;
        }
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintHeader()
     */
    protected String getInnerPrintHeader() {
        return header.replaceAll(";", getSeparator());
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintData()
     */
    protected String getInnerPrintData() {
        return  this.getThreads() + getSeparator()
              + this.getMemory() + getSeparator()
              + this.getUsedMemory() + getSeparator()
              + this.getSessions() + getSeparator()
              + this.getPendingHttp() + getSeparator()
              + this.getPendingTx() + getSeparator()
              + this.getDSCnx() + getSeparator()
              + this.getDSCnxBusy() + getSeparator()
              + decimalFormat.format(getHttpRate()) + getSeparator()
              + decimalFormat.format(getTxRate()) + getSeparator()
              + decimalFormat.format(getCnxRate()) + getSeparator()
              + this.getDSFails() + getSeparator()
              + this.getDSWaits() + getSeparator()
              + this.getDSRejects() + getSeparator()
              + this.getDSLeaks() + getSeparator()
              + this.getHttpErrors() + getSeparator()
              + this.getRollbackTx() + getSeparator()
              + this.getEntityCacheNb() + getSeparator()
              + this.getEntityPoolNb() + getSeparator()
              + this.getSsbCacheNb() + getSeparator()
              + this.getSsbPoolNb() + getSeparator()
              + this.getSfbCacheNb() + getSeparator()
              + this.getSfbPoolNb();
    }

    @Override
    public Map<String, ?> getValueOfAttributes() {
        HashMap<String, Object> valueOfAttributes = new HashMap<String, Object>();
        valueOfAttributes.put("threads", threadsGauge);
        valueOfAttributes.put("heap", currentTotalMemory);
        valueOfAttributes.put("usedHeap", currentUsedMemory);
        valueOfAttributes.put("sessions", nbSessions);
        valueOfAttributes.put("pendingHttp", pendingHttpRequest);
        valueOfAttributes.put("pendingTx", pendingTxGauge);
        valueOfAttributes.put("dsCnx", dsCnxGauge);
        valueOfAttributes.put("dsCnxBusy", dsCnxbusy);
        valueOfAttributes.put("httpRate", httpRate);
        valueOfAttributes.put("txRate", txRate);
        valueOfAttributes.put("dsRate", cnxRate);
        valueOfAttributes.put("dsFails", dsOpenFailures);
        valueOfAttributes.put("dsWaits", dsWaits);
        valueOfAttributes.put("dsRejects", dsRejectedOpen);
        valueOfAttributes.put("dsLeaks", dsLeaks);
        valueOfAttributes.put("httpErrs", httpErrorCount);
        valueOfAttributes.put("rollbacks", txRollbackCount);
        valueOfAttributes.put("entityCache", entCacheNb);
        valueOfAttributes.put("entityFreePool", entPoolNb);
        valueOfAttributes.put("ssbCache", ssbCacheNb);
        valueOfAttributes.put("ssbFreePool", ssbPoolNb);
        valueOfAttributes.put("sfbCache", sfbCacheNb);
        valueOfAttributes.put("sfbFreePool", sfbPoolNb);
        return valueOfAttributes;
    }

    /**
     * @return  HTTP connection rate (calculated).
     */
    public double getHttpRate() {
        return httpRate;
    }

    /**
     * @return  Connection rate (calculated).
     */
    public double getCnxRate() {
        return cnxRate;
    }

    /**
     * @return  Throughput (calculated).
     */
    public double getTxRate() {
        return txRate;
    }

    /**
     * @param mem  Total memory.
     */
    public void setMemory(final long mem) {
        currentTotalMemory = mem;
    }

    /**
     * @return  Total memory.
     */
    public long getMemory() {
        return currentTotalMemory;
    }

    /**
     * @param mem  Used memory.
     */
    public void setUsedMemory(final long mem) {
        currentUsedMemory = mem;
    }

    /**
     * @return  Used memory.
     */
    public long getUsedMemory() {
        return currentUsedMemory;
    }

    /**
     * @param th  Number of threads.
     */
    public void setThreads(final int th) {
        threadsGauge = th;
    }

    /**
     * @return  Number of threads.
     */
    public int getThreads() {
        return threadsGauge;
    }

    /**
     * @param sess  Number of sessions.
     */
    public void setSessions(final long sess) {
        nbSessions = sess;
    }

    /**
     * @return  Number of sessions.
     */
    public long getSessions() {
        return nbSessions;
    }

    /**
     * @param req  Pending HTTP requests.
     */
    public void setPendingHttp(final long req) {
        pendingHttpRequest = req;
    }

    /**
     * @return  Pending HTTP requests.
     */
    public long getPendingHttp() {
        return pendingHttpRequest;
    }

    /**
     * @param req  HTTP requests.
     */
    public void setHttpRequests(final long req) {
        httpRequestCount = req;
    }

    /**
     * @return  HTTP requests.
     */
    public long getHttpRequests() {
        return httpRequestCount;
    }

    /**
     * @param req  HTTP errors.
     */
    public void setHttpErrors(final long req) {
        httpErrorCount = req;
    }

    /**
     * @return  HTTP errors.
     */
    public long getHttpErrors() {
        return httpErrorCount;
    }

    /**
     * @param tx  Pending transactions.
     */
    public void setPendingTx(final long tx) {
        this.pendingTxGauge = tx;
    }

    /**
     * @return  Pending transactions.
     */
    public long getPendingTx() {
        return pendingTxGauge;
    }

    /**
     * @param tx  Commited transactions.
     */
    public void setCommitedTx(final long tx) {
        this.txCommitedCount = tx;
    }

    /**
     * @return  Commited transactions.
     */
    public long getCommitedTx() {
        return txCommitedCount;
    }

    /**
     * @param tx  Rollbacked transactions.
     */
    public void setRollbackTx(final long tx) {
        this.txRollbackCount = tx;
    }

    /**
     * @return  Rollbacked transactions.
     */
    public long getRollbackTx() {
        return txRollbackCount;
    }

    /**
     * @param cnx  Datasource connections.
     */
    public void setDSCnx(final int cnx) {
        dsCnxGauge = cnx;
    }

    /**
     * @return  Datasource connections.
     */
    public int getDSCnx() {
        return dsCnxGauge;
    }

    /**
     * @param cnx  Busy datasource connections.
     */
    public void setDSCnxBusy(final int cnx) {
        this.dsCnxbusy = cnx;
    }

    /**
     * @return  Busy datasource connections.
     */
    public int getDSCnxBusy() {
        return dsCnxbusy;
    }

    /**
     * @param cnx  Failed datasource connections.
     */
    public void setDSFails(final int cnx) {
        dsOpenFailures = cnx;
    }

    /**
     * @return  Failed datasource connections.
     */
    public int getDSFails() {
        return dsOpenFailures;
    }

    /**
     * @param rj  Rejected datasource connections.
     */
    public void setDSRejects(final int rj) {
        dsRejectedOpen = rj;
    }

    /**
     * @return  Rejected datasource connections.
     */
    public int getDSRejects() {
        return dsRejectedOpen;
    }

    /**
     * @param lk  Datasource leaks.
     */
    public void setDSLeaks(final int lk) {
        dsLeaks = lk;
    }

    /**
     * @return  Datasource leaks.
     */
    public int getDSLeaks() {
        return dsLeaks;
    }

    /**
     * @param w  Datasource waits.
     */
    public void setDSWaits(final int w) {
        dsWaits = w;
    }

    /**
     * @return  Datasource waits.
     */
    public int getDSWaits() {
        return dsWaits;
    }

    /**
     * @return  Served datasources.
     */
    public int getDSServed() {
        return this.dsServed;
    }

    /**
     * @param dsServed  Served datasources.
     */
    public void setDSServed(final int dsServed) {
        this.dsServed = dsServed;
    }

    /**
     * @param nb  Number of entity caches.
     */
    public void setEntityCacheNb(final int nb) {
        entCacheNb = nb;
    }

    /**
     * @return  Number of entity caches.
     */
    public int getEntityCacheNb() {
        return entCacheNb;
    }

    /**
     * @param nb  Number of entity pools.
     */
    public void setEntityPoolNb(final int nb) {
        entPoolNb = nb;
    }

    /**
     * @return  Number of entity pools.
     */
    public int getEntityPoolNb() {
        return entPoolNb;
    }

    /**
     * @param nb  Number of SSB caches.
     */
    public void setSsbCacheNb(final int nb) {
        ssbCacheNb = nb;
    }

    /**
     * @return  Number of SSB caches.
     */
    public int getSsbCacheNb() {
        return ssbCacheNb;
    }

    /**
     * @param nb  Number of SSB pools.
     */
    public void setSsbPoolNb(final int nb) {
        ssbPoolNb = nb;
    }

    /**
     * @return  Number of SSB pools.
     */
    public int getSsbPoolNb() {
        return ssbPoolNb;
    }

    /**
     * @param nb  Number of SFB caches.
     */
    public void setSfbCacheNb(final int nb) {
        sfbCacheNb = nb;
    }

    /**
     * @return  Number of SFB caches.
     */
    public int getSfbCacheNb() {
        return sfbCacheNb;
    }

    /**
     * @param nb  Number of SFB pools.
     */
    public void setSfbPoolNb(final int nb) {
        sfbPoolNb = nb;
    }

    /**
     * @return  Number of SFB pools.
     */
    public int getSfbPoolNb() {
        return sfbPoolNb;
    }

    /**
     * HTTP connection rate (calculated).
     */
    private double httpRate = 0.0;

    /**
     * Connection rate (calculated).
     */
    private double cnxRate = 0.0;

    /**
     * Throughput (calculated).
     */
    private double txRate = 0.0;

    /**
     * Total memory.
     */
    private long currentTotalMemory = 0;

    /**
     * Used memory.
     */
    private long currentUsedMemory = 0;

    /**
     * Number of threads.
     */
    private int threadsGauge = 0;

    /**
     * Number of sessions.
     */
    private long nbSessions = 0;

    /**
     * Pending HTTP requests.
     */
    private long pendingHttpRequest = 0;

    /**
     * HTTP requests.
     */
    private long httpRequestCount = 0;

    /**
     * HTTP errors.
     */
    private long httpErrorCount = 0;

    /**
     * Pending transactions.
     */
    private long pendingTxGauge = 0;

    /**
     * Commited transactions.
     */
    private long txCommitedCount = 0;

    /**
     * Rollbacked transactions.
     */
    private long txRollbackCount = 0;

    /**
     * Datasource connections.
     */
    private int dsCnxGauge = 0;

    /**
     * Busy datasource connections.
     */
    private int dsCnxbusy = 0;

    /**
     * Failed datasource connections.
     */
    private int dsOpenFailures = 0;

    /**
     * Rejected datasource connections.
     */
    private int dsRejectedOpen = 0;

    /**
     * Datasource leaks.
     */
    private int dsLeaks = 0;

    /**
     * Datasource waits.
     */
    private int dsWaits = 0;

    /**
     * Served datasources.
     */
    private int dsServed = 0;

    /**
     * Number of entity caches.
     */
    private int entCacheNb = 0;

    /**
     * Number of entity pools.
     */
    private int entPoolNb = 0;

    /**
     * Number of StatelessSessionBean caches.
     */
    private int ssbCacheNb = 0;

    /**
     * Number of StatelessSessionBean pools.
     */
    private int ssbPoolNb = 0;

    /**
     * Number of StatefulSessionBean caches.
     */
    private int sfbCacheNb = 0;

    /**
     * Number of StatefulSessionBean pools.
     */
    private int sfbPoolNb = 0;

    private static String header = "threads;heap;usedHeap;sessions;pendingHttp;pendingTx;dsCnx;dsCnxBusy;httpRate;"
                                 + "txRate;dsRate;dsFails;dsWaits;dsRejects;dsLeaks;httpErrs;rollbacks;entityCache;"
                                 + "entityFreePool;ssbCache;ssbFreePool;sfbCache;sfbFreePool";
}
