/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxHelper.java 5940 2010-01-15 15:27:09Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;
import java.util.Enumeration;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

public class JmxHelper {

    /**
     * Logger
     */
    private static Log logger = LogFactory.getLog(JmxHelper.class);

    /**
     * Default JMX URL.
     */
    public static final String DEFAULT_URL = "service:jmx:rmi:///jndi/rmi://localhost:1099/jrmpconnector_jonas";

    /**
     * Default name of the properties file.
     */
    public static final String DEFAULT_FILE = "jmxurls.properties";

    /**
     * Map of JMX targets.
     */
    private static Map<String, String> jmxTargets = new TreeMap<String, String>();

    /**
     * Map of JMX user names.
     */
    private static Map<String, String> jmxUsers = new TreeMap<String, String>();

    /**
     * Map of JMX passwords.
     */
    private static Map<String, String> jmxPasswords = new TreeMap<String, String>();

    /**
     * Map of protocol provider package list. The provider packages are read
     * from the jasmine.jmx.protocol.providers.<target> property Useful for non
     * standard protocols such as t3 Example: if the JMX URL is:
     * service:jmx:t3:/
     * /frecb000650:7001/jndi/weblogic.management.mbeanservers.runtime then the
     * protocol provider is: weblogic.management.remote see also:
     * jmx.remote.protocol.provider.pkgs property in JMXConnectorFactory
     * definition
     *
     */
    private static Map<String, String> jmxProtocolProviderPackages = new TreeMap<String, String>();

    /**
     * Default JMX target.
     */
    private static String defaultTarget = null;

    /**
     * Context that is used by the JmxAps.
     */
    private static Context ctx = null;

    public static Context getContext() {
    	if (ctx != null) {
    		return ctx;
    	}
        try {
            ctx = new InitialContext();
        } catch (NamingException e) {
            logger.error("Error while creating initialContext : {0}. An unmanaged JMX connection will be used instead.", e.getMessage());
        }
        return ctx;
    }

    /**
     * The JMX URL of the default target.
     * @return JMX URL of the default target
     */
    public static String getDefaultJmxTarget() {
    	return jmxTargets.get(defaultTarget);
    }

    /**
     * @return Default target.
     */
    public static String getDefaultTarget() {
        return defaultTarget;
    }

    /**
     * @param jmxUrl A given jmx URL
     * @return The user name associated to this JMX URL
     */
    public static String getJmxUser(String jmxUrl) {
    	return jmxUsers.get(jmxUrl);
    }

    /**
     * @param jmxUrl A given jmx URL
     * @return The password associated to this JMX URL
     */
    public static String getJmxPassword(String jmxUrl) {
    	return jmxPasswords.get(jmxUrl);
    }

    /**
     *
     * @param jmxUrl jmxUrl A given jmx URL
     * @return The protocol provider package associated to this JMX URL
     */
    public static String getJmxProtocolProviderPackage(String jmxUrl) {
    	return jmxProtocolProviderPackages.get(jmxUrl);
    }

    /**
     * Gets the JMX URL for a given target.
     *
     * @param name Name of the target.
     *
     * @return JMX URL corresponding to that target.
     */
    public static String getJmxUrl(final String name) {
        if (name == null) {
            return jmxTargets.get(defaultTarget);
        } else if (name.startsWith("service:jmx:rmi")) {
            return name;
        } else {
            return jmxTargets.get(name);
        }
    }

    /**
     * @return An array containing the names of all current JMX targets.
     */
    public static String[] getJmxTargets() {

        String[] targets = null;
        if (!jmxTargets.isEmpty()) {
            targets = new String[jmxTargets.size()];
            targets = jmxTargets.keySet().toArray(targets);
        }
        return targets;
    }

    /**
     * Gets the names of all JMX targets matching a certain criteria.
     *
     * @param proposed Criteria.
     *
     * @return An array containing the names of all current JMX targets.
     */
    public static String[] getJmxTargets(final String[] proposed) {
        boolean all = false;
        String[] targetsArray = null;
        List<String> targetsList = new LinkedList<String>();

        // Look for defined target
        if (proposed != null) {
            for (int i = 0; i < proposed.length; i++) {
                if ("all".equalsIgnoreCase(proposed[i])) {
                    all = true;
                    break;
                }
                String url = getJmxUrl(proposed[i]);
                if (url != null) {
                    targetsList.add(proposed[i]);
                }
            }
        } else {
            if (defaultTarget != null) {
                targetsList.add(defaultTarget);
            }
        }

        if (all) {
            targetsArray = getJmxTargets();
        } else {
            if (targetsList.size() > 0) {
                targetsArray = new String[targetsList.size()];
                for (int i = 0; i < targetsList.size(); i++) {
                    targetsArray[i] = targetsList.get(i);
                }
            }
        }
        return targetsArray;
    }

    /**
     * Loads all JMX jmxConnection properties from a file.
     */
    private static void load() {

        Log logger = LogFactory.getLog(JmxHelper.class);

        // Default target is derived from the jasmine.jmx.url. It may also be
        // defined in the property file using the jasmine.jmx.default.target
        // property. If both are set, jasmine.jmx.url takes precedence.
        String dftTarget = null;
        String url = System.getProperty("jasmine.jmx.url");
        String user = System.getProperty("jasmine.jmx.user");
        String password = System.getProperty("jasmine.jmx.password");
        String filename = System.getProperty("jasmine.jmx.file");
        if (filename == null) {
            filename = DEFAULT_FILE;
        }
        String name = null;
        if (url != null) {
            name = "jonas";
            if (url.matches(".*connector_.*")) {
                int ix = url.lastIndexOf("connector_");
                name = url.substring(ix + "connector_".length());
            }
            jmxTargets.put(name, url);
            if (user != null && password != null) {
                jmxUsers.put(url, user);
                jmxPasswords.put(url, password);
            }
            defaultTarget = name;
        }

        // Load property file
        Properties props = null;
        try {
            props = getProperties(new File(filename));
        } catch (IOException ioe) {
            // If the file properties is not found in the current directory, try to
            // find it in the JONAS_BASE/conf one, in case the command is embedded
        	boolean isEmbeded = true;
            try {
                Class.forName("org.ow2.jonas.lib.bootstrap.JProp");
            } catch (ClassNotFoundException cnfe) {
            	logger.warn("Cannot read JMX configuration file : {0}", filename);
            	isEmbeded = false;
            }
            if (isEmbeded) {
                try {
            	    props = getProperties(new File(org.ow2.jonas.lib.bootstrap.JProp.getConfDir(), filename));
            	    if (logger.isDebugEnabled()) {
                       logger.debug("class loaded : org.ow2.jonas.lib.bootstrap.JProp");
            	    }
                } catch (IOException e3) {
            	    logger.warn("Cannot read JMX configuration file : {0} : {1}", filename, e3.getMessage());
                }
            }
        }

        if ((props != null) && (props.size() != 0)) {
            /*
             * Parse the property file CAUTION: enumeration is alphabetically
             * ordered, so proceed in two passes - first pass collects the URLs,
             * because URLs must be set before user/passwords - second pass
             * collects the user and passwords
             */
            // first pass
            Enumeration<?> en = props.propertyNames();
            while (en.hasMoreElements()) {
                String key = (String) en.nextElement();
                if (key.startsWith("jasmine.jmx.url.")) {
                    name = key.substring("jasmine.jmx.url.".length());
                    url = props.getProperty(key);
                    jmxTargets.put(name, url);
                }
            }

            // second pass
            en = props.propertyNames();
            while (en.hasMoreElements()) {
                String key = (String) en.nextElement();
                if (key.startsWith("jasmine.jmx.user.")) {
                    name = key.substring("jasmine.jmx.user.".length());
                    url = jmxTargets.get(name);
                    if (url == null) {
                        logger.warn("Cannot set user for target {0}", name);
                        continue;
                    }
                    jmxUsers.put(url, props.getProperty(key));
                } else if (key.startsWith("jasmine.jmx.password.")) {
                    name = key.substring("jasmine.jmx.password.".length());
                    url = jmxTargets.get(name);
                    if (url == null) {
                        logger.warn("Cannot set password for target {0}", name);
                        continue;
                    }
                    jmxPasswords.put(url, props.getProperty(key));
                } else if (key.startsWith("jasmine.jmx.protocol.providers.")) {
                    name = key.substring("jasmine.jmx.protocol.providers.".length());
                    url = jmxTargets.get(name);
                    if (url == null) {
                        logger.warn("Cannot set protocol providers for target {0}", name);
                        continue;
                    }
                    jmxProtocolProviderPackages.put(url, props.getProperty(key));
                } else if ("jasmine.jmx.default.target".equals(key)) {
                    if (defaultTarget == null) {
                        dftTarget = props.getProperty(key);
                    }
                }
            }
        }

        // Check that the default target belongs to the targets
        if (defaultTarget == null) {
            if (dftTarget != null) {
                if (jmxTargets.get(dftTarget) != null) {
                    defaultTarget = dftTarget;
                } else {
                    logger.warn("jasmine.jmx.default.target property specifies an invalid default target {0}. No default target set.", dftTarget);
                }
            } else {
                // use default URL as default target
                defaultTarget = "jonas";
                jmxTargets.put(defaultTarget, DEFAULT_URL);
                logger.warn("jasmine.jmx.default.target property not set. Setting default target to: {0}.", defaultTarget);
                logger.warn("Setting jmxUrl to {0}.", DEFAULT_URL);
            }
        }
    }
    /**
     * @param propertyFile File name of the properties file.
     *
     * @return Parsed version of the propertyFile file.
     *
     * @throws IOException If propertyFile cannot be read.
     */
    private static Properties getProperties(final File propertyFile) throws IOException {
        Properties properties = new Properties();

        FileInputStream in = new FileInputStream(propertyFile);
        properties.load(in);
        in.close();

        return properties;
    }


    static {
        load();
    }
}
