/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: GraphDisplay.java 3009 2009-01-21 16:19:51Z waeselynck $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.graph;

import java.util.Date;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.data.time.Millisecond;
import org.jfree.data.time.Second;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.ui.ApplicationFrame;
import org.jfree.ui.RefineryUtilities;
import org.ow2.jasmine.monitoring.mbeancmd.graph.conf.GraphConfig;

/**
 * Frame that displays the graph.
 */
public class GraphDisplay extends ApplicationFrame {
    /**
     * Generated serial version ID.
     */
    private static final long serialVersionUID = -2999008078436385027L;

    /**
     * Constructor: creates the graph frame based on the configuration.
     *
     * @param conf  Graph configuration.
     */
    public GraphDisplay(final GraphConfig conf) {
        //this(conf.getTitle(), conf.getSerieIds());
        super(conf.getTitle());
        String[] serieIds = conf.getSerieIds();
        
        // the name of the serie is the legend (not the id)
        for (int i = 0; i < serieIds.length; i++) {
            series.put(serieIds[i], new TimeSeries(conf.getLegend(serieIds[i]), Millisecond.class));
        }
        
        TimeSeriesCollection dataset = new TimeSeriesCollection();
        for (int i = 0; i < serieIds.length; i++) {
            dataset.addSeries((TimeSeries) series.get(serieIds[i]));
        }

        JFreeChart chart = ChartFactory.createTimeSeriesChart(
                                  conf.getTitle(),
                                  null, //"Time", 
                                  null, //"Value",
                                  dataset,
                                  true,
                                  true,
                                  false
                                  );
        ChartPanel chartPanel = new ChartPanel(chart);
        chartPanel.setPreferredSize(new java.awt.Dimension(500, 250));
        chartPanel.setMouseZoomable(true, false);
        setContentPane(chartPanel);
    }

    /**
     * Constructor: creates the graph frame.
     *
     * @param title  Graph title.
     * @param names  Series' names.
     */
    protected GraphDisplay(final String title, final String[] names) {
        super(title);

        for (int i = 0; i < names.length; i++) {
            series.put(names[i], new TimeSeries(names[i], Millisecond.class));
        }

        TimeSeriesCollection dataset = new TimeSeriesCollection();
        for (int i = 0; i < names.length; i++) {
            dataset.addSeries((TimeSeries) series.get(names[i]));
        }

        JFreeChart chart = ChartFactory.createTimeSeriesChart(
                title,
                null,    //"Time",
                null,    //"Value",
                dataset,
                true,
                true,
                false
        );
        ChartPanel chartPanel = new ChartPanel(chart);
        chartPanel.setPreferredSize(new java.awt.Dimension(500, 250));
        chartPanel.setMouseZoomable(true, false);
        setContentPane(chartPanel);
    }

    /**
     * Shows / updates the displayed graph.
     */
    public void showMe() {
        boolean shown;
        synchronized(this) {
            shown = isShown;
            if (!isShown) {
                isShown = true;
            }
        }

        if(!shown) {
            this.pack();
            RefineryUtilities.centerFrameOnScreen(this);
            RefineryUtilities.positionFrameRandomly(this);
            RefineryUtilities.positionFrameOnScreen(this, 1, 0);
            this.setVisible(true);
        }
    }

    /**
     * Updates the graph.
     *
     * @param id      Name of the series to update.
     * @param second  Seconds (x axis).
     * @param value   Value (y axis).
     */
    public void update(final String id, final Second second, final double value) {
        TimeSeries serie = (TimeSeries) series.get(id);

        synchronized(serie) {
            serie.addOrUpdate(second, value);
        }
        showMe();
    }

    /**
     * Updates the graph.
     *
     * @param id      Name of the series to update.
     * @param milli  Millisecond (x axis).
     * @param value   Value (y axis).
     */
    public void update(final String id, final Millisecond milli, final double value) {
        TimeSeries serie = (TimeSeries) series.get(id);

        synchronized(serie) {
            serie.addOrUpdate(milli, value);
        }
        showMe();
    }

    /**
     * Updates the graph.
     *
     * @param id      Name of the series to update.
     * @param t       Time (origin: 01/01/1970).
     * @param value   Value (y axis).
     */
    public void update(final String id, final long t, final double value) {
        //update(id, new Second(new Date(t)), value);
        update(id, new Millisecond(new Date(t)), value);
    }

    /**
     * Tests the GraphDisplay implementation.
     *
     * IMPORTANT: This method calls System.exit !
     *
     * @param args  Arguments to use when creating the graph.
     */
    public static void main(final String[] args) {
        boolean debug        = false;
        boolean relativeTime = false;
        boolean visible      = true;
        long    delay        = 2;       // delay (seconds)
        int     nSeries      = 2;       // number of series in the chart
        int     nData        = 10;      // number of generated data

        int errflg = 0;
        for (int i = 0; i < args.length; i++) {
            if ("-d".compareTo(args[i]) == 0) {
                debug = true;
            } else if ("-r".compareTo(args[i]) == 0) {
                relativeTime = !relativeTime;
            } else if ("-v".compareTo(args[i]) == 0) {
                visible = !visible;
            } else if ("-s".compareTo(args[i]) == 0) {
                if (++i >= args.length) {
                    errflg++; break;
                }
                try {
                    delay = Long.parseLong(args[i]);
                } catch (Exception e) {
                    errflg++;
                    break;
                }
                if (delay <= 0) {
                    errflg++;
                    break;
                }
            } else if ("-n".compareTo(args[i]) == 0) {
                if (++i >= args.length) {
                    errflg++;
                    break;
                }
                try {
                    nSeries = Integer.parseInt(args[i]);
                } catch (Exception e) {
                    errflg++; break;
                }
                if (nSeries < 1 || nSeries > 9) {
                    errflg++;
                    break;
                }
            } else if ("-m".compareTo(args[i]) == 0) {
                if (++i >= args.length) {
                    errflg++;
                    break;
                }
                try {
                    nData = Integer.parseInt(args[i]);
                } catch (Exception e) {
                    errflg++;
                    break;
                }
                if (nData < 2) {
                    errflg++;
                    break;
                }
            } else {
                errflg++;
            }
        }
        if (errflg > 0) {
            System.out.println("Usage: TimeSeriesDemo"
                    + " [-d] [-r] [-v] [-s delay] [-n series] [-m data]\n"
                    + "       -d        : debug\n"
                    + "       -r        : relative time\n"
                    + "       -v        : visible window at end of data only\n"
                    + "       -s delay  : update delay (min 1 second)\n"
                    + "       -n series : number of series (1 to 9)\n"
                    + "       -m data   : number of generated data\n"
            );
            System.exit(2);
        }
        if (debug) {
            System.err.print("+ TimeSeriesDemo -d");
            if (relativeTime) {
                System.err.print(" -r");
            }
            if (!visible) {
                System.err.print(" -v");
            }
            if (delay > 0) {
                System.err.print(" -s " + delay);
            }
            System.err.print(" -n " + nSeries);
            System.err.print(" -m " + nData);
            System.err.println();
        }

        String[] name = new String[nSeries];
        for (int i = 0; i < nSeries; i++) {
            name[i] = "Series" + (i + 1);
        }

        GraphDisplay demo = new GraphDisplay("Series", name);
        demo.pack();
        RefineryUtilities.centerFrameOnScreen(demo);
        RefineryUtilities.positionFrameRandomly(demo);
        RefineryUtilities.positionFrameOnScreen(demo, 1, 0);
        if (visible) {
            demo.setVisible(true);
        }

        System.out.print("seconds");
        for (int i = 0; i < nSeries; i++) {
            System.out.print("\t" + name[i]);
        }
        System.out.println();

        Random random = new Random();
        long startTime    = System.currentTimeMillis();
        long nextTime    = startTime;

        for (int m = 0; m < nData; m++) {
            double[] value = new double[nSeries];
            for (int i = 0; i < nSeries; i++) {
                value[i] = random.nextInt((i + 1) * 100);
                value[i] /= 10;
            }

            long elapsedTime = nextTime - startTime;
            long date = (relativeTime) ? elapsedTime : nextTime;
            Second second = new Second(new Date(date));

            String output = "" + (date / 1000);
            for (int i = 0; i < nSeries; i++) {
                output += "\t" + (float) value[i];
            }
            System.out.println(output);

            for (int i = 0; i < nSeries; i++) {
                demo.update(name[i], second, value[i]);
            }

            nextTime += delay * 1000;

            if (visible) {
                long sleepTime;
                while ((sleepTime = nextTime - System.currentTimeMillis()) > 10) {
                    try {
                        Thread.sleep(sleepTime);
                    } catch (InterruptedException e) {
                        break;
                    }
                }
            }
        }

        demo.setVisible(true);
        System.exit(0);
    }

    /**
     * Series' names.
     */
    private Map series = new TreeMap();

    /**
     * Is frame shown?
     */
    private boolean isShown = false;
}
