/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */

package org.ow2.jasmine.monitoring.mbeancmd.audit;

import java.util.Iterator;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.audit.util.MetricMath;

/**
 * @author waeselyf
 *
 * A metric is a collection of attributes for a given mbean
 */
public class Metric {

    public Metric(Long t, String source, ObjectName name, AttributeList attl) {
        this.timestamp = t;
        this.source = source;
        this.name = name;
        this.attributes = attl;
        this.radical = source + "@" + (name == null ? "null" : name.toString());
    }

    /**
     * Retrieve the attribute of name name
     * @param name
     * @return the attribute of name name, or null if it does not exist in the metric
     */
    public Attribute getAttribute(String name) {
        Attribute attRet = null;

        Iterator<?> it = attributes.iterator();
        while (it.hasNext()) {
            Attribute att = (Attribute) it.next();
            if (att.getName().equals(name)) {
                attRet = att;
                break;
            }
        }

        return attRet;
    }

    /**
     *
     * @return the list of attributes in the metric
     */
    public AttributeList getAttributes() {
        return attributes;
    }

    /**
     * @return the name
     */
    public ObjectName getName() {
        return name;
    }
    /**
     * @return the source
     */
    public String getSource() {
        return source;
    }
    /**
     * @return the timestamp
     */
    public long getTimestamp() {
        return timestamp;
    }

    /**
     *
     * @return radical
     */
    public String getRadical() {
        return radical;
    }

    /**
     * Computes the difference between a current and a baseline metric attribute
     * If attr is absent in current, or is not numeric, the value is null.
     * If attr is present in current but not in baseline, returns the value of attr
     *
     * @param current current metric
     * @param baseline baseline metric
     * @param attr the attribute
     * @return the difference of attr between current and baseline. A double value.
     */
    public static Metric delta(Metric current, Metric baseline, String attr) {

        Metric ret = null;

        Attribute catt = (current == null ? null : current.getAttribute(attr));
        Attribute batt = (baseline == null ? null : baseline.getAttribute(attr));

        if ((catt == null) || !MetricMath.isNum(catt.getValue())) return null;

        double c = MetricMath.toDouble(catt);
        double b = MetricMath.toDouble(batt);;
        AttributeList l = new AttributeList();
        l.add(new Attribute(attr,Double.valueOf(c - b)));

        ret = new Metric(current.getTimestamp(),current.getSource(), current.getName(),l);

        return ret;
    }


    /**
     *
     * @param radical
     * @return
     * @throws InvalidMetricRadical
     * @throws Exception
     */
    public static Metric newInstance(String radical) throws InvalidMetricRadical  {

        String[] s = radical.split("@");
        if (s.length != 2) {
            throw new InvalidMetricRadical(radical);
        }

        ObjectName on = null;

        try {
            on = ObjectName.getInstance(s[1]);
        } catch (MalformedObjectNameException e) {
            // Nothing to do
        }
        return new Metric(System.currentTimeMillis(),s[0],on,new AttributeList());
    }

    /**
     * Metric timestamp
     */
    private long timestamp = 0;

    /**
     * Source the metric has been collected from
     */
    private String source = null;

    /**
     * ObjectName of the Mbean
     */
    private ObjectName name = null;

    /**
     * Attribute list
     */
    private AttributeList attributes = null;

    /**
     * Radical of the metric
     */
    private String radical = null;
}
