/**
 * JASMINe
 * Copyright (C) 2005-2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Mbean.java 6128 2010-02-23 17:36:50Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.IOException;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.util.Date;
import java.util.Iterator;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAp;
import org.ow2.jasmine.monitoring.mbeancmd.JmxHelper;

/**
 * Command that can do any given action on any MBean.
 */
public class Mbean extends AbstractCommand {



    /**
     * Constructor: calls {@link Mbean#setOptions()}.
     */
    public Mbean() {
        this.setOptions();
    }

    /**
     * Implementation of inherited abstract method.
     *
     * Will never return if a period is set in the command line.
     *
     * @see AbstractCommand#exec()
     *
     * @return Will never return if a period is set in the command line.
     *         Otherwise, 0 if succeeded, an error code otherwise.
     */
    @Override
    public int exec(final CommandDispatcher cmdDispatcher) {
        this.cmdDispatcher = cmdDispatcher;

        // Parse the command line
        try {
            this.parseCommandLine(this.arguments);
            this.process();
            return 0;
        } catch (Exception e) {
            e.printStackTrace();
            return 1;
        }
    }

    /**
     * Implementation of inherited abstract method.
     *
     * Retrieves the command's summary.
     *
     * @see org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand#summary()
     *
     * @return Command's summary.
     */
    @Override
    public String summary() {
        return "provides access to MBeans in a J2EE server";
    }

    /**
     * Parses the command line arguments into {@link Mbean#commandLine}.
     *
     * @param args Arguments to parse.
     *
     * @throws ParseException If parsing fails.
     * @throws MalformedObjectNameException Object name given in the command
     *             line is invalid.
     */
    public void parseCommandLine(final String[] args) throws ParseException, MalformedObjectNameException {
        BasicParser bp = new BasicParser();
        this.commandLine = bp.parse(this.options, args);
        this.setObjectName();
        this.setMethod();
        if (this.commandLine.hasOption("p")) {
            this.setPeriod();
        }

        String[] targets = JmxHelper.getJmxTargets(this.commandLine.getOptionValues("target"));
        if ((targets != null) && (targets.length > 0)) {
            this.jmxap = new JmxAp(JmxHelper.getJmxUrl(targets[0]),this.cmdDispatcher);
        } else {
            this.jmxap = new JmxAp(this.cmdDispatcher);
        }
    }

    /**
     * Sets the object name based on the "name" argument in the command line.
     *
     * @throws MalformedObjectNameException Object name given in the command
     *             line is invalid.
     */
    private void setObjectName() throws MalformedObjectNameException {
        String mbean  = commandLine.getOptionValue("name");

        if(mbean.contains("\"")) {

            StringBuffer quotedMbean = new StringBuffer();
            StringTokenizer st = new StringTokenizer(mbean, "\"");
            quotedMbean.append(st.nextToken());
            while(st.hasMoreTokens()) {

                quotedMbean.append(ObjectName.quote(st.nextToken()));

                if(st.hasMoreTokens()) {
                    quotedMbean.append(st.nextToken());
                }
            }
            on = new ObjectName(quotedMbean.toString());
        } else {
            on = new ObjectName(mbean);
        }
    }

    /**
     * Sets the GET method if the command line has the "get" option.
     */
    private void setMethod() {
        Option opt = this.options.getOption("-get");
        this.method = this.options.getOptionGroup(opt).getSelected();
    }

    /**
     * Sets the period if the command line has the "p" option.
     *
     * @throws NumberFormatException Number after the "p" option is invalid.
     */
    private void setPeriod() throws NumberFormatException {
        Option opt = this.options.getOption("-p");
        this.period = Long.parseLong(opt.getValue());
    }

    /**
     * Queries the {@link Mbean#method} method of the {@link Mbean#on} object.
     *
     * @see Mbean#setMethod()
     * @see Mbean#setObjectName()
     *
     * @throws Exception If anything fails.
     */
    private void mbeanQuery() throws Exception {
        try {
            MBeanServerConnection cnx = this.jmxap.getMBeanServerConnection();

            Set s = cnx.queryNames(this.on, null);
            Iterator it = s.iterator();
            while (it.hasNext()) {
                System.out.println(it.next());
            }
            this.jmxap.releaseMBeanServerConnection();
        } finally {
            this.jmxap.releaseMBeanServerConnection();
        }
    }

    /**
     * Retrieves attributes of a mbean.
     *
     * @throws Exception If anything fails.
     */
    private void mbeanGet() throws Exception {
        try {
            String[] atts = this.commandLine.getOptionValues("get");
            MBeanServerConnection cnx = this.jmxap.getMBeanServerConnection();
            if (atts == null) {
                MBeanAttributeInfo[] info = cnx.getMBeanInfo(this.on).getAttributes();
                atts = new String[info.length];
                for (int i = 0; i < atts.length; i++) {
                    atts[i] = info[i].getName();
                }
            }
            AttributeList attl = cnx.getAttributes(this.on, atts);
            this.jmxap.releaseMBeanServerConnection();
            this.printMBeanGet(this.on, attl);
        } finally {
            this.jmxap.releaseMBeanServerConnection();
        }
    }

    /**
     * Retrieves attributes of one or more mbeans.
     *
     * @throws Exception If anything fails.
     */
    @SuppressWarnings("unchecked")
    private void mbeanMGet() throws Exception {
        try {
            String[] attsRequested = this.commandLine.getOptionValues("mget");
            String[] atts = attsRequested;
            MBeanServerConnection cnx = this.jmxap.getMBeanServerConnection();
            Set<ObjectName> s = cnx.queryNames(this.on, null);
            Iterator<ObjectName> it = s.iterator();
            while (it.hasNext()) {
                ObjectName onGot = it.next();
                try {
                    if (attsRequested == null) {
                        MBeanAttributeInfo[] info;
                        info = cnx.getMBeanInfo(onGot).getAttributes();
                        atts = new String[info.length];
                        for (int i = 0; i < atts.length; i++) {
                            atts[i] = info[i].getName();
                        }
                    }

                    AttributeList attl = cnx.getAttributes(onGot, atts);
                    this.printMBeanGet(onGot, attl);
                } catch (Exception e) {
                    System.err.println("Error on mbean " + onGot);
                    e.printStackTrace(System.err);
                }
            }
            this.jmxap.releaseMBeanServerConnection();
        } finally {
            this.jmxap.releaseMBeanServerConnection();
        }
    }

    /**
     * Prints the list of attributes for an MBean.
     *
     * @param ongot Object name.
     * @param attl List of attributes.
     */
    @SuppressWarnings("unchecked")
    private void printMBeanGet(final ObjectName ongot, final AttributeList attl) {
        boolean verbose = this.commandLine.hasOption("v");
        if (verbose) {
            System.out.println("#MBEAN : " + ongot);
        }
        Iterator<?> it = attl.iterator();
        while (it.hasNext()) {
            Attribute att = (Attribute) it.next();
            System.out.print((verbose ? att.getName() + "=" : ""));
            if ((att.getValue() != null) && att.getValue().getClass().isArray()) {
                Object obj = att.getValue();
                int length = Array.getLength(obj);
                for (int i = 0; i < length; i++) {
                    System.out.print((i == 0 ? "" : ",") + Array.get(obj, i));
                }
                System.out.println();
            } else {
                System.out.println(att.getValue());
            }
        }
    }

    /**
     * Periodically polls attributes of one or more mbeans.
     */
    @SuppressWarnings("unchecked")
    private void mbeanPoll() {
        String[] atts = this.getAttToPoll();
        this.printMBeanPollHeader(atts);

        // Get a JMX connection
        MBeanServerConnection cnx = this.jmxap.getMBeanServerConnection();

        // Poll
        while (cnx != null) {
            try {
                // dummy step for the first loop, but useful to get the
                // connection back for the others
                cnx = this.jmxap.getMBeanServerConnection();

                Set<ObjectName> s = cnx.queryNames(this.on, null);
                Iterator<ObjectName> it = s.iterator();
                while (it.hasNext()) {
                    long t = System.currentTimeMillis();
                    ObjectName onGot = it.next();
                    AttributeList attl;
                    try {
                        attl = cnx.getAttributes(onGot, atts);
                        this.printMBeanPoll(t, attl);
                    } catch (Exception e) {
                        System.err.println("Error on mbean " + onGot + " : " + e);
                    }
                }
                // release the connection (not close)
                this.jmxap.releaseMBeanServerConnection();

            } catch (IOException e1) {
                e1.printStackTrace(System.err);
                break;
            }

            // Sleep for a while
            try {
                Thread.sleep(this.period);
            } catch (InterruptedException e) {
                this.jmxap.releaseMBeanServerConnection();
                break;
            } finally {
                this.jmxap.releaseMBeanServerConnection();
            }
        }
    }

    /**
     * @return List of attributes to poll (read from command line).
     */
    @SuppressWarnings("unchecked")
    private String[] getAttToPoll() {
        String[] attsRequested = this.commandLine.getOptionValues("poll");
        String[] atts = attsRequested;

        try {
            MBeanServerConnection cnx = this.jmxap.getMBeanServerConnection();

            ObjectName onGot = null;
            Set<ObjectName> s = cnx.queryNames(this.on, null);
            Iterator<ObjectName> it = s.iterator();
            if (it.hasNext()) {
                onGot = it.next();
            }

            if (attsRequested == null) {
                MBeanAttributeInfo[] info;
                info = cnx.getMBeanInfo(onGot).getAttributes();
                atts = new String[info.length];
                for (int i = 0; i < atts.length; i++) {
                    atts[i] = info[i].getName();
                }
            }
        } catch (MalformedURLException e) {
            e.printStackTrace(System.err);
        } catch (IOException e) {
            e.printStackTrace(System.err);
        } catch (InstanceNotFoundException e) {
            e.printStackTrace(System.err);
        } catch (IntrospectionException e) {
            e.printStackTrace(System.err);
        } catch (ReflectionException e) {
            e.printStackTrace(System.err);
        } finally {
            this.jmxap.releaseMBeanServerConnection();
        }
        return atts;
    }

    /**
     * Prints the output header: date;time + atts.
     *
     * @param atts List of attributes.
     */
    private void printMBeanPollHeader(final String[] atts) {
        System.out.print("date;time;");
        for (int i = 0; i < atts.length; i++) {
            System.out.print((i == 0 ? "" : ";") + atts[i]);
        }
        System.out.println();
    }

    /**
     * Prints the output data: date;time + atts.
     *
     * @param t Time.
     * @param attl List of attributes.
     */
    private void printMBeanPoll(final long t, final AttributeList attl) {
        Date d = new Date(t);
        System.out.print(SIMPLEDATEFORMAT.format(d));
        System.out.print(";");
        System.out.print(t);
        System.out.print(";");
        Iterator<Object> it = attl.iterator();
        while (it.hasNext()) {
            Attribute att = (Attribute) it.next();
            if ((att.getValue() != null) && att.getValue().getClass().isArray()) {
                Object obj = att.getValue();
                int length = Array.getLength(obj);
                for (int i = 0; i < length; i++) {
                    System.out.print((i == 0 ? "" : ",") + Array.get(obj, i));
                }
            } else {
                System.out.print(att.getValue());
            }
            System.out.print(";");
        }
        System.out.println();
    }

    /**
     * Prints information about MBeans.
     */
    private void mbeanInfo() {
        boolean verbose = this.commandLine.hasOption("v");

        try {
            MBeanServerConnection cnx = this.jmxap.getMBeanServerConnection();
            MBeanAttributeInfo[] info = cnx.getMBeanInfo(this.on).getAttributes();
            if (verbose) {
                System.out.println("#MBEAN : " + this.on);
            }
            for (int i = 0; i < info.length; i++) {
                if (verbose) {
                    System.out.println("attribute" + " " + info[i].getName() + " " + (info[i].isReadable() ? "R" : "")
                            + (info[i].isWritable() ? "W" : "") + " " + info[i].getType() + " \"" + info[i].getDescription()
                            + "\"");
                } else {
                    System.out.println("attribute " + info[i].getName());
                }
            }
            MBeanOperationInfo[] opinfo = cnx.getMBeanInfo(this.on).getOperations();
            this.jmxap.releaseMBeanServerConnection();
            for (int i = 0; i < opinfo.length; i++) {
                if (verbose) {
                    System.out.print("operation" + " " + opinfo[i].getReturnType() + " " + opinfo[i].getName() + "(");
                    MBeanParameterInfo[] sig = opinfo[i].getSignature();
                    for (int j = 0; j < sig.length; j++) {
                        System.out.print(sig[j].getType() + " " + sig[j].getName() + " \"" + sig[j].getDescription() + "\""
                                + (j == (sig.length - 1) ? "" : ","));
                    }
                    System.out.println(")");
                } else {
                    System.out.println("operation " + opinfo[i].getName());
                }
            }
        } catch (MalformedURLException e) {
            e.printStackTrace(System.err);
        } catch (InstanceNotFoundException e) {
            e.printStackTrace(System.err);
        } catch (IntrospectionException e) {
            e.printStackTrace(System.err);
        } catch (ReflectionException e) {
            e.printStackTrace(System.err);
        } catch (IOException e) {
            e.printStackTrace(System.err);
        } finally {
            this.jmxap.releaseMBeanServerConnection();
        }
    }

    /**
     * Grabs the options for setting MBean attributes.
     *
     * @throws Exception If anything fails.
     */
    @SuppressWarnings("unchecked")
    private void mbeanSet() throws Exception {
        String[] attArgs = this.commandLine.getOptionValues("set");

        if (attArgs.length < 3) {
            throw new InvalidAttributeValueException("Wrong attribute specification");
        }

        // Build attribute
        String attName = attArgs[0];
        String attType = attArgs[1];
        Object attValue = null;

        Class<?> vc = Class.forName(attType);
        Class<?>[] sig = {Class.forName("java.lang.String")};

        if (vc.isArray()) {
            Constructor c = vc.getComponentType().getConstructor(sig);
            // values are in the range [2 ; length [
            // because the first 2 are the name and type
            int sz = attArgs.length - 2;

            Object arrayOfValues = Array.newInstance(vc.getComponentType(), sz);
            for (int i = 0; i < sz; i++) {
                String value = attArgs[2 + i];
                Object[] params = {value};
                Array.set(arrayOfValues, i, c.newInstance(params));
            }
            attValue = arrayOfValues;
        } else {
            Constructor c = vc.getConstructor(sig);
            String value = attArgs[2];
            Object[] params = {value};
            attValue = c.newInstance(params);
        }

        try {
            // Invoke SET operation
            MBeanServerConnection cnx = this.jmxap.getMBeanServerConnection();
            Attribute att = new Attribute(attName, attValue);
            cnx.setAttribute(this.on, att);
        } finally {
            this.jmxap.releaseMBeanServerConnection();
        }
    }

    /**
     * Parses a method signature to grab arguments. - Parameters are conveyed by
     * a list of type-value(s) - Parameter separator is ","
     *
     * Example: java.lang.string x , [Ljava.lang.String a b c
     *
     * @param args Signature to parse.
     * @return Parsed method signature. null if no arguments present.
     */
    private String[] parseMethodSig(final String[] args) {
        int nargs = 0;

        // compute the number of args
        // skip the method name (first element)
        boolean newArg = true;
        for (int i = 1; i < args.length; i++) {
            if (newArg) {
                nargs++;
                newArg = false;
            }
            if (",".equals(args[i])) {
                newArg = true;
            }
        }
        if (nargs == 0) {
            return null;
        }

        String[] opSig = new String[nargs];

        // look for the parameter signatures
        boolean isType = true;
        int curArg = 0;
        for (int i = 1; i < args.length; i++) {
            if (isType) {
                opSig[curArg++] = args[i];
                isType = false;
            }
            if (",".equals(args[i])) {
                isType = true;
            }
        }

        return opSig;
    }

    /**
     * Parses parameters of a given method.
     *
     * @param args Method signature to parse.
     * @param opSig
     * @see {@link Mbean#parseMethodSig(String[])}
     *
     * @return Parsed parameters.
     *
     * @throws ClassNotFoundException Class not found.
     * @throws NoSuchMethodException Method not found.
     * @throws InstantiationException Cannot instantiate argument's class.
     * @throws IllegalAccessException Cannot access class.
     * @throws InvocationTargetException Invocation problem.
     */
    @SuppressWarnings("unchecked")
    private Object[] parseMethodParams(final String[] args, final String[] opSig) throws ClassNotFoundException,
            NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException {
        int nargs = opSig.length;
        Object[] opParams = new Object[nargs];
        Object[] sParams;

        boolean newArg = true;
        Vector<String> v = null;
        int curArg = 0;

        // The method name is the first element of args : skip it !
        for (int i = 1; i < args.length; i++) {
            if (newArg) {
                v = new Vector<String>();
                newArg = false;
            } else if (",".equals(args[i])) {
                // Here, we have got all the values for the current parameter,
                // as we have reached the comma separating parameters
                sParams = v.toArray();
                Class<?>[] sig = {Class.forName("java.lang.String")};
                Class<?> vc = Class.forName(opSig[curArg]);

                if (vc.isArray()) {
                    // Here we have to build an array of values
                    // look for a constructor that takes a String argument
                    // Instanciate every elements of the array
                    Constructor c = vc.getComponentType().getConstructor(sig);
                    Object arrayOfValues = Array.newInstance(vc.getComponentType(), sParams.length);
                    for (int j = 0; j < sParams.length; j++) {
                        Object[] params = {sParams[j]};
                        Array.set(arrayOfValues, j, c.newInstance(params));
                    }

                    opParams[curArg] = arrayOfValues;
                } else {
                    // Here we have to build a single value.
                    // sParams should be of length 1. extra values are ignored
                    // Look for a constructor that takes a String argument
                    Constructor c = vc.getConstructor(sig);
                    Object[] params = {sParams[0]};
                    opParams[curArg] = c.newInstance(params);
                }

                curArg++;
                newArg = true;
            } else {
                // collate parameter values
                v.add(args[i]);
            }
        }

        return opParams;
    }

    /**
     * Invokes the operation given as command line argument.
     *
     * @throws Exception If anything fails.
     */
    private void mbeanInvoke() throws Exception {
        String[] args = this.commandLine.getOptionValues("invoke");
        if (args.length == 0) {
            throw new javax.management.JMException("Missing method name");
        }

        String op = args[0];
        String[] opSig = this.parseMethodSig(args);
        Object[] opParams = null;
        if (opSig != null) {
            opParams = this.parseMethodParams(args, opSig);
        }
        // Invoke operation
        Object result;
        try {
            result = null;
            MBeanServerConnection cnx = this.jmxap.getMBeanServerConnection();
            result = cnx.invoke(this.on, op, opParams, opSig);
        } finally {
            this.jmxap.releaseMBeanServerConnection();
        }

        // print result
        boolean verbose = this.commandLine.hasOption("v");
        if (verbose) {
            System.out.println("#MBEAN : " + this.on);
        }

        if (result != null) {
            if (result.getClass().isArray()) {
                Object[] to = (Object[]) result;
                for (int i = 0; i < to.length; i++) {
                    System.out.println(to[i]);
                }
            } else {
                System.out.println(result);
            }
        }
    }

    /**
     * Unregisters the MBean given as a command line argument.
     */
    private void mbeanUnregister() {
        MBeanServerConnection cnx;
        try {
            cnx = this.jmxap.getMBeanServerConnection();
            cnx.unregisterMBean(this.on);
        } catch (Exception e) {
            e.printStackTrace(System.err);
        } finally {
            this.jmxap.releaseMBeanServerConnection();
        }
    }

    /**
     * Sets all options that are parseable from the command line.
     *
     * @see Mbean#options
     */
    private void setOptions() {

        this.options = new Options();
        this.options.addOption("v", "verbose", false, "Verbose mode");

        // ObjectName
        Option name = new Option("name", "objectName", true, "An ObjectName passed to every method");
        name.setRequired(true);
        name.setArgName("mbean name");
        name.setArgs(1);
        this.options.addOption(name);

        // Group of Exclusive methods
        OptionGroup methods = new OptionGroup();
        methods.setRequired(true);

        // queryNames method
        Option query = new Option("query", "queryNames", false, "Query names to the Mbean server");
        methods.addOption(query);

        // info method
        Option info = new Option("info", "mbeanInfo", false, "Get the mbean (meta-)information");
        methods.addOption(info);

        // GET method
        Option get = new Option("get", "get", true, "Get attributes from a MBean");
        get.setOptionalArg(true);
        get.setArgs(Option.UNLIMITED_VALUES);
        get.setArgName("attributes");
        methods.addOption(get);

        // MGET method
        Option mget = new Option("mget", "mget", true, "Get attributes from multiple MBeans");
        mget.setOptionalArg(true);
        mget.setArgs(Option.UNLIMITED_VALUES);
        mget.setArgName("attributes");
        methods.addOption(mget);

        // SET method
        Option set = new Option("set", "set", true, "Set an attribute of a MBean");
        set.setOptionalArg(true);
        set.setArgs(Option.UNLIMITED_VALUES);
        set.setArgName("attributes");
        methods.addOption(set);

        // INVOKE method
        Option invoke = new Option("invoke", "invoke", true, "Invoke an operation of a MBean");
        invoke.setOptionalArg(true);
        invoke.setArgs(Option.UNLIMITED_VALUES);
        invoke.setArgName("op params");
        methods.addOption(invoke);

        // unregister method
        Option unregister = new Option("unregister", "unregister", false, "Unregister a mbean");
        methods.addOption(unregister);

        // POLL method
        Option poll = new Option("poll", "poll", true, "Poll attributes from multiple MBeans periodically");
        poll.setOptionalArg(true);
        poll.setArgs(Option.UNLIMITED_VALUES);
        poll.setArgName("attributes");
        methods.addOption(poll);

        // Period
        Option period = new Option("p", "period", true, "Polling period");
        period.setArgName("period");
        period.setArgs(1);
        this.options.addOption(period);

        this.options.addOptionGroup(methods);

        // Optional attribute target
        Option target = new Option("target", "target", true, "instance to access to");
        target.setRequired(false);
        target.setOptionalArg(true);
        target.setArgs(Option.UNLIMITED_VALUES);
        target.setArgName("instances");
        this.options.addOption(target);

    }

    /**
     * @return Returns stringified {@link Mbean#on}
     */
    @Override
    public String toString() {
        return this.on.toString();
    }

    /**
     * Processes the request.
     *
     * @throws Exception If anything fails.
     */
    public void process() throws Exception {
        if ("query".equals(this.method)) {
            this.mbeanQuery();
        } else if ("get".equals(this.method)) {
            this.mbeanGet();
        } else if ("mget".equals(this.method)) {
            this.mbeanMGet();
        } else if ("info".equals(this.method)) {
            this.mbeanInfo();
        } else if ("set".equals(this.method)) {
            this.mbeanSet();
        } else if ("invoke".equals(this.method)) {
            this.mbeanInvoke();
        } else if ("poll".equals(this.method)) {
            this.mbeanPoll();
        } else if ("u".equals(this.method)) {
            this.mbeanUnregister();
        }
    }

    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * The object name based on the "name" argument in the command line.
     */
    private ObjectName on = null;

    /**
     * Method to invoke.
     */
    private String method;

    /**
     * Polling period in milliseconds.
     */
    private long period = 10000;

    /**
     * JMX Access Point.
     */
    protected JmxAp jmxap = null;

    /**
     * Command dispatcher instance.
     */
    private CommandDispatcher cmdDispatcher = null;

}
