/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DSData.java 6090 2010-02-22 14:21:46Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.util.HashMap;
import java.util.Map;

import javax.management.Attribute;

/**
 * Data retrieved from datasources.
 */
public class DSData extends SampleData {
    /**
     * Implementation of inherited method.
     * DataSource sampling.
     * Compute <code>servedOpenRate</code> based on <code>servedOpen</code>
     * Compute <code>meanWaitTime</code> based on <code>waiterCount</code> and <code>waitingTime</code>
     * Compute <code></code> based on <code>servedOpen</code>
     * @see SampleData#compute(SampleData)
     */
    public void compute(final SampleData needs_cast) {
        DSData prev = (DSData) needs_cast;
    	if (prev != null) {
    		double sampleTime = this.computeSamplePeriod(prev);
    		// compute servedOpenRate
    		int deltaServedOpen = this.getAttributeAsInt("servedOpen") - prev.getAttributeAsInt("servedOpen");
    		this.setAttribute(new Attribute("servedOpenRate", new Double(deltaServedOpen / sampleTime)));
            // compute meanWaitTime
    		int deltaWaiterCount = this.getAttributeAsInt("waiterCount") - prev.getAttributeAsInt("waiterCount");
    		long deltaWaitingTime = this.getAttributeAsLong("waitingTime") - prev.getAttributeAsLong("waitingTime");
    		double meanWaitTime = (deltaWaiterCount > 0 ? 1.0 * deltaWaitingTime / deltaWaiterCount : 0.0);
    		this.setAttribute(new Attribute("meanWaitTime", new Double(meanWaitTime)));
    	} else {
    		this.setAttribute(new Attribute("servedOpenRate", new Double(0.0)));
    		this.setAttribute(new Attribute("meanWaitTime", new Double(0.0)));
    	}

    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintHeader()
     */
    protected String getInnerPrintHeader() {
        return header.replaceAll(";", getSeparator());
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintData()
     */
    protected String getInnerPrintData() {
        return  decimalFormat.format(this.getAttributeAsDouble("servedOpenRate")) + getSeparator()
              + this.getAttributeAsInt("servedOpen") + getSeparator()
              + this.getAttributeAsInt("jdbcMaxConnPool") + getSeparator()
              + this.getAttributeAsInt("currentOpened") + getSeparator()
              + this.getAttributeAsInt("busyMax") + getSeparator()
              + this.getAttributeAsInt("waitersHighRecent") + getSeparator()
              + this.getAttributeAsLong("waitingHighRecent") + getSeparator()
              + decimalFormat.format(this.getAttributeAsDouble("meanWaitTime")) + getSeparator()
              + this.getAttributeAsInt("rejectedOpen") + getSeparator()
              + this.getAttributeAsInt("connectionLeaks");
    }

    @Override
    public Map<String, ?> getValueOfAttributes() {
        HashMap<String, Object> valueOfAttributes = new HashMap<String, Object>();
        valueOfAttributes.put("servedRate", getAttributeAsDouble("servedOpenRate"));
        valueOfAttributes.put("servedOpen", getAttributeAsInt("servedOpen"));
        valueOfAttributes.put("jdbcMaxConnPool", getAttributeAsInt("jdbcMaxConnPool"));
        valueOfAttributes.put("currentOpen", getAttributeAsInt("currentOpened"));
        valueOfAttributes.put("busyMax", getAttributeAsInt("busyMax"));
        valueOfAttributes.put("waitersHighRecent", getAttributeAsInt("waitersHighRecent"));
        valueOfAttributes.put("waitingHighRecent", getAttributeAsLong("waitingHighRecent"));
        valueOfAttributes.put("meanWaitTime", getAttributeAsDouble("meanWaitTime"));
        valueOfAttributes.put("rejectedOpen", getAttributeAsInt("rejectedOpen"));
        valueOfAttributes.put("connectionLeaks", getAttributeAsInt("connectionLeaks"));
        return valueOfAttributes;
    }

    /**
     * Default header.
     */
    private static final String header = "servedRate;servedOpen;jdbcMaxConnPool;currentOpen;busyMax;"
                                       + "waitersHighRecent;waitingHighRecent;meanWaitTime;rejectedOpen;connectionLeaks";
}
