/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DSSampler.java 6157 2010-03-16 14:39:48Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.util.Iterator;

import javax.management.Attribute;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;

/**
 * Sampler for datasources.
 */
public class DSSampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public DSSampler(final SamplerContext context) {
        super(context);
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOnPattern() {
        return "*:j2eeType=JDBCDataSource,*";
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new XSampleData();
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#poll(SampleData)
     */
    protected SampleData poll(final SampleData data) {
        pollDataSources((XSampleData) data);
        data.setValid(true);
        return data;
    }

    /**
     * Polls datasource providers for data, will call
     * {@link DSSampler#pollDataSource(ObjectName, DSData)}
     * for each object name.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollDataSources(final XSampleData data) {
        data.setServerInfo(getName(), getJmxUrl(), getServer(), getDomain());

        ObjectName[] oNames = (ObjectName[]) context.getOnames().toArray(new ObjectName[context.getOnames().size()]);

        for (int i = 0; i < oNames.length; i++) {
            data.setObjectName(oNames[i]);

            DSData dsdata = new DSData();
            dsdata.setObjectName(oNames[i]);
            dsdata.setSampleTime(System.currentTimeMillis());
            pollDataSource(oNames[i], dsdata);

            if (dsdata.isValid()) {
                data.put(oNames[i], dsdata);
            }
        }
    }

    /**
     * Polls one datasource provider for data.
     *
     * @param on    Name of the provider.
     * @param data  ServerData to add polled information to.
     */
    private void pollDataSource(final ObjectName on, final DSData data) {
        try {
            Iterator it = getMBeanServerConnection().getAttributes(on, attIds).iterator();
            while (it.hasNext()) {
                data.setAttribute((Attribute) it.next());
            }
            data.setServerInfo(getName(), getJmxUrl(), getServer(), getDomain());
            data.setValid(true);
            releaseMbeanServerConnection();
        } catch (Exception e) {
            e.printStackTrace(System.err);
        }
    }

    /*
     * List of all available mbean statistics :
     *
     * - Managed connection statistics
     *      currentOpened
     *      currentBusy
     *      currentInTx
     *      busyMin
     *      busyMax
     *      jdbcMinConnPool
     *      jdbcMaxConnPool
     *
     *      jdbcConnMaxAge
     *      openedCount
     *      connectionFailures
     *
     *      jdbcMaxOpenTime
     *      connectionLeaks
     *
     * - Connection caller statistics
     *      jdbcMaxWaiters
     *      currentWaiters
     *      waitersHighRecent
     *      waitersHigh
     *      waiterCount
     *
     *      jdbcMaxWaitTime
     *      waitingHigh
     *      waitingHighRecent
     *      waitingTime
     *
     *      servedOpen
     *      rejectedOpen
     *      rejectedFull
     *      rejectedTimeout
     *      rejectedOther
     *
     *      jdbcSamplingPeriod
     *
     * - Based on these, we'll compute:
     *      servedOpenRate (double)
     *      meanWaittime (long)
     */
    private static final String[] attIds = {
             "currentOpened",
             "currentBusy",
             "currentInTx",
             "busyMin",
             "busyMax",
             "jdbcMinConnPool",
             "jdbcMaxConnPool",

             "jdbcConnMaxAge",
             "openedCount",
             "connectionFailures",

             "jdbcMaxOpenTime",
             "connectionLeaks",

             "jdbcMaxWaiters",
             "currentWaiters",
             "waitersHighRecent",
             "waitersHigh",
             "waiterCount",

             "jdbcMaxWaitTime",
             "waitingHigh",
             "waitingHighRecent",
             "waitingTime",

             "servedOpen",
             "rejectedOpen",
             "rejectedFull",
             "rejectedTimeout",
             "rejectedOther",

             "jdbcSamplingPeriod"
    };
}
