/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: EntitySampler.java 6157 2010-03-16 14:39:48Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.IOException;
import java.util.Iterator;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;

/**
 * Sampler for entity beans.
 */
public class EntitySampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public EntitySampler(final SamplerContext context) {
        super(context);
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOnPattern() {
        return "*:j2eeType=EntityBean,*";
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new XSampleData();
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#poll(SampleData)
     * @see EntitySampler#pollEntities(XSampleData)
     */
    protected SampleData poll(final SampleData data) {
        pollEntities((XSampleData) data);
        data.setValid(true);
        return data;
    }

    /**
     * Polls entity providers for data, will call
     * {@link EntitySampler#pollEntity(ObjectName, EntityData)}
     * for each object name.
     *
     * @param data  XSampleData to add polled information to.
     */
    private void pollEntities(final XSampleData data) {

        ObjectName[] oNames = (ObjectName[]) context.getOnames().toArray(new ObjectName[context.getOnames().size()]);

        for (int i = 0; i < oNames.length ; i++) {
            EntityData entdata = new EntityData();
            entdata.setObjectName(oNames[i]);
            entdata.setSampleTime(System.currentTimeMillis());
            pollEntity(oNames[i], entdata);

            if (entdata.isValid()) {
                data.put(oNames[i], entdata);
            }
        }
    }

    /**
     * Polls one entity provider for data.
     *
     * @param on    Name of the provider.
     * @param data  EntityData to add polled information to.
     */
    private void pollEntity(final ObjectName on, final EntityData data) {
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        Attribute att;
        try {
            cnx = getMBeanServerConnection();
            al = cnx.getAttributes(on,attIds);
            releaseMbeanServerConnection();
            cnx = null;
            Iterator it = al.iterator();
            while (it.hasNext()) {
                att = (Attribute) it.next();
                data.setAttribute(att);
            }
            data.setServerInfo(getName(), getJmxUrl(), getServer(), getDomain());
            data.setValid(true);
        } catch (InstanceNotFoundException e) {
            e.printStackTrace(System.err);
        } catch (ReflectionException e) {
            e.printStackTrace(System.err);
        } catch (IOException e) {
            e.printStackTrace(System.err);
        }
    }


    /**
     * List of polled attributes :
     *
     *      - PoolSize : Current nb of free instances.
     *      - CacheSize : Total nb of instances in memory.
     *
     *      - Entity Counters :
     *          0: Used in TX  : Current nb of instances used inside transactions.
     *          1: Used outside TX : Current nb of instances used outside transaction.
     *          2: Unused but ready : Current nb of unused, ready instances.
     *          3: Marked removed : Current nb of instances marked removed.
     *          4: PK number : Total nb of PK in memory.
     *          5: Passivated : Current nb of passivated instances.
     */
    private static final String[] attIds = {
             "name",
             "cacheSize",
             "poolSize",
             "entityCounters",
             "maxCacheSize",
             "minPoolSize"
    };
}
