/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JCACFSampler.java 6157 2010-03-16 14:39:48Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.IOException;
import java.util.Iterator;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;

/**
 * Sampler for JCA Connection Factories.
 */
public class JCACFSampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public JCACFSampler(final SamplerContext context) {
        super(context);
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOnPattern() {
        return "*:j2eeType=JCAConnectionFactory,*";
    }


    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new XSampleData();
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#poll(SampleData)
     * @see EntitySampler#pollEntities(XSampleData)
     */
    protected SampleData poll(final SampleData data) {
        pollJCACFs((XSampleData) data);
        data.setValid(true);
        return data;
    }

    /**
     * Polls JCA Connection Factory providers for data, will call
     * {@link JCACFSampler#pollJCACF(ObjectName, JCACFData)}
     * for each object name.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollJCACFs(final XSampleData data) {

        ObjectName[] oNames = (ObjectName[]) context.getOnames().toArray(new ObjectName[context.getOnames().size()]);

        for (int i = 0; i < oNames.length; i++) {
            JCACFData cfdata = new JCACFData();
            cfdata.setObjectName(oNames[i]);
            cfdata.setSampleTime(System.currentTimeMillis());
            pollJCACF(oNames[i], cfdata);

            if (cfdata.isValid()) {
                data.put(oNames[i], cfdata);
            }
        }
    }

    /**
     * Polls one JCA Connection Factory provider for data.
     *
     * @param on    Name of the provider.
     * @param data  ServerData to add polled information to.
     */
    private void pollJCACF(final ObjectName on, final JCACFData data) {
        AttributeList al = null;
        MBeanServerConnection cnx = null;

            Attribute att;
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(on, attIds);
                releaseMbeanServerConnection();
                cnx = null;
                Iterator it = al.iterator();
                while (it.hasNext()) {
                    att = (Attribute) it.next();
                    data.setAttribute(att);
                }
                data.setServerInfo(getName(), getJmxUrl(), getServer(), getDomain());
                data.setValid(true);
            } catch (InstanceNotFoundException e) {
                e.printStackTrace(System.err);
            } catch (ReflectionException e) {
                e.printStackTrace(System.err);
            } catch (IOException e) {
                e.printStackTrace(System.err);
            }
    }

    /**
     * List of polled attributes.
     */
    private static final String[] attIds = {"currentOpened", "currentBusy", "maxSize", "currentInTx",
                                            "busyMin", "busyMax", "openedCount", "connectionFailures",
                                            "connectionLeaks", "currentWaiters", "waitersHighRecent",
                                            "waitersHigh", "waiterCount", "waitingHigh", "waitingHighRecent",
                                            "waitingTime", "servedOpen", "rejectedOpen", "rejectedFull",
                                            "rejectedTimeout", "rejectedOther"};
}
