/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: WebSampler.java 6157 2010-03-16 14:39:48Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.util.Iterator;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;

/**
 * Sampler that retrieves statistics from the HTTP server.
 */
public class WebSampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public WebSampler(final SamplerContext context) {
        super(context);
    }

    /**
     * Implementation of inherited abstract method. Returns null since
     * ServerSampler doesn't have one default object name.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOnPattern() {
        return null;
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new WebData();
    }

    /**
     * Implementation of inherited abstract method, will call all pollers of
     * this class.
     *
     * @see Sampler#poll(SampleData)
     *
     * @see WebSampler#pollSessions(WebData)
     * @see WebSampler#pollGlobalRequestProcessor(WebData)
     * @see WebSampler#pollThreadPool(WebData)
     */
    protected SampleData poll(final SampleData data) {
        // All poll must be OK
        boolean status = true;

        pollSessions((WebData) data);
        status = status && data.isValid();

        pollGlobalRequestProcessor((WebData) data);
        status = status && data.isValid();

        pollThreadPool((WebData) data);
        status = status && data.isValid();

        data.setServerInfo(getName(), getJmxUrl(), getServer(), getDomain());
        data.setValid(status);
        data.setObjectName(getOnWebStats());

        return data;
    }

    /**
     * Polls session provider for data.
     *
     * @param data
     *            WebData to add polled information to.
     */
    private void pollSessions(final WebData data) {
        long nbSess = 0;
        long sessionCnt = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = { "activeSessions", "sessionCounter" };

        refreshSessionList();
        if (onSessions != null) {
            for (int i = 0; i < onSessions.length; i++) {
                if (onSessions[i] != null) {
                    try {
                        cnx = getMBeanServerConnection();
                        al = cnx.getAttributes(onSessions[i], attIds);
                        releaseMbeanServerConnection();
                        cnx = null;
                        Iterator it = al.iterator();
                        Attribute att = (Attribute) it.next();
                        nbSess += ((Integer) att.getValue()).intValue();
                        att = (Attribute) it.next();
                        sessionCnt += ((Integer) att.getValue()).intValue();
                        data.setValid(true);
                    } catch (Exception e) {
                        refreshableSessions = true;
                        e.printStackTrace(System.err);
                        e.getCause().printStackTrace(System.err);
                    }
                }
            }
        }
        data.setSessionGauge(nbSess);
        data.setSessionCount(sessionCnt);
    }

    /**
     * Refresh session list. The actual policy is to refresh the list until it
     * is not null;
     */
    private void refreshSessionList() {

        if (onSessions == null) {
            refreshableSessions = true;
        }
        if (refreshableSessions) {
            ObjectName[] ons = null;
            try {
                ons = queryON("*:type=Manager,*");
            } catch (Exception e) {
                // TODO log on debug
            }
            onSessions = ons;
            if (ons != null) {
                refreshableSessions = false;
            }
        }
    }

    /**
     * Polls global request processor provider for data.
     *
     * @param data
     *            WebData to add polled information to.
     */
    private void pollGlobalRequestProcessor(final WebData data) {
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = { "requestCount", "errorCount", "processingTime", "maxTime", "bytesSent", "bytesReceived" };

        // TODO add: bytesSent, bytesReceived

        long reqCount = 0;
        long errCount = 0;
        long procTime = 0;
        long maxTime = 0;
        long bytesSent = 0;
        long bytesReceived = 0;

        refreshGRPList();
        if (onGRP != null) {

            for (int i = 0; i < onGRP.length; i++) {
                try {
                    cnx = getMBeanServerConnection();
                    al = cnx.getAttributes(onGRP[i], attIds);
                    releaseMbeanServerConnection();
                    cnx = null;
                    Iterator it = al.iterator();
                    Attribute att = (Attribute) it.next(); // requestCount
                    reqCount += ((Integer) att.getValue()).intValue();
                    att = (Attribute) it.next(); // errorCount
                    errCount += ((Integer) att.getValue()).intValue();
                    att = (Attribute) it.next(); // processingTime
                    procTime += ((Long) att.getValue()).longValue();
                    att = (Attribute) it.next(); // maxTime
                    long max = ((Long) att.getValue()).longValue();
                    maxTime = (max > maxTime ? max : maxTime);
                    att = (Attribute) it.next(); // bytesSent
                    bytesSent += ((Long) att.getValue()).longValue();
                    att = (Attribute) it.next(); // bytesReceived
                    bytesReceived += ((Long) att.getValue()).longValue();

                    data.setValid(true);
                } catch (Exception e) {
                    refreshableGRPs = true;
                    data.setValid(false);
                    e.printStackTrace(System.err);
                    //e.getCause().printStackTrace(System.err);
                }
            }
        }

        data.setHttpRequestCount(reqCount);
        data.setHttpErrorsCount(errCount);

        data.setProcTimeCount(procTime);
        data.setMaxProcTime(maxTime);
        data.setBytesSentCount(bytesSent);
        data.setBytesReceivedCount(bytesReceived);
    }

    /**
     * Refresh GlobalRequestPrtocessor list. The actual policy is to refresh the
     * list until it is not null;
     */
    private void refreshGRPList() {

        if (onGRP == null) {
            refreshableGRPs = true;
        }
        if (refreshableGRPs) {
            ObjectName[] ons = null;
            try {
                ons = queryON("*:type=GlobalRequestProcessor,*");
            } catch (Exception e) {
                // TODO log on debug
            }
            onGRP = ons;
            if (ons != null) {
                refreshableGRPs = false;
            }
        }
    }

    /**
     * Refresh ThreadPool list. The actual policy is to refresh the list until
     * it is not null;
     */
    private void refreshTreadPoolList() {

        if (onThreadPool == null) {
            refreshableThreadPools = true;
        }
        if (refreshableThreadPools) {
            ObjectName[] ons = null;
            try {
                ons = queryON("*:type=ThreadPool,*");
            } catch (Exception e) {
                // TODO log on debug
            }
            onThreadPool = ons;
            if (ons != null) {
                refreshableThreadPools = false;
            }
        }
    }

    /**
     * Polls thread pools for data.
     *
     * @param data
     *            WebData to add polled information to.
     */
    private void pollThreadPool(final WebData data) {
        // long nbTh = 0;
        long nbBusy = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = { "currentThreadsBusy" };
        refreshTreadPoolList();
        if (onThreadPool != null) {
            for (int i = 0; i < onThreadPool.length; i++) {
                try {
                    cnx = getMBeanServerConnection();
                    al = cnx.getAttributes(onThreadPool[i], attIds);
                    releaseMbeanServerConnection();
                    cnx = null;
                    Iterator it = al.iterator();
                    Attribute att = (Attribute) it.next();
                    // Even if there are no pending requests
                    // there is always a busy thread =>
                    // if busy = 1 don't count it
                    long busy = ((Integer) att.getValue()).intValue();
                    nbBusy += (busy == 1 ? 0 : busy);
                    data.setValid(true);
                } catch (Exception e) {
                    refreshableThreadPools = true;
                    data.setValid(false);
                    e.printStackTrace(System.err);
                    e.getCause().printStackTrace(System.err);
                }
            }
        }
        data.setPendingHttpRequests(nbBusy);
    }

    /**
     * Web statistics Object Name.
     */
    private ObjectName onWebStats = null;

    public ObjectName getOnWebStats() {
        if (onWebStats == null) {
            try {
                onWebStats = new ObjectName(this.getDomain() + ":type=HttpStatistics,J2EEServer=" + this.getServer());
            } catch (MalformedObjectNameException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
        return onWebStats;
    }

    public void setOnWebStats(ObjectName onWebStats) {
        this.onWebStats = onWebStats;
    }

    /**
     * Global request processors.
     */
    private ObjectName[] onGRP = null;

    /**
     * Session managers.
     */
    private ObjectName[] onSessions = null;

    /**
     * Thread pools.
     */
    private ObjectName[] onThreadPool = null;

    /**
     * Refresh MBean lists
     */
    private boolean refreshableSessions = true;

    private boolean refreshableGRPs = true;

    private boolean refreshableThreadPools = true;
}
