/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxAp.java 6157 2010-03-16 14:39:48Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd;

import javax.management.MBeanServerConnection;
import javax.naming.Context;
import javax.naming.NamingException;

import javax.resource.ResourceException;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import org.ow2.jasmine.adapter.jmx.pool.api.IJMXConnectionFactory;
import org.ow2.jasmine.adapter.jmx.pool.api.IJMXConnectionServer;
import org.ow2.jasmine.adapter.jmx.pool.api.JMXConnectionParam;

public class JmxAp {

    /**
     * Command dispatcher instance.
     */
    private CommandDispatcher cmdDispatcher = null;

    /**
     * Current JMX URL.
     */
    private String jmxUrl = JmxHelper.DEFAULT_URL;

    /**
     * The JMX factory to get connection in embedded mode.
     */
    private IJMXConnectionFactory jmxFactory = null;

    /**
     * Current JMX jmxConnection.
     */
    private MBeanServerConnection jmxConnection = null;

    /**
     * Current JMX jmxConnection in embedded mode.
     */
    private IJMXConnectionServer jmxConnectionEmbedded = null;

    /**
     * Logger
     */
    private Log logger = LogFactory.getLog(JmxAp.class);

    /**
     * Creates a JmxAp that connects to a JMX URL.
     *
     * @param jmxUrl URL to connect to.
     * @param cmdDispatcher CommandDispatcher instance corresponding to the current MBeanCmd command.
     */
    public JmxAp(final String jmxUrl, final CommandDispatcher cmdDispatcher) {
        this.jmxUrl = jmxUrl;
        if (cmdDispatcher != null) {
            this.cmdDispatcher = cmdDispatcher;
            if (this.cmdDispatcher.isEmbedded()) {
                this.connectJMX();
            }
        }
    }

    /**
     * Creates a JmxAp that connects to the default target's JMX URL.
     */
    public JmxAp(final CommandDispatcher cmdDispatcher) {
        this(JmxHelper.getDefaultJmxTarget(), cmdDispatcher);
    }

    /**
     * @return The String version of the JMX URL connected to.
     */
    public String getJmxUrl() {
        return this.jmxUrl;
    }

    /**
     * @return The commandDispatcher corresponding to the current MBeanCmd command.
     */
    public CommandDispatcher getCmdDispatcher() {
        return this.cmdDispatcher;
    }

    /**
     * Establishes a JMX connection. This method is called only if MBeanCmd is
     * embedded. First, the naming context is get. If an error occurs during
     * this first step, the context will stay null. The second step is a look up
     * on the naming context to get the factory which will provide a JMX
     * connection. The JNDI name looked up is the url of the JMX url to get
     * information from where all ":" and "/" are replaced by "_". This means
     * that the RA has to be registered using this way.
     */
    private void connectJMX() {
        if (logger.isDebugEnabled()) {
            logger.debug("Connecting to the jmxFactory ...");
        }
        Context ctx = JmxHelper.getContext();
        if (ctx == null) {
            return;
        }

        String lookup = this.jmxUrl.replaceAll(":|/", "_");
        try {
            jmxFactory = (IJMXConnectionFactory) ctx.lookup(lookup);
        } catch (NamingException e) {
            logger.error("{1} on JMX connection factory named {0}.  An unmanaged JMX connection will be used instead. ",lookup, e.getMessage());
        }

    }

    /**
     * Get the JMX connection. If MBeanCmd is embedded, the connection will be
     * provide by a factory previously get. If MBeanCmd runs in standalone mode
     * or if the factory is null, a non managed JMX connection will be set.
     *
     * @return The MBeanServerConnection instance used to connect to the J2EE
     *         server via JMX.
     */
    public MBeanServerConnection getMBeanServerConnection(boolean usePool) {
        if(logger.isDebugEnabled()) {
            logger.debug("getting MBeanserverConnection");
        }
        if (this.jmxConnection == null) {
            String jmxUser = JmxHelper.getJmxUser(jmxUrl);
            String jmxPassword = JmxHelper.getJmxPassword(this.jmxUrl);
            String jmxProtocolProviderPackage = JmxHelper.getJmxProtocolProviderPackage(jmxUrl);
            if (usePool && this.cmdDispatcher != null && this.cmdDispatcher.isEmbedded() && this.jmxFactory != null) {
                try {
                    if(logger.isDebugEnabled()) {
                        logger.debug("getting MBeanserverConnection using the pool");
                    }
                    JMXConnectionParam jmxConnectionParam = new JMXConnectionParam(jmxUrl, jmxUser, jmxPassword, jmxProtocolProviderPackage);
                    jmxConnectionEmbedded = (IJMXConnectionServer) this.jmxFactory.getConnection(jmxConnectionParam);
                    jmxConnection = jmxConnectionEmbedded;
                } catch (ResourceException e) {
                    // TODO this.releaseMBeanServerConnection();
                    logger.error("Error while getting a connection : {0}", e.getMessage());
                }
            } else {
                if(logger.isDebugEnabled()) {
                    logger.debug("Creating a simple JMX Connection without using the pool, URL is {0}", this.jmxUrl);
                }
                jmxConnection = new JmxCnxWrapper(jmxUrl, jmxUser, jmxPassword, jmxProtocolProviderPackage);
            }
        }
        return jmxConnection;
    }


    /**
     * Get the JMX connection. If MBeanCmd is embedded, the connection will be
     * provide by a factory previously get. If MBeanCmd runs in standalone mode
     * or if the factory is null, a non managed JMX connection will be set.
     *
     * @return The MBeanServerConnection instance used to connect to the J2EE
     *         server via JMX.
     */
    public MBeanServerConnection getMBeanServerConnection() {
        return getMBeanServerConnection(true);
    }


    /**
     * In embedded mode only.
     *
     * Explicitly close the JMX connection in the pool, physically.
     */
    public void closeMBeanServerConnection() {
        if (this.cmdDispatcher != null && this.cmdDispatcher.isEmbedded() && this.jmxFactory != null) {
            try {
                this.jmxConnectionEmbedded.close();
                this.jmxConnectionEmbedded = null;
            } catch (ResourceException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
    }

    /**
     * In embedded mode only.
     *
     * Release the connection in the pool.
     */
    public void releaseMBeanServerConnection() {
        if (this.cmdDispatcher != null && this.cmdDispatcher.isEmbedded() && this.jmxFactory != null) {
            this.jmxConnectionEmbedded.release();
            this.jmxConnection = null;
        }
    }
}
