/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxAp.java 7534 2011-01-26 14:04:31Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd;

import javax.management.MBeanServerConnection;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Jmx Access Point
 * @author francois Waeselynck
 */
public class JmxAp {

    /**
     * Command dispatcher instance.
     */
    private CommandDispatcher cmdDispatcher = null;

    /**
     * Current JMX URL.
     */
    private String jmxUrl = JmxHelper.DEFAULT_URL;

    /**
     * Current JMX jmxConnection.
     */
    private MBeanServerConnection jmxConnection = null;

    /**
     * Logger
     */
    private Log logger = LogFactory.getLog(JmxAp.class);

    /**
     * Creates a JmxAp that connects to a JMX URL.
     *
     * @param jmxUrl URL to connect to.
     * @param cmdDispatcher CommandDispatcher instance corresponding to the current MBeanCmd command.
     */
    public JmxAp(final String jmxUrl, final CommandDispatcher cmdDispatcher) {
        this.jmxUrl = jmxUrl;
        if (cmdDispatcher != null) {
            this.cmdDispatcher = cmdDispatcher;
        }
    }

    /**
     * Creates a JmxAp that connects to the default target's JMX URL.
     */
    public JmxAp(final CommandDispatcher cmdDispatcher) {
        this(JmxHelper.getDefaultJmxTarget(), cmdDispatcher);
    }

    /**
     * @return The String version of the JMX URL connected to.
     */
    public String getJmxUrl() {
        return this.jmxUrl;
    }

    /**
     * @return The commandDispatcher corresponding to the current MBeanCmd command.
     */
    public CommandDispatcher getCmdDispatcher() {
        return this.cmdDispatcher;
    }

    /**
     * Get the JMX connection. If MBeanCmd is embedded, the connection will be
     * provided by a factory previously get. If MBeanCmd runs in standalone mode
     * or if the factory is null, a non managed JMX connection will be set.
     * @param usePool true if try to use the pool (most of the time)
     * @return The MBeanServerConnection instance used to connect to the J2EE
     *         server via JMX.
     */
    public MBeanServerConnection getMBeanServerConnection(boolean usePool) {
        if (this.jmxConnection == null) {
            logger.debug("getting MBeanserverConnection");
            String jmxUser = JmxHelper.getJmxUser(jmxUrl);
            String jmxPassword = JmxHelper.getJmxPassword(this.jmxUrl);
            String jmxProtocolProviderPackage = JmxHelper.getJmxProtocolProviderPackage(jmxUrl);
            jmxConnection = new JmxCnxWrapper(jmxUrl, jmxUser, jmxPassword, jmxProtocolProviderPackage);
        }
        return jmxConnection;
    }

    /**
     * Get the JMX connection. If MBeanCmd is embedded, the connection will be
     * provide by a factory previously get. If MBeanCmd runs in standalone mode
     * or if the factory is null, a non managed JMX connection will be set.
     *
     * @return The MBeanServerConnection instance used to connect to the J2EE
     *         server via JMX.
     */
    public MBeanServerConnection getMBeanServerConnection() {
        return getMBeanServerConnection(false);
    }

    /**
     * Release the connection in the pool, if pool is used.
     */
    public void releaseMBeanServerConnection() {
        if (cmdDispatcher != null && !cmdDispatcher.isEmbedded()) {
            // stand-alone, nothing to do ?
            return;
        }
    }
}
