/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */

package org.ow2.jasmine.monitoring.mbeancmd.audit;

import java.io.IOException;
import java.io.PrintStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Set;

import javax.management.AttributeList;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.ReflectionException;

/**
 * @author waeselyf
 * This classes manages poll of mbeans to target JOnAS or JVM instances
 */
public class Collector {

    /**
     * Triggers a poll. The name is used to build a pollID, from which the poll data can be retrieved.
     * All targets are polled. 
     *  
     * @param name
     * @return the PollID of the poll
     */
    public PollID poll(String name) {
        PollID id = new PollID(name);
        LinkedList<Metric> metrics = new LinkedList<Metric>();
        for (int i = 0; i < context.length; i++) {
            
            pollTarget(context[i],metrics);
        }
        id.setStamp();
        polls.put(id, metrics);
        perr.println("End of poll : " + id);
        perr.flush();
        return id;
    }

    /**
     * Polls a single target
     * @param ctx context associated to the target
     * @param metrics collected data is appended to metrics
     */
    private void pollTarget(AuditContext ctx,LinkedList<Metric> metrics) {
        
        perr.print("Polling " + ctx.getName() + " ... ");
        perr.flush();
        int n0 = metrics.size();
        int errors = 0;
        int tries = 0;
        
        MBeanServerConnection cnx = null;        
        try {
            cnx = ctx.getJmxap().getMBeanServerConnection();            
            Set<ObjectName> onset = cnx.queryNames(onPattern, null);
            Iterator<ObjectName> it = onset.iterator();
            while(it.hasNext()) {
                tries++;
                long t = System.currentTimeMillis();
                ObjectName on = it.next();
                AttributeList attl;
                try {
                    attl = cnx.getAttributes(on, attributes);
                    metrics.add(new Metric(t, ctx.getName(), on, attl));
                } catch (InstanceNotFoundException e) {
                    e.printStackTrace(perr);
                    errors++;
                } catch (ReflectionException e) {
                    e.printStackTrace(perr);
                    errors++;
                }                
            }
        } catch (IOException e) {
            e.printStackTrace(perr);
            errors++;
        } finally {
            ctx.getJmxap().releaseMBeanServerConnection();
        }
        perr.print(" " + (metrics.size() - n0) + " mbeans collected (" + tries + " tries, " + errors + " errors).");
        perr.flush();
    }

    /**
     * Get the metrics collected by the poll identified by id
     * @param id identifier of the poll
     * @return the list of metrics LinkedList<Metric>
     */
    public LinkedList<Metric> getPoll(PollID id) {
        return polls.get(id);        
    }
    
    /**
     * @return the context
     */
    public AuditContext[] getContext() {
        return context;
    }

    /**
     * @param context the context to set
     */
    public void setContext(AuditContext[] context) {
        this.context = context;
    } 

    /**
     * @return the onPattern
     */
    public ObjectName getOnPattern() {
        return onPattern;
    }

    /**
     * @param onPattern the onPattern to set
     */
    public void setOnPattern(ObjectName onPattern) {
        this.onPattern = onPattern;
    }

    /**
     * @return the attributes
     */
    public String[] getAttributes() {
        return attributes;
    }

    /**
     * @param attributes the attributes to set
     */
    public void setAttributes(String[] attributes) {
        this.attributes = attributes;
    }

    /**
     * @return the perr
     */
    public PrintStream getPerr() {
        return perr;
    }

    /**
     * @param perr the perr to set
     */
    public void setPerr(PrintStream perr) {
        this.perr = perr;
    }

    /**
     * ObjectName pattern to query MBeans 
     */
    private ObjectName onPattern = null;
    
    /**
     * Attributes to fetch 
     */
    private String[] attributes = null;
    
    /**
     * Audit targets
     */
    private AuditContext[] context = null;
    
    /*
     * Polls
     */
    private HashMap<PollID,LinkedList<Metric>> polls = new HashMap<PollID,LinkedList<Metric>>();

    /**
     * Error stream
     */
    private PrintStream perr = null;
    
}
