/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Replay.java 1928 2008-07-30 08:45:19Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.File;
import java.io.IOException;
import java.io.PipedOutputStream;
import java.io.PrintStream;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.Iner;
import org.ow2.jasmine.monitoring.mbeancmd.Outer;
import org.ow2.jasmine.monitoring.mbeancmd.graph.Grapher;


/**
 * Replays an mbeancmd session based on the standard input or a file.
 */
public class Replay extends AbstractCommand {
    /**
     * Constructor: calls {@link Replay#setOptions()}.
     */
    public Replay() {
        this.setOptions();
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see AbstractCommand#exec()
     */
    public int exec(final CommandDispatcher cmdDispatcher) {
        Thread tailThread = null;
        try {
            parseCommandLine(arguments);

            PipedOutputStream out = null;
            PipedOutputStream src = null;
            PrintStream target = null;

            // Input flow
            Iner iner = null;
            if (this.inputFile != null) {
                try {
                    iner = new Iner(new File(this.inputFile));
                } catch (Exception e) {
                    e.printStackTrace();
                    return 1;
                }
            } else {
                iner = new Iner(System.in);
            }
            src = iner.getSink();

            Outer matchOuter = null;
            if (isTimeMatchOption || isRowMatchOption || isColumnOption) {
                out = new PipedOutputStream();
                target = new PrintStream(out);
                matchOuter = new Outer(src, target);
                matchOuter.setId("matchOuter");

                if (isTimeMatchOption) {
                    matchOuter.setTimeFilter(
                            timeMatchArgs[0],
                            ("long".equalsIgnoreCase(timeMatchArgs[1]) ? null : timeMatchArgs[1]),
                            timeMatchArgs[2],
                            timeMatchArgs[3]
                    );
                }

                if (isRowMatchOption) {
                    matchOuter.setRegexp(this.rowMatchPattern);
                }

                if (isColumnOption) {
                    matchOuter.setFormatter(columns);
                }
                src = out;
            }

            if (this.graphDef != null) {
                try {
                    Grapher gr = new Grapher(src, this.graphDef);
                    gr.start();
                } catch (IOException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
            } else {
                Outer outer = null;
                if (this.outputFilePath != null) {
                    outer = new Outer(src, new File(outputFilePath));
                } else {
                    outer = new Outer(src, System.out);
                }
                outer.setId("ouput");

                tailThread = new Thread(outer);
                tailThread.start();
            }

            if (matchOuter != null) {
                new Thread(matchOuter).start();
            }
            if (iner != null) {
                new Thread(iner).start();
            }
        } catch (Exception e) {
            e.printStackTrace();
            return 2;
        }

        // if output, join on the outer thread
        if (tailThread != null) {
            try {
                tailThread.join();
            } catch (InterruptedException e) {
                // nothing to do
            }
        }

        // If graph, loop. Closing the graph window will terminate the program.
        if (this.graphDef != null) {
            // Sleep for about 292 million 471 thousand 208 years,
            // then repeat infinitely many times.
            while (true) {
                try {
                    Thread.sleep(Long.MAX_VALUE);
                } catch (InterruptedException e) {
                    break;
                }
            }
        }

        return 0;
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see AbstractCommand#summary()
     */
    public String summary() {
        return "Replays captures from a text file.";
    }

    /**
     * Tests the Replay implementation.
     *
     * @param args  Arguments to pass to Poll#setArgs.
     */
    public static void main(final String[] args) {
        Replay p = new Replay();
        p.setArgs("replay", args);
        p.exec(null);
    }

    /**
     * Parses the command line arguments into {@link Replay#commandLine}.
     *
     * @param args  Arguments to parse.
     *
     * @throws ParseException  If parsing fails.
     */
    public void parseCommandLine(final String[] args) throws ParseException {
        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options,args);

        if (commandLine.hasOption("i")) {
            this.inputFile = commandLine.getOptionValue("i");
        }

        if (commandLine.hasOption("o")) {
            this.outputFilePath = commandLine.getOptionValue("o");
        }

        if (commandLine.hasOption("graph")) {
            this.graphDef = commandLine.getOptionValue("graph");
        }

        this.isRowMatchOption = commandLine.hasOption("m");
        if (isRowMatchOption) {
            this.rowMatchPattern = commandLine.getOptionValue("m");
        }

        this.isTimeMatchOption = commandLine.hasOption("t");
        if (isTimeMatchOption) {
            this.timeMatchArgs = commandLine.getOptionValues("t");
        }

        this.isSeparator = commandLine.hasOption("s");
        if (isSeparator) {
            this.separator = commandLine.getOptionValue("s");
            Outer.setSeparator(separator);
        }
        this.isColumnOption = commandLine.hasOption("c");
        if (isColumnOption) {
            this.columns = commandLine.getOptionValues("c");
        }
    }

    /**
     * Sets all options that are parseable from the command line.
     *
     * @see Replay#options
     */
    private void setOptions() {

        options = new Options();
        //options.addOption("v", "verbose",false,"Verbose mode");

        // Input file
        Option ifile = new Option("i", "input", true, "Input file");
        ifile.setRequired(false);
        ifile.setArgName("file");
        ifile.setArgs(1);
        options.addOption(ifile);

        // row matching
        Option rowmatch = new Option("m", "match", true, "Row matching pattern");
        rowmatch.setRequired(false);
        rowmatch.setArgName("pattern");
        rowmatch.setArgs(1);
        options.addOption(rowmatch);


        // Time interval
        Option timematch = new Option("t", "time", true, "Time interval matching pattern");
        timematch.setRequired(false);
        timematch.setArgName("field format from to");
        timematch.setArgs(4);
        options.addOption(timematch);

        // Columns to print
        Option columns = new Option("c", "columns", true, "Specifies the columns to print");
        columns.setRequired(false);
        columns.setArgName("column list");
        columns.setArgs(Option.UNLIMITED_VALUES);
        options.addOption(columns);

        // graph output
        Option graph = new Option("graph", "graph", true, "Enable graphical output");
        graph.setRequired(false);
        graph.setArgName("graphDefinition");
        graph.setArgs(1);
        options.addOption(graph);

        // file output
        Option file = new Option("o", "output", true, "Send output to file instead of stdout");
        file.setRequired(false);
        file.setArgName("path");
        file.setArgs(1);
        options.addOption(file);

        // Field separator
        Option separator = new Option("s", "separator", true, "Set the column separator");
        separator.setRequired(false);
        separator.setArgName("separator");
        separator.setArgs(1);
        options.addOption(separator);
    }

    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * Input file name, null for no file input.
     */
    private String inputFile = null;

    /**
     * Graph definitions, null for no graph output.
     */
    private String graphDef = null;

    /**
     * Output file path, null for no file output.
     */
    private String outputFilePath = null;

    /**
     * Filter rows ?
     */
    private boolean isRowMatchOption = false;

    /**
     * Row filter pattern.
     */
    private String rowMatchPattern = null;

    /**
     * Filter time ?
     */
    private boolean isTimeMatchOption = false;

    /**
     * Time filter arguments.
     */
    private String[] timeMatchArgs = null;

    /**
     * Is separator set ?
     */
    private boolean isSeparator = true;

    /**
     * Separator to use.
     */
    private String separator = ";";

    /**
     * Only output certain columns ?
     */
    private boolean isColumnOption = false;

    /**
     * Columns to output.
     */
    private String[] columns = null;
}
