/**
 * JASMINe
 * Copyright (C) 2005-2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Stat.java 7482 2011-01-21 07:33:16Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.File;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeSet;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.management.openmbean.CompositeData;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAp;
import org.ow2.jasmine.monitoring.mbeancmd.JmxHelper;
import org.ow2.jasmine.monitoring.mbeancmd.Outer;
import org.ow2.jasmine.monitoring.mbeancmd.context.StatContext;
import org.ow2.jasmine.monitoring.mbeancmd.graph.Grapher;
import org.ow2.jasmine.monitoring.mbeancmd.graph.conf.Configurator;
import org.ow2.jasmine.monitoring.mbeancmd.jasmine.JasmineConnector;

/**
 * Periodically polls attributes of one or more mbeans on one or more J2EE
 * servers.
 */
public class Stat extends AbstractCommand {

    protected boolean goOn;

    /**
     */
    public Stat() {
        setOptions();
    }

    /**
     * Tests the Stat implementation.
     *
     * @param args
     *            Arguments to pass to Stat#setArgs.
     */
    public static void main(final String[] args) {
        Stat p = new Stat();
        p.setArgs("stat", args);
        p.exec(null);
    }

    public void stop() {
        goOn = false;
    }

    /**
     * Implementation of inherited abstract method.
     *
     * Will never return except in the case of failure.
     *
     * @see AbstractCommand#exec()
     */
    public int exec(final CommandDispatcher cmdDispatcher) {

        this.cmdDispatcher = cmdDispatcher;

        try {
            parseCommandLine(arguments);
        } catch (Exception e) {
            cmdDispatcher.setFailed(e.getMessage());
            return 1;
        }

        /*
         * out (or pout) is : - the output of the sampler - the input for the
         * Outer to print into System.out or into a file
         *
         * sink is: - the sink of the Outer - the input for the graph, if any
         */
        PipedOutputStream out = new PipedOutputStream();
        pout = new PrintStream(out);

        Outer outer = null;
        try {
            if (isConsoleOption) {
                // output on console forced
                outer = new Outer(out, System.out);
            }

            if (this.outputFilePath != null) {
                // create outer for a file
                if (outer == null) {
                    outer = new Outer(out, new File(outputFilePath));
                } else {
                    // output on both file and console
                    Outer consoleOuter = outer;
                    PipedOutputStream sink = new PipedOutputStream();
                    consoleOuter.setSink(sink);
                    outer = new Outer(sink, new File(outputFilePath));
                    new Thread(consoleOuter).start();
                }
            } else if (this.jasmineURI == null && outer == null) {
                outer = new Outer(out, System.out);
            }

            if (this.jasmineURI != null) {
                if (outer == null) {
                    outer = JasmineConnector.connect(out, this.jasmineURI);
                } else {
                    Outer oldOuter = outer;
                    outer = JasmineConnector.connect(outer, this.jasmineURI);
                    new Thread(oldOuter).start();
                }
            }
        } catch (IOException e) {
            logger.error("Cannot set up file or jasmine output :  {0}", e.getMessage(), e);
        }

        if (this.graphDef != null) {
            try {
                PipedOutputStream sink = new PipedOutputStream();
                outer.setSink(sink);
                Grapher gr = new Grapher(sink, this.graphDef);
                gr.start();
            } catch (IOException e) {
                logger.error("Cannot set up graph output :  {0}", e.getMessage(), e);
            }
        }
        new Thread(outer).start();

        try {
            process();
            cmdDispatcher.setStopped();
        } catch (Exception e) {
            cmdDispatcher.setFailed(e.getMessage());
        }
        return 0;
    }

    /**
     * Calls {@link AbstractCommand#help()} and prints the DTD for the graph.
     */
    public void help() {
        super.help();

        System.out.println("\n" + "DTD for graph definitions:\n" + "--------------------------\n\n");
        LineNumberReader ln = Configurator.getDTD();
        if (ln != null) {
            try {
                String line = null;
                while ((line = ln.readLine()) != null) {
                    System.out.println(line);
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            logger.error("Error: DTD not found !");
        }
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see AbstractCommand#summary()
     */
    public String summary() {
        return "Periodically probes MBeans in one or multiple J2EE servers";
    }

    /**
     * Parses the command line arguments into {@link Stat#commandLine}.
     *
     * @param args
     *            Arguments to parse.
     *
     * @throws ParseException
     *             If parsing fails.
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    public void parseCommandLine(final String[] args) throws ParseException, MalformedObjectNameException {
        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);

        setObjectName();

        if (commandLine.hasOption("p")) {
            setPeriod();
        }
        if (commandLine.hasOption("r")) {
            setRefreshPeriod();
        }

        if (commandLine.hasOption("cmdid")) {
            this.cmdid = commandLine.getOptionValue("cmdid");
        }

        if (commandLine.hasOption("graph")) {
            this.graphDef = commandLine.getOptionValue("graph");
        }
        if (commandLine.hasOption("f")) {
            this.outputFilePath = commandLine.getOptionValue("f");
        }
        if (commandLine.hasOption("jasmine")) {
            this.jasmineURI = commandLine.getOptionValue("jasmine");
        }

        this.isConsoleOption = commandLine.hasOption("console");

        if (commandLine.hasOption("s")) {
            this.separator = commandLine.getOptionValue("s");
            Outer.setSeparator(separator);
        }

        if (commandLine.hasOption("rate")) {
            this.rateOption = true;
            this.attsRate = commandLine.getOptionValues("rate");
        }

        if (commandLine.hasOption("delta")) {
            this.deltaOption = true;
            this.attsDelta = commandLine.getOptionValues("delta");
        }
        if (commandLine.hasOption("slope")) {
            this.slopeOption = true;
            String[] slopeOptionValues = commandLine.getOptionValues("slope");
            if ((slopeOptionValues.length % 2) != 0) {
                /* option = slope -> number of attributes should be pair */
                throw new MalformedObjectNameException("Command failed at: slope option. Number of attributes should be pair.");
            }
            this.attsSlope = slopeOptionValues;
        }

        if (commandLine.hasOption("a")) {
            this.attsSimple = commandLine.getOptionValues("a");
            this.aOption = true;
        } else {
            // if neither -a, nor -rate, -delta, -slope, that means all the attributes must be polled
            if (!this.deltaOption && !this.rateOption && !this.slopeOption) {
                this.aOption = true;
                // attsSimple is not set at this point, it will be set further ..
            }
        }

        if (this.deltaOption || this.rateOption || this.slopeOption) {
            this.oldValues = true;
        }
    }

    /**
     * Sets the object name based on the "name" argument in the command line.
     *
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    private void setObjectName() throws MalformedObjectNameException {
        String mbean = commandLine.getOptionValue("name");

        if (mbean.contains("\"")) {

            StringBuffer quotedMbean = new StringBuffer();
            StringTokenizer st = new StringTokenizer(mbean, "\"");
            quotedMbean.append(st.nextToken());
            while (st.hasMoreTokens()) {

                quotedMbean.append(ObjectName.quote(st.nextToken()));

                if (st.hasMoreTokens()) {
                    quotedMbean.append(st.nextToken());
                }
            }
            on = new ObjectName(quotedMbean.toString());
        } else {
            on = new ObjectName(mbean);
        }
    }

    /**
     * Sets the period if the command line has the "p" option.
     *
     * @throws NumberFormatException
     *             Number after the "p" option is invalid.
     */
    private void setPeriod() throws NumberFormatException {
        period = Long.parseLong(commandLine.getOptionValue("p"));
    }

    /**
     * Sets the refresh period if the command line has the "r" option.
     *
     * @throws NumberFormatException
     *             Number after the "r" option is invalid, note that "never" is
     *             a valid number.
     */
    private void setRefreshPeriod() throws NumberFormatException {
        String val = commandLine.getOptionValue("r");
        if ("never".equalsIgnoreCase(val)) {
            isRefreshable = false;
        } else {
            refreshPeriod = Long.parseLong(val);
        }
    }

    /**
     * Polls the MBeans.
     */
    private void mbeanPoll() throws Exception {

        if (oldValues) {
            // Determine all the attributes concerned by rate, delta, slope computing
            attsRateDelatSlope = new ArrayList<String>();
            if (attsRate != null) {
                for (String att : attsRate) {
                    if (!attsRateDelatSlope.contains(att)) {
                        attsRateDelatSlope.add(att);
                    }
                }
            }
            if (attsDelta != null) {
                for (String att : attsDelta) {
                    if (!attsRateDelatSlope.contains(att)) {
                        attsRateDelatSlope.add(att);
                    }
                }
            }
            if (attsSlope != null) {
                for (String att : attsSlope) {
                    if (!attsRateDelatSlope.contains(att)) {
                        attsRateDelatSlope.add(att);
                    }
                }
            }
        }

        ArrayList<String> allAttributes = new ArrayList<String>();
        if (this.aOption) {
            for (int i = 0; i < attsSimple.length; i++) {
                if (!allAttributes.contains(attsSimple[i])) {
                    allAttributes.add(attsSimple[i]);
                }
            }
        }
        if (this.rateOption) {
            for (int i = 0; i < attsRate.length; i++) {
                if (!allAttributes.contains(attsRate[i])) {
                    allAttributes.add(attsRate[i]);
                }
            }
        }
        if (this.deltaOption) {
            for (int i = 0; i < attsDelta.length; i++) {
                if (!allAttributes.contains(attsDelta[i])) {
                    allAttributes.add(attsDelta[i]);
                }
            }
        }
        if (this.slopeOption) {
            for (int i = 0; i < attsSlope.length; i++) {
                if (!allAttributes.contains(attsSlope[i])) {
                    allAttributes.add(attsSlope[i]);
                }
            }
        }

        // Check at least one attribute to poll
        if (allAttributes.isEmpty()) {
            throw new Exception("No attribute to poll");
        }

        MBeanServerConnection cnx = null;

        // Poll
        goOn = (contexts != null ? true : false);
        boolean refreshNow = false;
        while (goOn) {
            // create one thread per polled server
            for (int i = 0; i < contexts.length; i++) {
                // The polled target is represented by a 'context'
                StatContext context = contexts[i];

                cnx = context.getJmxap().getMBeanServerConnection();
                long now = System.currentTimeMillis();
                refreshNow = isRefreshable && (now >= context.getRefreshDeadLine());
                if (refreshNow) {
                    try {
                        context.updateOnames(on);
                        context.setRefreshDeadLine(now + this.refreshPeriod * 1000);
                    } catch (Exception e) {
                        // was: continue; !
                        throw e;
                    }
                }
                Iterator<?> it = context.getOnames().iterator();
                while (it.hasNext()) {
                    long t = System.currentTimeMillis();
                    // The polled object (MBean) is represented by 'onGot'
                    ObjectName onGot = (ObjectName) it.next();
                    AttributeList attl = new AttributeList();
                    try {
                        /*
                         * Workarround for WLS
                         * attl = cnx.getAttributes(onGot, atts);
                         */
                        // A 'currentValues' map is created for each object (onGot) of each target (context)
                        // It gives the value of all the attributes corresponding to the different attribute options.
                        HashMap<String, Object> currentValues = getAttributes(cnx, onGot, allAttributes, context.getName());

                        if (this.aOption) {
                            for (int j = 0; j < attsSimple.length; j++) {
                                Attribute attribute = new Attribute(attsSimple[j], currentValues.get(attsSimple[j]));
                                attl.add(attribute);
                            }
                        }
                        if (this.rateOption) {
                            try {
                                AttributeList attributesRate = getAttributesRate(context, onGot, attsRate, currentValues);
                                attl.addAll(attributesRate);
                            } catch (MalformedObjectNameException ex) {
                                ex.printStackTrace();
                                throw ex;
                            }
                        }
                        if (this.deltaOption) {
                            try {
                                AttributeList attributesDelta = getAttributesDelta(context, onGot, attsDelta, currentValues);
                                attl.addAll(attributesDelta);
                            } catch (MalformedObjectNameException ex) {
                                ex.printStackTrace();
                                throw ex;
                            }
                        }
                        if (this.slopeOption) {
                            try {
                                AttributeList attributesSlope = getAttributesSlope(context, onGot, attsSlope, currentValues);
                                attl.addAll(attributesSlope);
                            } catch (MalformedObjectNameException ex) {
                                ex.printStackTrace();
                                throw ex;
                            }
                        }

                        if (oldValues) {
                            updateOldValues(onGot, context, currentValues);
                        }

                        printMBeanPoll(t, context, attl, onGot);
                        if (!context.isStarted()) {
                            context.setStarted(true);
                        }
                        currentValues = null;
                    } catch (Exception e) {
                        logger.error("Error on mbean {0} for {1} - {2}", onGot, context.getName(), e.getMessage());
                        throw e;
                    }
                }
            }

            // Sleep for a while
            try {
                Thread.sleep(period * 1000);
            } catch (InterruptedException e) {
                goOn = false;
            }
        }
        pout.close();
    }

    /**
     * Construct a Map containing mapping for each attribute name in attsName list, the attribute's value.
     * @param cnx connection for the polled target
     * @param on polled MBean's name
     * @param attsName list of attribute names
     * @param target target's name
     * @return  a map of attributes name to attributes values.
     * @throws AttributeNotFoundException
     * @throws InstanceNotFoundException
     * @throws MBeanException
     * @throws ReflectionException
     * @throws IOException
     */
    private HashMap<String, Object> getAttributes(final MBeanServerConnection cnx, final ObjectName on, final ArrayList<String> attsName, final String target)
        throws AttributeNotFoundException, InstanceNotFoundException, MBeanException, ReflectionException, IOException {

        HashMap<String, Object> newValueOfAttributes = new HashMap<String, Object>();
        boolean tryOneByOne = false;

        /*
         * Try to get all attributes in one shot
         */
        try {
            AttributeList attributesList = cnx.getAttributes(on, toArray(attsName));
            if (attributesList != null) {
                for (int i = 0; i < attributesList.size(); i++) {
                    Attribute attribute = (Attribute) attributesList.get(i);
                    newValueOfAttributes.put(attribute.getName(), attribute.getValue());
                }
            }
        } catch (ArrayIndexOutOfBoundsException e) {
            /*
             * workarround for WLS, if attributes cannot be read in one shot
             * due to IOP trouble which raise an ArrayIndexOutOfBoundsException exception.
             * A warning is printed every day
             */
            if ((warningTrigger % (86400 / period)) == 0) {
                logger.warn("MBeanCmd stat : cannot get attributes in one shot, trying one by one ({0}) for {1}", on, target);
                warningTrigger++;
            }
            tryOneByOne = true;
        }
        if (tryOneByOne) {
            /*
             * get attributes one by one
             */
            for (int i = 0; i < attsName.size(); i++) {
                newValueOfAttributes.put(attsName.get(i), cnx.getAttribute(on, attsName.get(i)));
            }
        }
        return newValueOfAttributes;
    }

    /**
     * Construct a list of javax.management.Attribute objects corresponding to the attributes for which the rate must be calculated.
     * @param cnx connection for the polled target
     * @param on polled MBean's name
     * @param attsRate attributes name for which the rate must be calculated
     * @param currentValues last polled values of all the attributes
     * @return Each Attribute object has a name = rate_attributeName and the value the rate calculated using current value - previous value / period
     * @throws MalformedObjectNameException
     */
    private AttributeList getAttributesRate(final StatContext context, final ObjectName on, final String[] attsRate, final HashMap<String, Object> currentValues) throws MalformedObjectNameException {

        AttributeList result = new AttributeList();

        /*
         * Calculate the rate of each attribute
         */
        for (int i = 0; i < attsRate.length; i++) {
            String attName = attsRate[i];
            Object currentValue = currentValues.get(attName);

            boolean hasNumeriqueValue = currentValue instanceof Integer || currentValue instanceof Double || currentValue instanceof Long || currentValue instanceof Float;
            if (hasNumeriqueValue) {
                String name = "rate_" + attsRate[i];
                Object oldValue = context.getOldValue(on, attName);
                if (oldValue == null) {
                    // First time, init old value by current Value
                    oldValue = currentValue;
                    context.setOldValue(on, attName, oldValue);
                }

                Double dX = new Double(currentValue.toString()) - new Double(oldValue.toString());
                Double dT = new Double(this.period);

                Double value;

                if (dT == 0) {
                    value = Double.MAX_VALUE;
                } else {
                    /* Rounding dX/dY double with "numberDecimalPlaces" elements after the comma */
                    value = Stat.floor(dX / dT, this.numberDecimalPlaces);
                }

                Attribute attribute = new Attribute(name, value);
                result.add(attribute);
            } else {
                throw new MalformedObjectNameException("Command failed at: All the attributes of rate option should be numeric. Problem with " + attName + " of MBean " + on.toString());
            }
        }
        return result;
    }

    private void updateOldValues(final ObjectName on, final StatContext context, final HashMap<String, Object> currentValues) {
        for (String attName : attsRateDelatSlope) {
            Object currentValue = currentValues.get(attName);
            // current value becomes old
            context.setOldValue(on, attName, currentValue);
        }
    }

    /**
     * Construct a list of javax.management.Attribute objects corresponding to the attributes for which the delta must be calculated.
     * @param cnx connection for the polled target
     * @param on polled MBean's name
     * @param attsRate attributes names for which the delta must be calculated
     * @param currentValues last polled values of all the attributes
     * @return Each Attribute object has a name = delta_attributeName and the value =  current value - previous value
     * @throws MalformedObjectNameException
     */
    private AttributeList getAttributesDelta(final StatContext context, final ObjectName on, final String[] attsDelta, final HashMap<String, Object> currentValues) throws MalformedObjectNameException {

        AttributeList result = new AttributeList();

        /*
         * Calculate delta
         */
        for (int i = 0; i < attsDelta.length; i++) {
            String attName = attsDelta[i];
            Object currentValue = currentValues.get(attName);

            boolean hasNumeriqueValue = currentValue instanceof Integer || currentValue instanceof Double || currentValue instanceof Long || currentValue instanceof Float;
            if (hasNumeriqueValue) {
                String name = "delta_" + attsDelta[i];
                Object oldValue = context.getOldValue(on, attName);
                if (oldValue == null) {
                    // First time, init old value by current Value
                    oldValue = currentValue;
                    context.setOldValue(on, attName, oldValue);
                }

                Double dX = new Double(currentValue.toString()) - new Double(oldValue.toString());

                Attribute attribute = new Attribute(name, dX);
                result.add(attribute);
            } else {
                throw new MalformedObjectNameException("Command failed at: All the attributes of delta option should be numeric : " + attName + ".");
            }
        }
        return result;
    }

    /**
     * Construct a list of javax.management.Attribute objects corresponding to the pair of attributes for which the slope must be calculated.
     * @param cnx connection for the polled target
     * @param on polled MBean's name
     * @param attsRate attribute name pairs (attName1, attName2) for which the delta must be calculated
     * @param currentValues last polled values of all the attributes
     * @return Each Attribute object has a name = attName1_per_attName2 and the value =  delta(attName1) / delta(attName2)
     * @throws MalformedObjectNameException
     */
    private AttributeList getAttributesSlope(final StatContext context, final ObjectName on, final String[] attsSlope, final HashMap<String, Object> currentValues) throws MalformedObjectNameException {

        AttributeList result = new AttributeList();

        /*
         * Calculate the slope
         */
        for (int i = 0; i < (attsSlope.length - 1); i += 2) {
            String attName1 = attsSlope[i];
            Object currentValue1 = currentValues.get(attName1);
            Object oldValue1 = context.getOldValue(on, attName1);
            if (oldValue1 == null) {
                // First time, init old value by current Value
                oldValue1 = currentValue1;
                context.setOldValue(on, attName1, oldValue1);
                context.setOldValue(on, attName1, currentValue1);
            }

            String attName2 = attsSlope[i + 1];
            Object currentValue2 = currentValues.get(attName2);
            Object oldValue2 = context.getOldValue(on, attName2);
            if (oldValue2 == null) {
                // First time, init old value by current Value
                oldValue2 = currentValue2;
                context.setOldValue(on, attName2, oldValue2);
            }

            boolean hasNumeriqueValue1 = currentValue1 instanceof Integer || currentValue1 instanceof Double || currentValue1 instanceof Long || currentValue1 instanceof Float;
            boolean hasNumeriqueValue2 = currentValue2 instanceof Integer || currentValue2 instanceof Double || currentValue2 instanceof Long || currentValue2 instanceof Float;

            if (hasNumeriqueValue1 && hasNumeriqueValue2) {
                String name = attName1 + "_per_" + attName2;

                Double dX = new Double(currentValue1 + "") - new Double(oldValue1 + "");
                Double dY = new Double(currentValue2 + "") - new Double(oldValue2 + "");
                Double value;
                if (dX == 0) {
                    value = new Double(0);
                } else if (dY == 0) {
                    value = Double.MAX_VALUE;
                } else {
                    /* Rounding dX/dY double with "numberDecimalPlaces" elements after the comma */
                    value = Stat.floor(dX / dY, this.numberDecimalPlaces);
                }

                Attribute attribute = new Attribute(name, value);
                result.add(attribute);
            } else {
                if (!hasNumeriqueValue1 && !hasNumeriqueValue2) {
                    throw new MalformedObjectNameException("Command failed at: All the attributes of slope option should be numeric : " + attName1 + "," + attName2 + ".");
                } else if (!hasNumeriqueValue1) {
                    throw new MalformedObjectNameException("Command failed at: All the attributes of slope option should be numeric : " + attName1 + ".");
                } else {
                    throw new MalformedObjectNameException("Command failed at: All the attributes of slope option should be numeric : " + attName2 + ".");
                }
            }
        }
        return result;
    }

    /**
     * Try to guess attributes of MBean(s).
     * @param context the target on which we are searching
     * @param on the name or the pattern of the MBean(s) we are looking for
     */
    private String[] getAttsToPoll(final StatContext context, final ObjectName on) throws IOException {
        String[] atts = null;
        ObjectName onGot = null;
        Iterator<?> it = null;
        try {
            it = context.getMBeanServerConnection().queryNames(on, null).iterator();
        } catch (IOException e) {
            logger.error("Cannot get MBeans using {0} in target {1}", on.toString(), context.getName());
            throw e;
        }
        while (it.hasNext()) {
            // PhD Take only the first one ?
            onGot = (ObjectName) it.next();
            MBeanAttributeInfo[] info;
            try {
                info = context.getMBeanServerConnection().getMBeanInfo(onGot).getAttributes();
                atts = new String[info.length];
                for (int i = 0; i < atts.length; i++) {
                    atts[i] = info[i].getName();
                }
                break;
            } catch (Exception e) {
                logger.error("Cannot get attributes for MBean {0} in target {1}", onGot.toString(), context.getName());
            }
        }
        if (atts == null) {
            logger.warn("No attribute found for MBeans using {0} in target {1}", on.toString(), context.getName());
            atts = new String[0];
        }
        return atts;
    }

    /**
     * Prints a given header.
     *
     * @param attl
     *            Header attributes.
     *
     * @return Printed header.
     */
    private String getMBeanPollHeader(final AttributeList attl) {
        if (header == null) {
            StringBuffer buf = new StringBuffer(512);
            buf.append("date");
            buf.append(separator);
            buf.append("time");
            buf.append(separator);
            buf.append("sname");
            buf.append(separator);
            buf.append("server");
            buf.append(separator);
            buf.append("domain");
            buf.append(separator);
            buf.append("mbean");
            buf.append(separator);
            buf.append("cmdid");
            Iterator<?> it = attl.iterator();
            while (it.hasNext()) {
                buf.append(separator);
                Attribute att = (Attribute) it.next();
                if ((att.getValue() != null) && att.getValue().getClass().isArray()) {
                    putArrayIntoHeader(buf, att);
                } else if ((att.getValue() != null) && (att.getValue() instanceof Map)) {
                    putMapIntoHeader(buf, att);
                } else if ((att.getValue() != null) && (att.getValue() instanceof CompositeData)) {
                    putCompositeDataIntoHeader(buf, att);
                } else {
                    buf.append(att.getName());
                }
            }

            header = buf.toString();
        }
        return header;
    }

    /**
     * Puts a list of attributes into a buffer.
     *
     * @param buf
     *            String buffer to print into.
     * @param att
     *            Attribute list.
     */
    private void putArrayIntoHeader(final StringBuffer buf, final Attribute att) {
        Object obj = att.getValue();
        int length = Array.getLength(obj);
        for (int i = 0; i < length; i++) {
            buf.append((i == 0 ? "" : separator));
            buf.append(att.getName());
            buf.append("_");
            buf.append(i);
        }
    }

    /**
     * Puts a map of attributes into a buffer.
     *
     * @param buf
     *            String buffer to print into.
     * @param att
     *            Attribute map.
     */
    private void putMapIntoHeader(final StringBuffer buf, final Attribute att) {
        Map<?, ?> map = (Map<?, ?>) att.getValue();
        TreeSet<?> ts = new TreeSet<Object>(map.keySet());
        Iterator<?> it = ts.iterator();
        String sep = "";
        while (it.hasNext()) {
            buf.append(sep);
            sep = separator;
            buf.append(att.getName());
            buf.append("_");
            buf.append(it.next());
        }
    }

    /**
     * Puts composite data into a buffer.
     *
     * @param buf
     *            String buffer to print into.
     * @param att
     *            Attribute data.
     */
    private void putCompositeDataIntoHeader(final StringBuffer buf, final Attribute att) {
        CompositeData cdata = (CompositeData) att.getValue();
        Set<?> itemNames = cdata.getCompositeType().keySet();
        Iterator<?> it = itemNames.iterator();
        String sep = "";
        while (it.hasNext()) {
            buf.append(sep);
            sep = separator;
            buf.append(att.getName());
            buf.append("_");
            buf.append(it.next());
        }
    }

    /**
     * Prints out the poll data based on a header.
     *
     * @param t
     *            Time.
     * @param context
     *            Statistics context.
     * @param attl
     *            Attribute list.
     * @param oname
     *            Object to print as data.
     */
    private void printMBeanPoll(final long t, final StatContext context, final AttributeList attl, final ObjectName oname) {
        if (header == null) {
            pout.println(this.getMBeanPollHeader(attl));
        }

        Date d = new Date(t);
        pout.print(SIMPLEDATEFORMAT.format(d));
        pout.print(separator);
        pout.print(t);
        pout.print(separator);
        pout.print(context.getJmxUrl());
        pout.print(separator);
        pout.print(context.getServer());
        pout.print(separator);
        pout.print(context.getDomain());
        pout.print(separator);
        pout.print(oname);
        pout.print(separator);
        pout.print(cmdid);
        Iterator<?> it = attl.iterator();
        while (it.hasNext()) {
            pout.print(separator);
            Attribute att = (Attribute) it.next();
            if ((att.getValue() != null) && att.getValue().getClass().isArray()) {
                Object obj = att.getValue();
                int length = Array.getLength(obj);
                for (int i = 0; i < length; i++) {
                    pout.print((i == 0 ? "" : separator) + Array.get(obj, i));
                }
            } else if ((att.getValue() != null) && (att.getValue() instanceof Map)) {
                Map<?, ?> map = (Map<?, ?>) att.getValue();
                TreeSet<?> ts = new TreeSet<Object>(map.keySet());
                String sep = "";
                Iterator<?> it2 = ts.iterator();
                while (it2.hasNext()) {
                    pout.print(sep);
                    sep = separator;
                    pout.print(map.get(it2.next()));
                }
            } else if ((att.getValue() != null) && (att.getValue() instanceof CompositeData)) {
                CompositeData cdata = (CompositeData) att.getValue();
                Set<?> itemNames = cdata.getCompositeType().keySet();
                Iterator<?> it2 = itemNames.iterator();
                String sep = "";
                while (it2.hasNext()) {
                    pout.print(sep);
                    sep = separator;
                    pout.print(cdata.get((String) it2.next()));
                }
            } else {
                pout.print(att.getValue());
            }
        }
        pout.println();
    }

    /**
     * Sets all options that are parseable from the command line.
     *
     * @see Stat#options
     */
    private void setOptions() {
        options = new Options();
        options.addOption("v", "verbose", false, "Verbose mode");

        // ObjectName
        Option name = new Option("name", "objectName", true, "An ObjectName passed to every method");
        name.setRequired(true);
        name.setArgName("mbean name");
        name.setArgs(1);
        options.addOption(name);

        // Optional attribute list
        Option atts = new Option("a", "atts", false, "attributes to poll");
        atts.setRequired(false);
        atts.setOptionalArg(true);
        atts.setArgs(Option.UNLIMITED_VALUES);
        atts.setArgName("attributes");
        options.addOption(atts);

        // Period
        Option period = new Option("p", "period", true, "Polling period");
        period.setRequired(false);
        period.setArgName("period");
        period.setArgs(1);
        options.addOption(period);

        // MBean list refresh period
        Option refresh = new Option("r", "refresh", true, "Period for rebuilding the mbean list");
        refresh.setRequired(false);
        refresh.setArgName("refreshPeriod");
        refresh.setArgs(1);
        options.addOption(refresh);

        // graph output
        Option graph = new Option("graph", "graph", true, "Enable graphical output");
        graph.setRequired(false);
        graph.setArgName("graphDefinition");
        graph.setArgs(1);
        options.addOption(graph);

        // console output
        Option console = new Option("console", "console", false, "Send output to stdout");
        console.setRequired(false);
        options.addOption(console);

        // file output
        Option file = new Option("f", "file", true, "Send output to file");
        file.setRequired(false);
        file.setArgName("path");
        file.setArgs(1);
        options.addOption(file);

        // jasmine connector output
        Option jasmine = new Option("jasmine", "jasmine", true, "Output logged data to jasmine (will disable stdout)");
        jasmine.setRequired(false);
        jasmine.setArgName("jasmineURI");
        jasmine.setArgs(1);
        options.addOption(jasmine);

        // Field separator
        Option separator = new Option("s", "separator", true, "Set the column separator");
        separator.setRequired(false);
        separator.setArgName("separator");
        separator.setArgs(1);
        options.addOption(separator);

        // cmd id
        Option cmdid = new Option("cmdid", "cmdid", true, "Cmd Identifier");
        cmdid.setRequired(false);
        cmdid.setArgName("cmdid");
        cmdid.setArgs(1);
        options.addOption(cmdid);

        // Optional attribute target
        Option target = new Option("target", "target", true, "instances to poll");
        target.setRequired(false);
        target.setOptionalArg(true);
        target.setArgs(Option.UNLIMITED_VALUES);
        target.setArgName("instances");
        options.addOption(target);

        // Optional attribute rate list
        Option rate = new Option("rate", "rate", false, "attributes rate to poll");
        rate.setRequired(false);
        rate.setOptionalArg(true);
        rate.setArgs(Option.UNLIMITED_VALUES);
        rate.setArgName("rate");
        options.addOption(rate);

        // Optional attribute delta list
        Option delta = new Option("delta", "delta", false, "attributes delta to poll");
        delta.setRequired(false);
        delta.setOptionalArg(true);
        delta.setArgs(Option.UNLIMITED_VALUES);
        delta.setArgName("delta");
        options.addOption(delta);

        // Optional attribute slope list
        Option slope = new Option("slope", "slope", false, "attributes slope to poll");
        slope.setRequired(false);
        slope.setOptionalArg(true);
        slope.setArgs(Option.UNLIMITED_VALUES);
        slope.setArgName("slope");
        options.addOption(slope);
    }

    /**
     * @return The ObjectName.
     */
    public String toString() {
        return on.toString();
    }

    /**
     * Initializes data polling and calls {@link Stat#mbeanPoll()}.
     */
    private void process() throws Exception {
        String[] targets = JmxHelper.getJmxTargets(commandLine.getOptionValues("target"));

        if ((targets != null) && (targets.length > 0)) {
            // Construct target contexts
            contexts = new StatContext[targets.length];
            for (int i = 0; i < contexts.length; i++) {
                // If oldValues=true, old attribute values are managed in StatContext
                contexts[i] = new StatContext(oldValues);
                contexts[i].setName(targets[i]);
                contexts[i].setJmxUrl(JmxHelper.getJmxUrl(targets[i]));
                contexts[i].setJmxap(new JmxAp(contexts[i].getJmxUrl(), this.cmdDispatcher));
                try {
                    ObjectName j2eeinstance = ObjectName.getInstance("*:j2eeType=J2EEServer,*");
                    MBeanServerConnection mbscnx = contexts[i].getJmxap().getMBeanServerConnection();
                    Iterator<?> onames = mbscnx.queryNames(j2eeinstance, null).iterator();
                    contexts[i].setServer(null);
                    while (contexts[i].getServer() == null) {
                        ObjectName server = (ObjectName) onames.next();
                        contexts[i].setServer((String) mbscnx.getAttribute(server, "serverName"));
                        contexts[i].setDomain(server.getDomain());
                    }
                } catch (Exception e) {
                    contexts[i].setServer("unknown_server_name");
                    contexts[i].setDomain("unknown_domain");
                    throw e;
                } finally {
                    contexts[i].getJmxap().releaseMBeanServerConnection();
                }
                // Determine attributes to poll in case they must be 'guessed'
                if (aOption && (attsSimple == null || attsSimple.length == 0)) {
                    attsSimple = getAttsToPoll(contexts[i], on);
                }
                logger.info("Target {0} - {1}", contexts[i].getName(), contexts[i].getJmxUrl());
            }
            // TODO: don't set Running if another target failed.
            cmdDispatcher.setRunning();
            mbeanPoll();
        } else {
            logger.error("The target(s) are not defined. Check the URL or the JMX URL property file.");
            throw new Exception("No target defined");
        }
    }

    /**
     * Rounding a double with n elements after the comma.
     * @param a The value to convert.
     * @param n The number of decimal places to retain.
     * @return Value rounded to n decimal places.
     */
    private static double floor(double a, int n) {
        double p = Math.pow(10.0, n);
        return Math.floor((a * p) + 0.5) / p;
    }

    /**
     * Convert ArrayList of String to Array of String
     * @param arrayList
     * @return array
     */
    private static String[] toArray(ArrayList<String> arrayList) {
        int size = arrayList.size();
        String[] array = new String[size];
        for (int i = 0; i < size; i++) {
            array[i] = arrayList.get(i);
        }
        return array;
    }
    /**
     * Statistics contexts.
     */
    private StatContext[] contexts = null;
    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;
    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;
    /**
     * ObjectName to use.
     */
    private ObjectName on = null;
    /**
     * Polling period in seconds.
     */
    private long period = 10;
    /**
     * Refresh period, in seconds. A value of 0 means refresh at each poll
     */
    private long refreshPeriod = 300;
    /**
     * Is content refresheable ?
     */
    private boolean isRefreshable = true;
    /**
     * Graph definitions, null for no graph output.
     */
    private String graphDef = null;
    /**
     * Ident of the cmd
     */
    private String cmdid = null;
    /**
     * Output file path, null for no file output.
     */
    private String outputFilePath = null;
    /**
     * URI of the JASMINe event switch, null for no JASMINe output.
     */
    private String jasmineURI = null;
    /**
     * Output stream.
     */
    private PrintStream pout = null;
    /**
     * If true, the output to console is forced.
     */
    private boolean isConsoleOption = false;
    /**
     * Separator to use.
     */
    private String separator = ";";
    /**
     * Header to use.
     */
    private String header = null;
    /**
     * Command dispatcher instance.
     */
    private CommandDispatcher cmdDispatcher = null;
    /**
     * IOP trouble workarround : Warning counter
     */
    private long warningTrigger = 0;
    /**
     * Attributes
     *
     */
    private boolean aOption = false;
    /**
     * Rate
     * To calculate number of "task" per second
     */
    private boolean rateOption = false;
    /**
     * Delta
     * To calculate the number of "task" per period
     */
    private boolean deltaOption = false;
    /**
     * Slope
     * To calculate the change in the X coordinate divided by the corresponding change in the Y coordinate
     * dX/dY
     */
    private boolean slopeOption = false;

    /**
     * Becomes true if one of rateOption, deltaOption or slopeOption is true.
     * Otherwise is false, that means we don't have to keep old attribute values.
     */
    private boolean oldValues = false;

    /** The attributes name to poll
     */
    private String[] attsSimple = null;

    /** The attributes name for which rate is to be calculated
     */
    private String[] attsRate = null;

    /** The attributes name for which delta is to be calculated
     */
    private String[] attsDelta = null;

    /** The attributes name for which slope is to be calculated
     */
    private String[] attsSlope = null;

    /**
     * Attribute names for which one of the delta, rate or slope is to be calculated.
     */
    private List<String> attsRateDelatSlope = null;

    /**
     * The number of decimal places to retain
     */
    private int numberDecimalPlaces = 16;
}
