/**
 * JASMINe
 * Copyright (C) 2007-2008 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxContext.java 7005 2010-10-25 08:52:43Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.context;

import java.io.IOException;
import java.io.Serializable;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.JmxAp;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
/**
 * Main context for jmx environment.
 * @author Guillaume Renault
 *
 */
public abstract class JmxContext implements Serializable {

    protected Log logger = LogFactory.getLog(this.getClass());
    /**
     *
     */
    private static final long serialVersionUID = -9159123017384356614L;

    /**
     * Name set for the target in the jmxurls.properties file.
     */
    protected String name = null;

    /**
     * Server name.
     */
    protected String server = null;

    /**
     * Server domain.
     */
    protected String domain = null;

    /**
     * JMX URL to connect to.
     */
    protected String jmxUrl = null;

    /**
     * JMX Access Point.
     */
    protected JmxAp jmxap = null;

    /**
     * Targeted names.
     */
    protected Set<ObjectName> onames = null;

    /**
     * cmd id
     */
    protected String cmdid = null;

    /**
     *
     * @return the target name
     */
    public String getName() {
        return name;
    }

    /**
     * @param name target name
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @return the server name
     */
    public String getServer() {
        return server;
    }

    /**
     *
     * @param server the server name
     */
    public void setServer(String server) {
        this.server = server;
    }

    public String getDomain() {
        return domain;
    }

    public void setDomain(String domain) {
        this.domain = domain;
    }

    public String getJmxUrl() {
        return jmxUrl;
    }

    public void setJmxUrl(String jmxUrl) {
        this.jmxUrl = jmxUrl;
    }

    public JmxAp getJmxap() {
        return jmxap;
    }

    public void setJmxap(JmxAp jmxap) {
        this.jmxap = jmxap;
    }

    public String getCmdId() {
        return cmdid;
    }

    public void setCmdId(String c) {
        this.cmdid = c;
    }

    public Set<ObjectName> getOnames() {
        return onames;
    }

    public void setOnames(Set<ObjectName> onames) {
        this.onames = onames;
    }

    public MBeanServerConnection getMBeanServerConnection() {
        return jmxap.getMBeanServerConnection();
    }

    /**
     * Connect to the target and get the list of MBeans corresponding to a given ObjectName pattern.
     * Called on poll methods.
     * @param onPattern  the given ObjectName pattern.
     * @throws Exception could not get the MBean's list at this moment.
     */
    public void updateOnames(String onPattern) throws Exception {
        Set<ObjectName> onSet = getObjectNames(onPattern);
        setOnames(onSet);
    }

    /**
     * Connect to the target and get the list of MBeans corresponding to a given ObjectName pattern.
     * Called on poll methods.
     * @param onPattern  the given ObjectName pattern.
     * @throws Exception could not get the MBean's list at this moment.
     */
    public void updateOnames(ObjectName onPattern) throws Exception {
        Set<ObjectName> onSet = getObjectNames(onPattern);
        setOnames(onSet);
    }

    /**
     * Return a set of ObjectNames corresponding to a given ON pattern. The search is done in the current JMXContext.
     * @param onPattern the given ON pattern
     * @return the ObjectName set that was found
     * @throws Exception
     */
    public Set<ObjectName> getObjectNames(String onPattern) throws Exception {
        MBeanServerConnection cnx = getMBeanServerConnection();
        Set<ObjectName> onSet = null;
        if (onPattern != null) {
            ObjectName on = null;
            try {
                on = ObjectName.getInstance(onPattern);
            } catch (javax.management.MalformedObjectNameException me) {
                logger.error("MBean discovery: failed probe on target {0} because of erroneous ObjectName: {1}, caught exception with message: {2}", getName(), onPattern, me.getMessage());
                throw me;
            }
            onSet = getObjectNames(on);
        } else {
            return new HashSet<ObjectName>();
        }
        return onSet;
    }

    /**
     * Return a set of ObjectNames corresponding to a given ON pattern. The search is done in the current JMXContext.
     * @param onPattern the given ON pattern
     * @return the ObjectName set that was found
     * @throws Exception
     */
    public Set<ObjectName> getObjectNames(ObjectName onPattern) throws Exception {
        MBeanServerConnection cnx = getMBeanServerConnection();
        Set<ObjectName> onSet = null;
        if (onPattern != null) {
            try {
                onSet = cnx.queryNames(onPattern, null);
                if (onSet.isEmpty()) {
                    successfull = false;
                } else {
                    if (successfull == false) {
                        // first succeded update
                        logger.info("MBean discovery succesful for target {0} and pattern {1}", getName(), onPattern.toString());
                        setUpManagentInfos();
                        successfull = true;
                    } else {
                        // normal update
                        logger.debug("MBean discovery succesful for target {0} and pattern {1}", getName(), onPattern.toString());
                    }
                }
                if (interrupted && isStarted()) {
                    interrupted = false;
                    logger.info("MBean discovery succesful for target {0} and pattern {1}", getName(), onPattern.toString());
                    // update management info
                    setUpManagentInfos();
                }
            } catch (IOException e) {
                if (!interrupted) {
                    resetInterruptionTime();
                    interrupted = true;
                    // Log message at the first occurence of the IOException
                    logger.error("MBean discovery: failed probe on target {0}, caught exception with message: {1}", getName(), e.getMessage());
                } else {
                    boolean deltaElapsed = isDeltaTimeElapsed();
                    if (deltaElapsed) {
                        // Log message after the delta (LOG_PERIOD) elapsed
                        logger.error("MBean discovery: failed probe on target {0}, caught exception with message: {1}", getName(), e.getMessage());
                        resetInterruptionTime();
                    }
                }
                throw e;
            }
        } else {
            return new HashSet<ObjectName>();
        }
        return onSet;
    }

    public boolean isStarted() {
        return started;
    }

    public void setStarted(boolean started) {
        this.started = started;
    }

    /**
     * Connect to the target in order to get management information allowing to
     * identify the target :
     * - management domain name
     * - server name
     *  Here we suppose that the target is a J2eeServer that exposes standard MBeans like JVM MBean.
     */
    public void setUpManagentInfos() {
        MBeanServerConnection cnx = getMBeanServerConnection();
        ObjectName jvmOnPattern = null;
        try {
            jvmOnPattern = ObjectName.getInstance("*:j2eeType=JVM,*");
        } catch (MalformedObjectNameException e) {
            // Should never occur
            e.printStackTrace();
            return;
        }
        Set jvmOns = null;
        try {
            jvmOns = cnx.queryNames(jvmOnPattern, null);
        } catch (IOException ioe) {
            // At this point, the IOException was analysed and logged by the JmxCnxWrapper
            // What happens if the cnx id provided by a IJMXConnectionFactory ??
            return;
        }
        if (jvmOns.isEmpty()) {
            // No jvminstance found, can't set up informations
            return;
        }
        ObjectName jvmOn = (ObjectName) jvmOns.iterator().next();
        setServer(jvmOn.getKeyProperty("J2EEServer"));
        setDomain(jvmOn.getDomain());
        logger.debug("JmxContext management info was updated with server name: {0}, and domain name : {1}", getServer(), getDomain());
    }

    /**
     * Sampler started ?
     */
    private boolean started = false;

    private boolean successfull = false;

    /**
     * Pass true if a IOException is thrown when trying to update the target ObjectNames
     */
    private boolean interrupted = false;

    private long interruptionTime;
    private void resetInterruptionTime() {
        interruptionTime = System.currentTimeMillis();
    }
    private boolean isDeltaTimeElapsed() {
        long deltaTime = System.currentTimeMillis() - interruptionTime;
        // deltaTimime > 10 min (600 sec)
        if (deltaTime > LOG_PERIOD) {
            return true;
        }
        return false;
    }
    // In millis
    private static final long MINUTE = 60000;
    // Fix log period to 10 minutes
    private static final long LOG_PERIOD = 10 * MINUTE;
}
