/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Grapher.java 6434 2010-05-06 07:49:17Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.graph;

import java.io.IOException;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.ow2.jasmine.monitoring.mbeancmd.Outer;
import org.ow2.jasmine.monitoring.mbeancmd.graph.conf.Configuration;
import org.ow2.jasmine.monitoring.mbeancmd.graph.conf.Configurator;
import org.ow2.jasmine.monitoring.mbeancmd.graph.conf.GraphConfig;

/**
 * Draws a graph based on data coming from an Outer pipe.
 */
public class Grapher {
    /**
     * Tests the Grapher implementation.
     *
     * @param args  Arguments to use when creating the graph.
     *
     * @throws IOException  If there's a problem reading the graph definition
     *                      XML or creating a pipe.
     */
    public static void main(final String[] args) throws IOException {
        PipedOutputStream src = new PipedOutputStream();
        PrintWriter wsrc = new PrintWriter(src);

        Grapher g = new Grapher(src, "src/xml/graphtest2.xml");

        g.start();

        injectData(wsrc, g);
        src.close();

        try {
            Thread.sleep(5000);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    /**
     * Injects data. Used by the testing application, @see Grapher#main
     *
     * @param pr  PrintWriter to the graph.
     * @param g   Grapher.
     */
    private static void injectData(final PrintWriter pr, final Grapher g) {
        String[] names = new String[] {
                    "fwa:type=aaa,name=n00", "fwa:type=bbb,name=n01",
                    "fwa:type=aaa,name=n02", "fwa:type=aaa,name=n03",
                    "fwa:type=bbb,name=n04", "fwa:type=bbb,name=n05",
                    "fwa:type=aaa,name=n06", "fwa:type=aaa,name=n07",
                    "fwa:type=aaa,name=n08", "fwa:type=bbb,name=n09",
                    "fwa:type=bbb,name=n10"
                };

        String sep = ";";
        SimpleDateFormat df = g.config.getDateFormat();
        pr.println("time" + sep + "a" + sep + "b" + sep + "c" + sep + "d" + sep + "e" + sep + "f" + sep + "name");
        for (int i = 1; i <= 60; i++) {
            long t = System.currentTimeMillis() + i;
            pr.print(
                        (df == null ? Long.toString(t) : df.format(new Date(t))) + sep
                      + (100 + i) + sep + (200 + i) + sep + (300 + i) + sep
                      + (400 + i) + sep + (500 + i) + sep + (600 + i) + sep
                      + names[(i - 1)%10] + (i == 100 ? "" : "\n"));
            pr.flush();
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Creates a new Grapher.
     *
     * @param in           Input pipe.
     * @param definitions  Path to XML file defining the graph configuration.
     *
     * @throws IOException  If ther's a problem reading definitions.
     */
    public Grapher(final PipedOutputStream in, final String definitions) throws IOException {
        config = new Configurator();
        config.loadConfig(definitions);

        // Create and chain series, each Serie is associated to an Outer:
        //      - The Outers are chained so that they forward the
        //        main data flow from the one to the next
        //      - An outer object is responsible for filtering and reformatting
        //        the data read from the main flow and outputting this filtered
        //        stream onto the target.
        //      - The target of an Outer is the input of the associated Serie
        //      - The sink of the Outer is chained to the source of the next
        //        Outer in the chain
        String[] sids = config.getSerieIds();
        series = new Serie[sids.length];
        outers = new Outer[sids.length + 1];
        for (int i = 0; i < sids.length; i++) {
            series[i] = new Serie(config.getSerieConfig(sids[i]));
            outers[i] = new Outer(
                            (i == 0 ? in : outers[i - 1].getSink()),
                            new PrintStream(series[i].getSource()));
            outers[i].setSink(new PipedOutputStream());
            String[] format = new String[] {
                    series[i].getConfig().getXAxis(),
                    series[i].getConfig().getYAxis() };
            outers[i].setFormatter(format);
            outers[i].setRegexp(series[i].getConfig().getMbeanPattern());
        }

        outers[outers.length - 1] = new Outer(outers[outers.length - 2].getSink(), (PrintStream) null);

        // Set the separators into outers
        Outer.setSeparator(config.getSeparator());

        // Create graphs and link them to their series.
        String[] gids = config.getGraphIds();
        Graph[] graphs = new Graph[gids.length ];
        for (int i = 0; i < graphs.length; i++) {
            if (gids[i] == null) {
                // Some errors were commited in the graph definition !
                continue;
            }
            GraphConfig gcf = config.getGraphConfig(gids[i]);
            graphs[i] = new Graph(gcf);
            String[] gseries = gcf.getSerieIds();
            for (int j = 0; j < gseries.length; j++) {
                for (int k = 0; k < series.length; k++) {
                    if (gseries[j].equals(series[k].getConfig().getId())) {
                        series[k].addSerieListener(graphs[i]);
                    }
                }
            }
        }
    }

    /**
     * Starts the Series and then the Outers.
     */
    public void start() {
        for (int i = 0; i < series.length; i ++) {
            new Thread(series[i]).start();
        }

        for (int i = outers.length - 1; i >= 0; i--) {
            new Thread(outers[i]).start();
        }
    }

    /**
     * Grapher configuration.
     */
    private Configuration config = null;

    /**
     * Series.
     */
    private Serie[] series = null;

    /**
     * Outers that feed the series.
     */
    private Outer[] outers = null;
}
