/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Serie.java 7076 2010-11-08 15:52:48Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.graph;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.ow2.jasmine.monitoring.mbeancmd.graph.conf.Configuration;
import org.ow2.jasmine.monitoring.mbeancmd.graph.conf.Constants;
import org.ow2.jasmine.monitoring.mbeancmd.graph.conf.SerieConfig;

/**
 * Represents a series of data.
 */
public class Serie implements Runnable {
    /**
     * Creates the series based on a configuration.
     *
     * @param conf  Configuration.
     *
     * @throws IOException  If the associated pipe cannot be created.
     */
    public Serie(final SerieConfig conf) throws IOException {
        this.conf = conf;
        Configuration cf = conf.getParent();
        if (cf == null) {
            logger.severe("SerieConfig has no parent");
            return;
        }
        SimpleDateFormat parentDateFormat = cf.getDateFormat();
        if (parentDateFormat == null) {
            this.dateFormat = null;
        } else {
            this.dateFormat = (SimpleDateFormat) parentDateFormat.clone();
        }
        source = new PipedOutputStream();
        in = new LineNumberReader(new InputStreamReader(new PipedInputStream(source)));
        if (logger.isLoggable(Level.INFO)) {
            logger.info("New serie: " + conf.getId());
        }
    }

    /**
     * @return  The source pipe.
     */
    public PipedOutputStream getSource() {
        return source;
    }

    /**
     * @return  The Series configuration.
     */
    public SerieConfig getConfig() {
        return conf;
    }

    /**
     * Adds a series listener.
     *
     * @param listener  Listener to add.
     */
    public void addSerieListener(final SerieListener listener) {
        listeners.add(listener);
    }

    /**
     * Implementation of inherited method.
     *
     * @see Thread#run()
     */
    public void run() {
        if (logger.isLoggable(Level.FINEST)) {
            logger.finest("Serie: " + conf.getTitle()+ " Starting...");
        }

        String sep = conf.getParent().getSeparator();
        boolean goOn = true;
        boolean isHeader = true;
        while (goOn) {
            try {
                String ln = in.readLine();
                if (ln == null) {
                    goOn = false;
                    continue;
                }

                // ignore the first row (header)
                if (isHeader) {
                    isHeader = false;
                } else {
                    String[] s = ln.split(sep);
                    if (s.length != 2) {
                        if (logger.isLoggable(Level.SEVERE)) {
                            logger.severe("Format error on series '" + this.getConfig().getId()
                                    + "' : 2 values where expected, but read [" + ln + "]");
                        }
                        throw new IllegalStateException("No value to be represented.");
                    }
                    Iterator<SerieListener> it = listeners.iterator();
                    while (it.hasNext()) {
                        SerieListener sl = it.next();
                        if ((conf.getXType() == Constants.TIME) && (conf.getYType() == Constants.DOUBLE)) {
                            sl.update(getConfig().getId(), parseTime(s[0]), Double.parseDouble(s[1]));
                        } else if ((conf.getXType() == Constants.TIME) && (conf.getYType() == Constants.LONG)){
                            sl.update(getConfig().getId(), parseTime(s[0]), Long.parseLong(s[1]));
                        } else if ((conf.getXType() == Constants.LONG) && (conf.getYType() == Constants.DOUBLE)) {
                            sl.update(getConfig().getId(), Long.parseLong(s[0]), Double.parseDouble(s[1]));
                        } else if ((conf.getXType() == Constants.LONG) && (conf.getYType() == Constants.LONG)){
                            sl.update(getConfig().getId(), Long.parseLong(s[0]), Long.parseLong(s[1]));
                        } else {
                            throw new IOException("Unsupported axis type : " + conf.getXType() + ", " + conf.getYType());
                        }
                    }
                }
            } catch (Exception e) {
                logger.log(Level.INFO, "Exception in serie: " + conf.getTitle(), e);
                goOn = false;
            }
        }
        if (logger.isLoggable(Level.INFO)) {
            logger.info("Serie: " + conf.getTitle()+ " complete.");
        }
    }

    /**
     * Parses a time string.
     *
     * @param time  Time string.
     *
     * @return  Parsed string, origin is 01/01/1970.
     *
     * @throws ParseException  If format doesn't match input.
     */
    private long parseTime(final String time) throws ParseException {
        if (this.dateFormat == null) {
            return Long.parseLong(time);
        } else {

            return this.dateFormat.parse(time).getTime();
        }
    }

    /**
     * The Series' configuration.
     */
    private SerieConfig conf = null;

    /**
     * The Serie's dateFormat provided by the parent.
     */
    private SimpleDateFormat dateFormat = null;

    /**
     * Source stream.
     */
    private PipedOutputStream source = null;

    /**
     * Reader associated with the stream.
     */
    private LineNumberReader in = null;

    /**
     * List of listeners.
     */
    private List<SerieListener> listeners = new LinkedList<SerieListener>();

    /**
     * Logger.
     */
    private Logger logger = Logger.getLogger(this.getClass().getPackage().getName());
}
