/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JoramQSampler.java 7005 2010-10-25 08:52:43Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.IOException;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;

/**
 * Sampler for JORAM Queues.
 */
public class JoramQSampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public JoramQSampler(final SamplerContext context) {
        super(context);
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOnPattern() {
        return "joramClient:type=Queue,*";
    }

    public ObjectName getJoramAdapter() {
        try {
            return ObjectName.getInstance("joramClient:type=JoramAdapter");
        } catch (MalformedObjectNameException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return null;
        }
    }

    public String getDestinationType() {
        return "Queue";
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new XSampleData();
    }

    /**
     * Build the list of monitored queues
     *
     */
    private void refreshQueueON() throws Exception {

        if (!isRefreshable() && isHasFailed()) {
            // It is not refreshable as it worked, then failed.
            // After a restart, try to refresh
            setRefreshable(true);
            setHasFailed(false);
        }

        if (isRefreshable()) {
            // Do refreshing
            if (isHasFailed()) {
                // In order not to let hasFailed sticked to true
                setHasFailed(false);
            }
            ObjectName[] ons = null;
            LinkedList<ObjectName> lon = new LinkedList<ObjectName>();

            // Destinations in the local JORAM server.
            ObjectName[] onDest = queryON("*:type=Destination,*");

            // JoramClient
            //ObjectName[] joramClients = queryON(getOnPattern());
            LinkedList<ObjectName> joramClientsList = new LinkedList<ObjectName>();
            ObjectName joramAdaperOn = getJoramAdapter();
            if (joramAdaperOn != null) {
                MBeanServerConnection cnx = getMBeanServerConnection();
                short serverId = (Short) cnx.getAttribute(joramAdaperOn, "ServerId");
                Short[] params = new Short[1];
                params[0] = serverId;
                String[] signature = {"short"};
                String[] destinationOns = (String[]) cnx.invoke(joramAdaperOn, "getDestinations", params, signature);
                for (int i = 0; i < destinationOns.length; i++) {
                    ObjectName destinationOn = null;
                    try {
                        destinationOn = ObjectName.getInstance(destinationOns[i]);
                    } catch (MalformedObjectNameException e) {
                        // TODO Auto-generated catch block
                        e.printStackTrace();
                    }
                    if (destinationOn != null && destinationOn.getKeyProperty("type").equals(getDestinationType())) {
                        joramClientsList.add(destinationOn);
                    }
                }
            }

            // select all local queues, that is queues with an associated
            // Destination MBean.
            if (onDest != null) {
                if (joramClientsList != null) {
                    for (ObjectName destON : onDest) {
                        for (ObjectName queueON : joramClientsList) {
                            String destName = destON.getKeyProperty("name");
                            String queueName = queueON.getKeyProperty("name");
                            if (queueName.startsWith(destName)) {
                                lon.add(destON);
                            }
                        }
                    }
                }
            }
            if (!lon.isEmpty()) {
                ons = lon.toArray(new ObjectName[lon.size()]);
            }
            onQueues = ons;

            // TODO : changer par une mise � jour en N minutes
            if ((onQueues != null) && !isHasFailed()) {
                setRefreshable(false);
                setHasFailed(false);
                refreshDeadLine = System.currentTimeMillis() + refreshTimeout;
            }
        } else {
            if (System.currentTimeMillis() >= refreshDeadLine) {
                setRefreshable(true);
            }
        }
    }


    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#poll(SampleData)
     * @see EntitySampler#pollEntities(XSampleData)
     */
    protected SampleData poll(final SampleData data) {
        pollJoramQueues((XSampleData) data);
        data.setValid(true);
        return data;
    }

    /**
     * Polls JCA Connection Factory providers for data, will call
     * {@link JoramQSampler#pollJoramQueue(ObjectName, JoramQData)} for each
     * object name.
     *
     * @param data
     *            ServerData to add polled information to.
     */
    private void pollJoramQueues(final XSampleData data) {

        try {
            refreshQueueON();
        } catch (Exception e) {
            // Problem with ObjectNames
            return;
        }

        if (onQueues != null) {

            // for (int i = 0; i < onQueues.length; i++) {
            for (ObjectName queue : onQueues) {

                JoramQData qdata = new JoramQData();
                qdata.setObjectName(queue);
                qdata.setSampleTime(System.currentTimeMillis());
                pollJoramQueue(queue, qdata);

                if (qdata.isValid()) {
                    data.put(queue, qdata);
                }
            }
        }
    }

    /**
     * Polls one JCA Connection Factory provider for data.
     *
     * @param on
     *            Name of the provider.
     * @param data
     *            ServerData to add polled information to.
     */

    /*
     * OLD style private void pollJoramQueue(final ObjectName on, final
     * JoramQData data) { AttributeList al = null; MBeanServerConnection cnx =
     * null;
     *
     * Attribute att; try { cnx = getMBeanServerConnection(); al =
     * cnx.getAttributes(on, attIds); releaseMbeanServerConnection(); cnx =
     * null; Iterator it = al.iterator(); while (it.hasNext()) { att =
     * (Attribute) it.next(); if ("Statistic".equals(att.getName())) { // this
     * attribute is a HashTable with the following entries: - creationDate, -
     * nbMsgsReceiveSinceCreation, - nbMsgsSendToDMQSinceCreation, -
     * nbMsgsDeliverSinceCreation // Hashtable ht = (Hashtable) att.getValue(); //
     * retrieve creationDate Attribute a1 = new Attribute("creationDate",
     * ht.get("creationDate")); data.setAttribute(a1); // retrieve
     * nbMsgsReceiveSinceCreation (Long) a1 = new
     * Attribute("nbMsgsReceiveSinceCreation",
     * ht.get("nbMsgsReceiveSinceCreation")); data.setAttribute(a1); // retrieve
     * nbMsgsSendToDMQSinceCreation a1 = new
     * Attribute("nbMsgsSendToDMQSinceCreation",
     * ht.get("nbMsgsSendToDMQSinceCreation")); data.setAttribute(a1); //
     * retrieve nbMsgsDeliverSinceCreation a1 = new
     * Attribute("nbMsgsDeliverSinceCreation",
     * ht.get("nbMsgsDeliverSinceCreation")); data.setAttribute(a1); } else {
     * data.setAttribute(att); } } data.setServerInfo(getName(), getServer(),
     * getDomain()); data.setValid(true); } catch (InstanceNotFoundException e) {
     * e.printStackTrace(System.err); } catch (ReflectionException e) {
     * e.printStackTrace(System.err); } catch (IOException e) {
     * e.printStackTrace(System.err); } }
     *
     */

    private void pollJoramQueue(final ObjectName on, final JoramQData data) {
        AttributeList al = null;
        MBeanServerConnection cnx = null;

        /*
         * attIds : "NbMsgsDeliverSinceCreation", "NbMsgsReceiveSinceCreation",
         * "NbMsgsSentToDMQSinceCreation", "PendingMessageCount",
         * "WaitingRequestCount", "CreationTimeInMillis ", "NbMaxMsg" };
         */
        Attribute att;
        try {
            cnx = getMBeanServerConnection();
            al = cnx.getAttributes(on, attIds);
            releaseMbeanServerConnection();
            cnx = null;
            Iterator it = al.iterator();

            /*
             * If there are some attributes, add the AdminName attribute
             */
            if (it.hasNext()) {
                data.setAttribute(new Attribute("AdminName", on.getKeyProperty("name")));
            }

            while (it.hasNext()) {
                att = (Attribute) it.next();
                data.setAttribute(att);
            }
            data.setServerInfo(getName(), getJmxUrl(), getServer(), getDomain());
            data.setCmdId(getCmdId());
            data.setValid(true);
        } catch (InstanceNotFoundException e) {
            System.err.println("Error on mbean " + on + " for " + context.getName() + " : " + e.getMessage());
            setRefreshable(true);
            setHasFailed(true);
        } catch (ReflectionException e) {
            System.err.println("Error on mbean " + on + " for " + context.getName() + " : " + e.getMessage());
            setRefreshable(true);
            setHasFailed(true);
        } catch (IOException e) {
            System.err.println("Error on mbean " + on + " for " + context.getName() + " : " + e.getMessage());
            setRefreshable(true);
            setHasFailed(true);
        }
    }

    /*
     * This method is dropped, because a JORAM bug, when accessing a
     * joramClient:type=queue,* Where the queue is heavily loaded, accessing the
     * Statistic attribue does not return ! If a different attribute is accessed
     * (e.g: AdminName, it returns immediately the right value. test made with
     * MBeanCmd.
     */
    private void pollJoramQueue_DEPRECATED(final ObjectName on, final JoramQData data) {
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        // on heavy load, JORAM returns a null 'Statistic' attribute !
        boolean hasStatistics = false;

        Attribute att;
        try {
            cnx = getMBeanServerConnection();
            al = cnx.getAttributes(on, attIds);
            releaseMbeanServerConnection();
            cnx = null;
            Iterator it = al.iterator();
            while (it.hasNext()) {
                att = (Attribute) it.next();
                if ("Statistic".equals(att.getName())) {
                    /*
                     * this attribute is a HashTable with the following entries: -
                     * creationDate, - nbMsgsReceiveSinceCreation, -
                     * nbMsgsSendToDMQSinceCreation, -
                     * nbMsgsDeliverSinceCreation
                     */
                    Hashtable ht = (Hashtable) att.getValue();

                    // retrieve creationDate
                    Attribute a1 = new Attribute("creationDate", ht.get("creationDate"));
                    data.setAttribute(a1);

                    // retrieve nbMsgsReceiveSinceCreation (Long)
                    a1 = new Attribute("nbMsgsReceiveSinceCreation", ht.get("nbMsgsReceiveSinceCreation"));
                    data.setAttribute(a1);

                    // retrieve nbMsgsSendToDMQSinceCreation
                    a1 = new Attribute("nbMsgsSendToDMQSinceCreation", ht.get("nbMsgsSendToDMQSinceCreation"));
                    data.setAttribute(a1);

                    // retrieve nbMsgsDeliverSinceCreation
                    a1 = new Attribute("nbMsgsDeliverSinceCreation", ht.get("nbMsgsDeliverSinceCreation"));
                    data.setAttribute(a1);

                    hasStatistics = true;
                } else {
                    data.setAttribute(att);
                }
            }
            data.setServerInfo(getName(), getJmxUrl(), getServer(), getDomain());
            data.setCmdId(getCmdId());
            if (hasStatistics) {
                data.setValid(true);
            } else {
                // invalidate the sample if 'Statistic' attribute is null
                data.setValid(false);
                System.err.println("Error on mbean " + on + " for " + context.getName() + " : no Statistic attribute ");
            }
        } catch (InstanceNotFoundException e) {
            System.err.println("Error on mbean " + on + " for " + context.getName() + " : " + e.getMessage());
            setRefreshable(true);
            setHasFailed(true);
        } catch (ReflectionException e) {
            System.err.println("Error on mbean " + on + " for " + context.getName() + " : " + e.getMessage());
            setRefreshable(true);
            setHasFailed(true);
        } catch (IOException e) {
            System.err.println("Error on mbean " + on + " for " + context.getName() + " : " + e.getMessage());
            setRefreshable(true);
            setHasFailed(true);
        }
    }

    /**
     * List of polled attributes. Note that 'Statistic' is a Hastable which is
     * expected to hold the following entries: - creationDate -
     * nbMsgsReceiveSinceCreation - nbMsgsSendToDMQSinceCreation -
     * nbMsgsDeliverSinceCreation
     */

    private static final String[] attIds = { "NbMsgsDeliverSinceCreation", "NbMsgsReceiveSinceCreation",
            "NbMsgsSentToDMQSinceCreation", "PendingMessageCount", "WaitingRequestCount", "CreationTimeInMillis", "NbMaxMsg" };

    // Commented out due to a JORAM bug
    // private static final String[] attIds = { "AdminName", "Type",
    // "PendingMessages", "NbMaxMsg", "PendingRequests", "Statistic","Threshold"
    // };

    /**
     * Queues
     */
    private ObjectName[] onQueues = null;

    /**
     * Refresh MBean lists
     */
    private boolean refreshable = true;

    public boolean isRefreshable() {
        return refreshable;
    }

    public void setRefreshable(boolean refreshable) {
        this.refreshable = refreshable;
    }

    private long refreshDeadLine = 0L;

    private long refreshTimeout = 300000L;
}
