/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: SampleData.java 7232 2010-12-01 11:02:13Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.PrintStream;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import java.util.TreeMap;

import javax.management.Attribute;
import javax.management.ObjectName;

/**
 * Base class for all data classes.
 */
public abstract class SampleData {
    /**
     * Compute based on a previous measurement, will fill all fields that are
     * marked as being "computed" and don't have a public setter.
     *
     * @param prev  Previous measurement.
     */
    public abstract void compute(final SampleData prev) throws Exception;

    /**
     * @param prev data obtained at the previous sampling
     * @return the time elapsed between the 2 samplings in seconds
     */
    protected double computeSamplePeriod(final SampleData prev) {
        return (this.getSampleTime() - prev.getSampleTime()) / 1000.0;
    }

    /**
     * Returns the full printout header for this data. The child class should
     * not implement this method, it should only implement
     * {@link SampleData#getInnerPrintHeader()} instead.
     *
     * @return  The printout header format for this data.
     */
    public final String getPrintHeader() {
        String header = "time" + getSeparator() + "date" + getSeparator()
                      + "sname" + getSeparator() + "server" + getSeparator()
                      + "domain" + getSeparator() + "mbean" + getSeparator()
                      + "cmdid" + getSeparator() + getInnerPrintHeader();

        return header;
    }

    /**
     * @return  The inner (specialized) part of the header for this data type.
     */
    protected abstract String getInnerPrintHeader();

    /**
     * Prints out all data in attributes based on the print header.
     *
     * @see SampleData#getPrintHeader()
     * @see SampleData#getInnerPrintData()
     *
     * @param out  Stream to print data values into.
     */
    public void printData(final PrintStream out) {
        Date d = new Date(getSampleTime());
        String result = getSampleTime() + getSeparator() + simpleDateFormat.format(d) + getSeparator()
                      + getJmxURL() + getSeparator() + getServer() + getSeparator()
                      + getDomain() + getSeparator() + objectName + getSeparator()
                      + getCmdId() + getSeparator()
                      + getInnerPrintData();

        out.println(result);
    }


    /**
     * @return  The inner (specialized) part of the data for this data type.
     */
    protected abstract String getInnerPrintData();

    public String getCmdId() {
        return cmdid;
    }

    public void setCmdId(String cmdid) {
        this.cmdid = cmdid;
    }

    public String getObjectName() {
        return objectName;
    }

    /**
     * @param on  Object name to set.
     */
    protected final void setObjectName(final ObjectName on) {
        //this.objectName = on.getKeyProperty("name");
        if (on != null) {
            this.objectName = on.getCanonicalName();
        }
    }

    /**
     * @return  Time of sampling, origin is 01/01/1970.
     */
    public final long getSampleTime() {
        return sampleTime;
    }

    /**
     * @param t  Time of sampling, origin is 01/01/1970.
     */
    protected final void setSampleTime(final long t) {
        sampleTime = t;
    }

    /**
     * @param att  Attribute to add in the list of data attributes.
     */
    protected final void setAttribute(final Attribute att) {
        atts.put(att.getName(), att.getValue());
    }

    /**
     * @param name  Name of the attribute to get.
     *
     * @return  Attribute corresponding to name, null if none match.
     */
    protected final Object getAttribute(final String name) {
        return atts.get(name);
    }

    /**
     * @param name  Name of the attribute to get.
     *
     * @return  Attribute corresponding to name, transformed into an integer.
     */
    protected final Integer getAttributeAsInt(final String name) {
        if (atts.get(name) == null) {
            return null;
        }
        return (Integer) atts.get(name);
    }

    /**
     * This method provides support for Null data elements.
     * The current implementation is to add only separator to the printData.
     * If we have the String a; in printData, the result will be a;;
     * @param printData the current printData
     * @param dataElement the data to be added
     * @return the new printData containing the concatenated element
     */
    protected void concatDataElement (StringBuffer printData, final Object dataElement, boolean end) {
        if (dataElement != null) {
            printData.append(dataElement);
        }
        if (!end) {
            printData.append(getSeparator());
        }
    }

    /**
     * @param name  Name of the attribute to get.
     *
     * @return  Attribute corresponding to name, transformed into a long.
     */
    protected final Long getAttributeAsLong(final String name) {
        if (atts.get(name) == null) {
            return null;
        }
        return (Long) atts.get(name);
    }

    /**
     * @param name  Name of the attribute to get.
     *
     * @return  Attribute corresponding to name, transformed into a double.
     */
    protected final Double getAttributeAsDouble(final String name) {
        if (atts.get(name) == null) {
            return null;
        }
        return ((Double) atts.get(name)).doubleValue();
    }

    /**
     * @return  Current separator.
     */
    protected static final String getSeparator() {
        return separator;
    }

    /**
     * @param s  Separator to set.
     */
    public static final void setSeparator(final String s) {
        separator = s;
    }

    /**
     * @return  Name set for the server in the jmxurls.properties file.
     */
    public final String getName() {
        return this.name;
    }

    /**
     * @return The JMX URL of the monitored server.
     */
    public final String getJmxURL() {
        return this.jmxURL;
    }

    /**
     * @return  Name of the server.
     */
    public final String getServer() {
        return this.server;
    }

    /**
     * @return  Name of the server's domain.
     */
    public final String getDomain() {
        return this.domain;
    }

    /**
     * @param name    Name set for the server in the jmxurls.properties file.
     * @param jmxURL  JMX URL of the monitored server.
     * @param server  Name of the server.
     * @param domain  Domain name of the server.
     */
    protected final void setServerInfo(final String name, final String jmxURL, final String server, final String domain) {
        this.name = name;
        this.jmxURL = jmxURL;
        this.server = server;
        this.domain = domain;
    }

    /**
     * @param status  true to mark the data as valid, false otherwise.
     */
    protected final void setValid(final boolean status) {
        this.status = status;
    }

    /**
     * @return true if data valid, false otherwise.
     */
    protected final boolean isValid() {
        return status;
    }

    /**
     * Time at which data has been sampled.
     */
    private long sampleTime = 0;

    /**
     * Name set for the server in the jmxurls.properties file.
     */
    private String name = null;

    /**
     * JMX URL of the monitored server
     */
    private String jmxURL = null;

    /**
     * Name of the server.
     */
    private String server = null;

    /**
     * Domain name of the server.
     */
    private String domain = null;

    /**
     * Object name.
     */
    private String objectName = null;

    /**
     * Cmd Id
     */
    private String cmdid = null;

    /**
     * Is data valid ?
     */
    private boolean status = false;

    /**
     * Decimal formatter.
     */
    protected static final DecimalFormat decimalFormat = (DecimalFormat) DecimalFormat.getInstance();

    /**
     * Date formatter.
     */
    private static SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

    /**
     * Separator.
     */
    private static String separator = ";";

    /**
     * The attributes and their values. A value can be null.
     */
    private Map<String, Object> atts = new TreeMap<String, Object>();

    /**
     * Format decimal parsing.
     */
    static {
        DecimalFormatSymbols sym = new DecimalFormatSymbols();
        sym.setDecimalSeparator('.');
        decimalFormat.setDecimalFormatSymbols(sym);
        decimalFormat.applyPattern("#.#");
    }

    public Map<String, ?> getValueOfAttributes() {
        throw new UnsupportedOperationException();
    }
}
