/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Sampler.java 7005 2010-10-25 08:52:43Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.PrintStream;
import java.util.Set;

import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Base class for all sampler classes.
 */
public abstract class Sampler {

    /**
     * Logger.
     */
    protected Log logger = LogFactory.getLog(this.getClass());

    /**
     * Creates the Sampler using a JMX URL.
     *
     * @param urlString
     *            JMX URL.
     */
    public Sampler(final SamplerContext context) {
        this.context = context;
    }

    /**
     * @return Connection to MBean server. Will return existing instance if
     *         already connected, and try to connect if no connection exists.
     *         Returns null on failure.
     */
    protected final MBeanServerConnection getMBeanServerConnection() {
        return this.context.getJmxap().getMBeanServerConnection();
    }

    protected void releaseMbeanServerConnection() {
        this.context.getJmxap().releaseMBeanServerConnection();
    }

    /**
     * Gets the list of object names that match a given object name prototype,
     * and set the OnjectName list in the SamplerContext (onames).
     *
     * @param onProto
     *            Object name filtering pattern to use.
     *
     * @return All objects that match onProto. Null if no objects found.
     */
    protected final ObjectName[] queryON(final String onProto) throws Exception {
        Set<ObjectName> ons = context.getObjectNames(onProto);
        if (ons != null) {
            return (ObjectName[]) ons.toArray(new ObjectName[ons.size()]);
        }
        return null;
    }

    /**
     * @return Default Object name filtering pattern for this sampler. If null,
     *         we'll consider that the Sampler doesn't have one default object
     *         name filter.
     */
    protected abstract String getDefaultOnPattern();

    /**
     * @return Object name filtering pattern.
     */
    protected String getOnPattern() {
        return this.onPattern;
    }

    /**
     * Set pattern, then
     * set the OnjectName list in the SamplerContext (onames).
     * @param onProto
     *            Object name filtering pattern to set.
     */
    public void setOnPattern(final String onPattern) {
        this.onPattern = onPattern;
        if (onPattern == null) {
            this.onPattern = getDefaultOnPattern();
        }
    }

    /**
     * Samples data based on a previous sample.
     *
     * @param previous
     *            Previous sample.
     *
     * @return Sampled data,
     * @see SampleData#compute(SampleData)
     */
    public final SampleData sample(final SampleData previous) {
        SampleData data = (SampleData) newSampleData();

        try {
            data.setSampleTime(System.currentTimeMillis());
            poll(data);
            if (data.isValid()) {
                data.compute(previous);
            }
        } catch (Exception e) {
            this.logger.error("Failed sampling", e);
        }

        if (data.isValid()) {
            return data;
        } else {
            return null;
        }
    }

    /**
     * @return Name set for the server in the jmxurls.properties file.
     */
    public String getName() {
        return this.context.getName();
    }

    /**
     * @return
     */
    public String getJmxUrl() {
        return this.context.getJmxUrl();
    }

    /**
     * @return Server name.
     */
    public String getServer() {
        return this.context.getServer();
    }

    /**
     * @return Server's domain name.
     */
    public String getDomain() {
        return this.context.getDomain();
    }

    public String getCmdId() {
        return this.context.getCmdId();
    }

    /**
     * @return New sample data.
     */
    protected abstract SampleData newSampleData();

    /**
     * Poll based on a previous sample.
     *
     * @param prev
     *            Previous sample.
     *
     * @return Resulting data.
     */
    protected abstract SampleData poll(final SampleData prev);

    /**
     * Object name pattern
     */
    private String onPattern = null;

    private boolean hasFailed = false;

    public boolean isHasFailed() {
        return hasFailed;
    }

    public void setHasFailed(boolean hasFailed) {
        this.hasFailed = hasFailed;
    }
    /**
     * context of the sampler
     */
    protected SamplerContext context;

}
