/**
 * JASMINe
 * Copyright (C) 2005-2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServletSampler.java 7005 2010-10-25 08:52:43Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.util.Iterator;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;

/**
 * Sampler for servlet data.
 */
public class ServletSampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public ServletSampler(final SamplerContext context) {
        super(context);
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOnPattern() {
        return "*:j2eeType=Servlet,*";
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new XSampleData();
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#poll(SampleData)
     * @see ServletSampler#pollServlets(XSampleData)
     */
    protected SampleData poll(final SampleData data) {
        pollServlets((XSampleData) data);
        data.setValid(true);
        return data;
    }

    /**
     * Polls servlet providers for data, will call
     * {@link ServletSampler#pollServlet(ObjectName, ServletData)}
     * for each object name.
     *
     * @param data  XSampleData to add polled information to.
     */
    private void pollServlets(final XSampleData data) {

        if (context.getOnames() == null) {
            return;
        }
        ObjectName[] oNames = (ObjectName[]) context.getOnames().toArray(new ObjectName[context.getOnames().size()]);

        for (int i = 0; i < oNames.length; i++) {
            ServletData sdata = new ServletData();
            sdata.setObjectName(oNames[i]);
            sdata.setSampleTime(System.currentTimeMillis());
            pollServlet(oNames[i], sdata);

            if (sdata.isValid()) {
                data.put(oNames[i], sdata);
            }
        }
    }

    /**
     * Polls one servlet provider for data.
     *
     * @param on    Name of the provider.
     * @param data  ServletData to add polled information to.
     */
    private void pollServlet(final ObjectName on, final ServletData data) {
        int reqCnt = 0;
        int errCnt = 0;
        long procTime = 0;
        long maxTime = 0;
        long minTime = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {"requestCount", "errorCount", "processingTime", "maxTime", "minTime"};

        try {
            cnx = getMBeanServerConnection();
            al = cnx.getAttributes(on, attIds);
            releaseMbeanServerConnection();
            cnx = null;
            if (al.isEmpty()) {
                /**
                 *  MBean has unexpected attributes.
                 */
                return;
            }
            Iterator it = al.iterator();
            Attribute att = (Attribute) it.next();              // requestCount
            reqCnt = ((Integer) att.getValue()).intValue();
            att = (Attribute) it.next();                        // errorCount
            errCnt = ((Integer) att.getValue()).intValue();
            att = (Attribute) it.next();                        // processingTime
            procTime = ((Long) att.getValue()).longValue();
            att = (Attribute) it.next();                        // maxTime
            maxTime = ((Long) att.getValue()).longValue();
            att = (Attribute) it.next();                        // minTime
            minTime = ((Long) att.getValue()).longValue();

            data.setRequestCount(reqCnt);
            data.setErrorCount(errCnt);
            data.setProcessingTime(procTime);
            data.setMaxTime(maxTime);
            data.setMinTime(minTime);

            data.setServerInfo(getName(), getJmxUrl(), getServer(), getDomain());
            data.setCmdId(getCmdId());
            data.setValid(true);
        } catch (Exception e) {
            this.logger.error("MBean error: {0}", e.getMessage());
        }
    }
}
