/**
 * JASMINe
 * Copyright (C) 2005-2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CommandDispatcher.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.TreeMap;

import org.ow2.jasmine.monitoring.mbeancmd.loader.CmdLoaderHelper;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * This is the main class of the archive.
 *
 * It dispatches the processing of the command to the right class.
 *
 * The available commands in the current archive are retrieved by introspection
 * of the archive.
 *
 * In order to execute a command, a dispatcher instance is created and initialized with the command's arguments.
 * The command object gets a reference on its dispatcher, so it can report its state changes to the dispatcher.
 * Then, the dispatcher sets status information in the thread executing the command.
 */
public class CommandDispatcher {

    protected Command cmd = null;

    /**
     * Dummy constructor to avoid the class from being used in any other way
     * than calling the CommandDispatcher.main method.
     */
    protected CommandDispatcher() {
        // Nothing
    }

    /**
     * Main method. Calls {@link CommandDispatcher#start(String[])}(args)
     * followed by System.exit of the returned value.
     *
     * IMPORTANT: This method calls System.exit !
     *
     * @param args See {@link CommandDispatcher#start(String[])}.
     */
    public static void main(final String[] args) {
        setClassLoader();
        CommandDispatcher dispatcher = new CommandDispatcher();
        dispatcher.setEmbedded(false);
        dispatcher.init(args);
    }

    /**
     * Start module when embedded in Jasmine
     * @param args Arguments of the command, including the command itself at the
     *            first position.
     *
     * @return 0 if succeeded, an error code otherwise.
     */
    public static int start(final String[] args) {
        CommandDispatcher dispatcher;
        logger.debug("start Command Dispatcher");
        dispatcher = new CommandDispatcher();
        dispatcher.setEmbedded(true);
        // Set status to STARTED
        dispatcher.setStarted();
        return dispatcher.init(args);
    }

    public void stopCommand() {
        if (cmd != null) {
            cmd.stop();
        }
    }

    /**
     * The class is selected from the first argument. If no class is available
     * to process the command, it dispatches it to the built-in help class.
     * @param args of the command, including the command itself at the
     *            first position.
     * @return 0 if succeeded, an error code otherwise.
     */
    public int init(final String[] args) {
        String cmdName = null;
        String[] cmdArgs = null;
        cmd = null;

        // Obtain the list of available commands
        if (commands == null) {
            findCommands();
        }

        // Retrieve and check the command
        if (args.length > 0) {
            cmdName = args[0];
            cmd = getCommand(cmdName);
        }

        // If the command is null display help, else set arguments
        if (cmd == null) {
            if (isEmbedded) {
                return -1;
            } else {
                cmdName = "help";
                cmd = getCommand(cmdName);
                if (cmd == null) {
                    logger.error("Cannot exec this command");
                    return -1;
                }
            }
        } else if (args.length > 1) {
            cmdArgs = new String[args.length - 1];
            System.arraycopy(args, 1, cmdArgs, 0, args.length - 1);
        }
        cmd.setArgs(cmdName, cmdArgs);

        // Execute command
        return cmd.exec(this);
    }

    /**
     * Returns an instance of Command.
     *
     * @param name the name of the command.
     *
     * @return Command instance corresponding to that name.
     */
    public Command getCommand(final String name) {
        Command cmd = null;
        String cname = commands.get(name);
        if (cname == null) {
            logger.error("Cannot find command " + name);
            return cmd;
        }
        try {
            Class<?> cl = loadClass(cname);
            cmd = (Command) cl.newInstance();
        } catch (Exception e) {
            logger.warn("Error when trying to instanciate {0} {1}: {2}", name, cname, e);
        }
        return cmd;
    }

    /**
     * Returns the names of the available commands in the archive.
     *
     * @return Names of the available commands in the archive, null if no
     *         command has been set yet.
     */
    public String[] getAvailableCommands() {
        if (commands == null) {
            return null;
        }
        String[] cmds = new String[commands.size()];
        cmds = commands.keySet().toArray(cmds);
        return cmds;
    }

    /**
     * Gets the radix of a given function.
     *
     * @param fqn Function name.
     *
     * @return Radix of that function.
     */
    private String getRadix(final String fqn) {
        String radix = fqn.substring(fqn.lastIndexOf(".") + 1);
        return radix;
    }

    /**
     * Retrieves the list of available commands.
     *
     * @see CommandDispatcher#commands
     */
    private void findCommands() {
        try {
            InputStreamReader in0 = getInputStreamReader(COMMAND_DEFINITIONS);
            if (in0 != null) {
                BufferedReader in = new BufferedReader(in0);
                String command = null;
                while ((command = in.readLine()) != null) {
                    if ("".equals(command) || command.indexOf("#") == 0) {
                        continue;
                    }
                    Class<?> cl;
                    try {
                        cl = loadClass(command);
                        if (Command.class.isAssignableFrom(cl)) {
                            if (commands == null) {
                                commands = new TreeMap<String, String>();
                            }
                            commands.put(getRadix(command).toLowerCase(), command);
                        }
                    } catch (ClassNotFoundException e) {
                        e.printStackTrace(System.err);
                    }
                }
            }
        } catch (IOException e) {
            System.err.println("Cannot access " + COMMAND_DEFINITIONS);
            e.printStackTrace();
        }
    }

    /**
     * Return an inputStreamReader for the specified resource, by using the context class loader.
     * @param name the resource name
     * @return an inputStreamReader for the specified resource
     * @throws IOException if an I/O exception occurs
     */
    protected InputStreamReader getInputStreamReader(final String name) throws IOException {
        InputStreamReader isr = null;
        InputStream is;
        try {
            is = Thread.currentThread().getContextClassLoader().getResourceAsStream(name);
            isr = new InputStreamReader(is);
        } catch (Exception e) {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            is = Thread.currentThread().getContextClassLoader().getResourceAsStream(name);
            isr = new InputStreamReader(is);
        }
        return isr;
    }

    /**
     * Load and return a class for a given class name, by using the defining class loader of the current class.
     * @param cname a class name
     * @return a loaded class for a given class name
     * @throws ClassNotFoundException if the class cannot be found
     */
    protected Class<?> loadClass(final String cname) throws ClassNotFoundException {
            return Class.forName(cname);
    }

    /**
     * Get the state of MBeanCmd.
     *
     * @return True if MBeanCmd is embedded in the EventSwitch, false otherwise.
     */
    public boolean isEmbedded() {
        return this.isEmbedded;
    }

    /**
     * Set the sate of MBeanCmd
     *
     * @param isEmbedded The state to set.
     */
    public void setEmbedded(final boolean isEmbedded) {
        this.isEmbedded = isEmbedded;
    }

    /**
     * Set an URL ClassLoader if an extra classpath is specified.
     * it allows to retrieve classes that are specific to mbeans
     * of managed resources.
     *
     *  Relevant only if mbeancmd is use as a command.
     */
    private static void setClassLoader() {

        URL[] urls = CmdLoaderHelper.getXClasspath();
        if (urls != null) {
            URLClassLoader urlLoader = new URLClassLoader(urls);
            Thread.currentThread().setContextClassLoader(urlLoader);
        }
    }

    /**
     * File containing all command definitions.
     */
    private static final String COMMAND_DEFINITIONS = "META-INF/mbeancmd/commands.defs";

    /**
     * List of commands.
     */
    private TreeMap<String, String> commands = null;

    /**
     * Whether MBeanCmd is embedded in EventSwitch or not. Default value is
     * true.
     */
    private boolean isEmbedded = true;

    public enum CommandStatus {STOPPED, FAILED, RUNNING, STARTED};

    private CommandStatus status;

    private String errormessage = null;

    /**
     * Start method was called.
     * The CommandDispatcher class is loaded.
     * Ready to create and initialize the command class.
     * If the command is sampler one (stat or poll), the
     * status will pass either to RUNNING or FAILED
     */
    public void setStarted() {
        logger.debug("");
        status = CommandStatus.STARTED;
        updateThreadStatus();
    }

    public void setFailed() {
        logger.debug("");
        status = CommandStatus.FAILED;
        errormessage="mbeancmd unknown error";
        updateThreadStatus();
    }

    public void setFailed(String error) {
        logger.debug(error);
        if (error == null) {
            Thread.dumpStack();
        }
        status = CommandStatus.FAILED;
        errormessage = error;
        updateThreadStatus();
    }

    public void setRunning() {
        logger.debug("");
        status = CommandStatus.RUNNING;
        updateThreadStatus();
    }

    public void setStopped() {
        logger.debug("");
        status = CommandStatus.STOPPED;
        updateThreadStatus();
    }

    /**
     * Update thread status, in case of embeded mbeancmd
     */
    private void updateThreadStatus() {
        Thread currentThread = Thread.currentThread();
        if (currentThread instanceof CmdThread) {
            CmdThread cmdThread = (CmdThread) currentThread;
            cmdThread.setErrorMessage(errormessage);
            cmdThread.setCmdStatus(status.toString());
        } else {
            logger.debug("Status not propagated");
        }
    }

    /**
     * Logger.
     */
    protected static Log logger = LogFactory.getLog(CommandDispatcher.class);
}
