/**
 * JASMINe
 * Copyright (C) 2005-2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Dump.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.IOException;
import java.util.Iterator;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAp;
import org.ow2.jasmine.monitoring.mbeancmd.JmxHelper;
import org.ow2.jasmine.monitoring.mbeancmd.context.DumpContext;

/**
 * Command that dumps thread information on a given J2EE server.
 */
public class Dump extends AbstractCommand {

    /**
     * JMX contexts.
     */
    private DumpContext[] contexts = null;


    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * The object name based on the "name" argument in the command line.
     */
    private ObjectName on = null;

    /**
     * Command dispatcher instance.
     */
    private CommandDispatcher cmdDispatcher = null;

    /**
     * Output file path, null for no file output.
     */
    private String outputFilePath = null;

    /**
     * Log thread stack dump in JOnAS log, false for no logging.
     */
    private boolean logger = false;

    /**
     * Constructor: calls {@link Dump#setOptions()}.
     */
    public Dump() {
        this.setOptions();
    }

    /**
     * Implementation of inherited abstract method.
     * Executes the command.
     * @see AbstractCommand#exec()
     * @param cmdDispatcher the command dispatcher
     * @return 0 if succeeded, an error code otherwise.
     */
    @Override
    public int exec(final CommandDispatcher cmdDispatcher) {
        this.cmdDispatcher = cmdDispatcher;

        try {
            if (arguments == null) {

                arguments=new String[1];
                arguments[0]="";

            }
            parseCommandLine(arguments);


        } catch (Exception e) {
            e.printStackTrace();
            return 1;
        }

        // retrieve the targets
        String[] targets = JmxHelper.getJmxTargets(commandLine.getOptionValues("target"));
        if ((targets != null) && (targets.length > 0)) {
            contexts = new DumpContext[targets.length];
            for (int i = 0; i < contexts.length; i++) {
                contexts[i] = new DumpContext();
                contexts[i].setName(targets[i]);
                contexts[i].setJmxUrl(JmxHelper.getJmxUrl(targets[i]));
                contexts[i].setJmxap(new JmxAp(contexts[i].getJmxUrl(),this.cmdDispatcher));
                // keep previous naming policy
                contexts[i].setAdminNaming(false);
                try {
                    ObjectName j2eeinstance = ObjectName.getInstance("*:j2eeType=J2EEServer,*");
                    MBeanServerConnection mbscnx = contexts[i].getJmxap().getMBeanServerConnection();
                    Iterator<?> onames = mbscnx.queryNames(j2eeinstance, null).iterator();
                    contexts[i].setServer(null);
                    while (contexts[i].getServer() == null) {
                        ObjectName server = (ObjectName) onames.next();
                        contexts[i].setServer((String) mbscnx.getAttribute(server, "serverName"));
                        contexts[i].setDomain(server.getDomain());
                    }
                } catch (Exception e) {
                    contexts[i].setServer("unknown_server_name");
                    contexts[i].setDomain("unknown_domain");
                } finally {
                    contexts[i].getJmxap().releaseMBeanServerConnection();
                }
                System.out.println("Target " + contexts[i].getName() + " - " + contexts[i].getJmxUrl());
            }

            getStackDump();

        } else {
            System.err.println("Target not found.");
            return 1;
        }
        return 0;
    }

    /**
     * Invoke stack dump operations on J2EEServer Mbean.
     */
    @SuppressWarnings("unchecked")
    private void getStackDump() {

        MBeanServerConnection cnx = null;
        String serverName = "";
        String domainName = "";
        for (int i = 0; i < contexts.length; i++) {
            DumpContext ctx = contexts[i];
            serverName = contexts[i].getServer();
            domainName = contexts[i].getDomain();
            try{
                setObjectName(domainName ,serverName);
                cnx = ctx.getJmxap().getMBeanServerConnection();
                ctx.setOnames(cnx.queryNames(on, null));

                Iterator<?> it = ctx.getOnames().iterator();
                while (it.hasNext()) {
                    ObjectName onGot = (ObjectName) it.next();
                    String opname ="";
                    try{

                        if (this.outputFilePath != null) {
                            opname = "printThreadStackDump";
                            // Create operation's parameter and signature arrays
                            Object [] opParams = {outputFilePath};
                            String [] opSig = {String.class.getName()};
                            // Invoke operation
                            cnx.invoke(onGot, opname, opParams, opSig);
                        }

                        if (this.logger) {
                            opname = "logThreadStackDump";
                            cnx.invoke(onGot, opname, null, null);
                        }
                        if ((this.outputFilePath == null) && (!this.logger)) {
                            opname = "threadStackDump";
                            System.out.println(cnx.getAttribute(onGot, opname));
                        }

                    } catch (InstanceNotFoundException e) {
                        System.err.println("Not available. Missing required mbean ("
                                + onGot + ")");
                        System.err.println("Run JOnAS on JDK 1.5 and enable management.");
                    } catch (Exception e) {
                        System.err.println("Could not invoke \"" + opname
                                + "\" on server \"" + serverName + "\"");
                        e.printStackTrace();
                    }
                }
            } catch (MalformedObjectNameException e) {
                System.err.println("Not available. Couldn't find mbean on server " + serverName);
                e.printStackTrace();
            } catch (IOException e) {
                System.err.println("Not available. Couldn't find mbean on server " + serverName);
                e.printStackTrace();
            } finally {
                ctx.getJmxap().releaseMBeanServerConnection();
            }

        }
    }


    /**
     * Parses the command line arguments into {@link Dump#commandLine}.
     * @param args
     *            Arguments to parse.
     * @throws ParseException
     *             If parsing fails.
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    public void parseCommandLine(final String[] args) throws ParseException,
    MalformedObjectNameException {

        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);

        if (commandLine.hasOption("l")) {
            this.logger = true;
        }

        if (commandLine.hasOption("f")) {
            this.outputFilePath = commandLine.getOptionValue("f");
        }
    }

    /**
     * Sets the object name based on the "name" argument in the command line.
     * @param domain
     *            The JOnAS domain name.
     * @param server
     *            The JOnAS server name.
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    private void setObjectName(String domain, String server) throws MalformedObjectNameException {

        this.on = new ObjectName(domain+":j2eeType=J2EEServer,name="+server);
    }

    /**
     * Sets all options that can be parsed from the command line.
     * @see Dump#options
     */
    private void setOptions() {
        options = new Options();

        // file printing operation
        Option file = new Option("f", "file", true,
        "Print thread stack dump in a given file");
        file.setRequired(false);
        file.setOptionalArg(false); // Sets whether this Option can have an
        // optional argument
        file.setArgName("file name"); // Sets the display name for the
        // argument value
        file.setArgs(1); // Sets the number of argument values this Option
        // can take
        options.addOption(file);

        // logging operation
        Option log = new Option("l", "log", false,
        "Print thread stack dump in a JOnAS log file");
        log.setRequired(false);
        options.addOption(log);

        // Optional attribute target
        Option target = new Option("target", "target", true, "instances to get thread dump from");
        target.setRequired(false);
        target.setOptionalArg(true);
        target.setArgs(Option.UNLIMITED_VALUES);
        target.setArgName("instances");
        options.addOption(target);
    }

    /**
     * Abstract redefinition of inherited method.
     * Retrieves the command's summary.
     * @see AbstractCommand#summary()
     * @return Command's summary.
     */
    @Override
    public String summary() {
        return "Dumps a thread stack from the JVM into stdout";
    }

    /**
     * Tests the Dump implementation.
     * @param args
     *            Arguments to pass to Dump#setArgs.
     */
    public static void main(final String[] args) {
        Dump d = new Dump();
        d.setArgs("dump", args);
        d.exec(null);
    }
}
