/**
 * JASMINe
 * Copyright (C) 2005-2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MonologJMX.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.File;
import java.io.IOException;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.Set;
import java.util.logging.LogRecord;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectName;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAp;
import org.ow2.jasmine.monitoring.mbeancmd.JmxHelper;
import org.ow2.jasmine.monitoring.mbeancmd.Outer;
import org.ow2.jasmine.monitoring.mbeancmd.context.MonologContext;
import org.ow2.jasmine.monitoring.mbeancmd.jasmine.JasmineConnector;

/**
 * Command that displays all Monolog logs from a given J2EE server.
 */
public class MonologJMX extends AbstractCommand {

    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * JMX contexts.
     */
    private MonologContext[] context = null;

    /**
     * Constructor.
     */
    public MonologJMX() {
        setOptions();
    }

    public int exec(CommandDispatcher cmdDispatcher) {

        try {
            parseCommandLine(arguments);
        } catch (Exception e) {
            e.printStackTrace();
            return 1;
        }

        /*
         * out (or pout) is : - the output of the sampler - the input for the
         * Outer to print into System.out or into a file sink is: - the sink of
         * the Outer - the input for the graph, if any
         */
        PipedOutputStream out = new PipedOutputStream();
        PrintStream pout = new PrintStream(out);

        Outer outer = null;
        try {
            if (commandLine.hasOption("f")) {
                String outputFilePath = commandLine.getOptionValue("f");
                if (outputFilePath != null) {
                    outer = new Outer(out, new File(outputFilePath));
                }
            }

            if (commandLine.hasOption("jasmine")) {
                String jasmineURI = commandLine.getOptionValue("jasmine");

                if (jasmineURI != null) {
                    if (outer == null) {
                        outer = JasmineConnector.connect(out, jasmineURI);
                    } else {
                        Outer oldOuter = outer;
                        outer = JasmineConnector.connect(outer, jasmineURI);
                        new Thread(oldOuter).start();
                    }
                }
            }

            if (outer == null) {
                outer = new Outer(out, System.out);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        new Thread(outer).start();

        /*
         *
         */
        // retrieve the method
        String[] targets = JmxHelper.getJmxTargets(commandLine
                .getOptionValues("target"));

        if ((targets != null) && (targets.length > 0)) {
            context = new MonologContext[targets.length];
            for (int i = 0; i < context.length; i++) {
                context[i] = new MonologContext();
                context[i].setName(targets[i]);
                context[i].setJmxUrl(JmxHelper.getJmxUrl(targets[i]));
                context[i].setJmxap(new JmxAp(context[i].getJmxUrl(), null));
                // keep previous naming policy
                context[i].setAdminNaming(false);
                try {
                    ObjectName j2eeinstance = ObjectName
                            .getInstance("*:j2eeType=J2EEServer,*");
                    MBeanServerConnection mbscnx = context[i].getJmxap()
                            .getMBeanServerConnection(false);
                    Iterator<?> onames = mbscnx.queryNames(j2eeinstance, null)
                            .iterator();
                    context[i].setServer(null);
                    while (context[i].getServer() == null) {
                        ObjectName server = (ObjectName) onames.next();
                        context[i].setServer((String) mbscnx.getAttribute(
                                server, "serverName"));
                        context[i].setDomain(server.getDomain());
                    }
                } catch (Exception e) {
                    context[i].setServer("unknown_server_name");
                    context[i].setDomain("unknown_domain");
                } finally {
                    context[i].getJmxap().releaseMBeanServerConnection();
                }
                System.err.println("Target " + context[i].getName() + " - "
                        + context[i].getJmxUrl());
            }
        }

        /*
         *
         */
        String[] managee = JmxHelper.getJmxTargets(commandLine
                .getOptionValues("target"));
        if ((managee != null) && (managee.length > 0)) {
            for (int i = 0; i < managee.length; i++) {

                try {

                    JmxAp jmxAp = new JmxAp(JmxHelper.getJmxUrl(managee[i]),
                            cmdDispatcher);

                    ObjectName on = ObjectName
                            .getInstance("*:type=service,name=log,*");

                    MBeanServerConnection mbscnx = null;
                    MonologContext context = new MonologContext();
                    context.setJmxap(jmxAp);
                    try {
                        ObjectName j2eeinstance = ObjectName
                                .getInstance("*:j2eeType=J2EEServer,*");
                        mbscnx = jmxAp.getMBeanServerConnection(false);
                        Iterator<?> onames = mbscnx.queryNames(j2eeinstance, null)
                                .iterator();
                        context.setServer(null);
                        while (context.getServer() == null) {
                            ObjectName server = (ObjectName) onames.next();
                            context.setServer((String) mbscnx.getAttribute(
                                    server, "serverName"));
                            context.setDomain(server.getDomain());
                        }
                    } catch (Exception e) {
                        context.setServer("unknown_server_name");
                        context.setDomain("unknown_domain");
                    } finally {
                        mbscnx = null;
                    }
                    context.setJmxUrl(JmxHelper.getJmxUrl(managee[i]));
                    context.setName(managee[i]);

                    NotificationListener listener = new MonologNotificationListener(
                            pout, context);

                    Set<?> objectNames = jmxAp.getMBeanServerConnection().queryNames(on, null);
                    if (objectNames.size() != 1) {
                        System.err.println("Unable to retrieve the MBean of service log!");
                    }

                    jmxAp.getMBeanServerConnection(false).addNotificationListener(
                            (ObjectName) objectNames.toArray()[0], listener, null, null);

                } catch (MalformedObjectNameException e) {
                    e.printStackTrace();
                } catch (NullPointerException e) {
                    e.printStackTrace();
                } catch (InstanceNotFoundException e) {
                    e.printStackTrace();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        } else {
            System.err.println("No target specified!");
            return 2;
        }

        while (true) {
            try {
                Thread.sleep(Long.MAX_VALUE);
            } catch (InterruptedException e) {
            }
        }
    }

    public String summary() {
        return "Returns Monolog logs from a given JavaEE server";
    }

    /**
     * Parses the command line arguments into {@link Poll#commandLine}.
     *
     * @param args
     *            Arguments to parse.
     * @throws ParseException
     *             If parsing fails.
     */
    public void parseCommandLine(final String[] args) throws ParseException {
        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);
    }

    /**
     *
     */
    public void setOptions() {

        options = new Options();

        // file output
        Option file = new Option("f", "file", true, "Send output to file instead of stdout");
        file.setRequired(false);
        file.setArgName("path");
        file.setArgs(1);
        options.addOption(file);

        // jasmine connector output
        Option jasmine = new Option("jasmine", "jasmine", true,
                "Output logged data to jasmine (will disable stdout)");
        jasmine.setRequired(false);
        jasmine.setArgName("jasmineURI");
        jasmine.setArgs(1);
        options.addOption(jasmine);

        // Optional attribute target
        Option target = new Option("target", "target", true,
                "instances to probe");
        target.setRequired(false);
        target.setOptionalArg(true);
        target.setArgs(Option.UNLIMITED_VALUES);
        target.setArgName("instances");
        options.addOption(target);

    }
}

class MonologNotificationListener implements NotificationListener {

    /**
     *
     */
    String header = "date;time;sname;server;domain;mbean;timestamp;stamp;sequence;loggername;level;message";

    /**
     *
     */
    boolean isHeaderSent = false;

    /**
     *
     */
    MonologContext context = null;

    /**
     *
     */
    public static final SimpleDateFormat SIMPLEDATEFORMAT = new SimpleDateFormat(
            "yyyy/MM/dd HH:mm:ss");

    /**
     * @param out
     */
    PrintStream out = null;

    /**
     * @param out
     * @param context
     */
    public MonologNotificationListener(PrintStream out, MonologContext context) {
        this.out = out;
        this.context = context;
    }

    /* (non-Javadoc)
     * @see javax.management.NotificationListener#handleNotification(javax.management.Notification, java.lang.Object)
     */
    public void handleNotification(final Notification nf, final Object arg1) {
        LogRecord record = (LogRecord) nf.getUserData();

        StringBuffer bf = new StringBuffer();

        if (!isHeaderSent){
            bf.append(header);
            bf.append("\n");

            isHeaderSent= !isHeaderSent;
        }

        bf.append(getHeaderBody());
        bf.append(";");
        bf.append(new Date(record.getMillis()));
        bf.append(";");
        bf.append(record.getMillis());
        bf.append(";");
        bf.append(record.getSequenceNumber());
        bf.append(";");
        bf.append(record.getLoggerName());
        bf.append(";");
        bf.append(record.getLevel().toString());
        bf.append(";");
        bf.append(record.getMessage());

        // Append the cause: useful to identify a java.lang.OutOfMemoryError
        Throwable throwable = record.getThrown();
        if (throwable != null) {
            bf.append(" - " + throwable.toString());
        }

        bf.append("\n");

        out.print(bf.toString());
    }

    /**
     * @return Values of header.
     */
    private String getHeaderBody() {

        StringBuffer buff = new StringBuffer();

        // date;time;sname;server;domain;mbean;

        long time = System.currentTimeMillis();
        buff.append(SIMPLEDATEFORMAT.format(new Date(time)));
        buff.append(";");
        buff.append(time);
        buff.append(";");
        buff.append(context.getJmxUrl());
        buff.append(";");
        buff.append(context.getServer());
        buff.append(";");
        buff.append(context.getDomain());
        buff.append(";");
        buff.append("jonas:type=service,name=log,fname=trace");

        return buff.toString();
    }
}
