/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CpuSunData.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.util.HashMap;
import java.util.Map;


/**
 * Data retrieved from the Sun Java VM.
 */
public class CpuSunData extends SampleData {
    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#compute(SampleData)
     */
    public void compute(final SampleData needs_cast) {
        CpuSunData prev = (CpuSunData) needs_cast;
        if (prev != null) {
            this.processCpuTime = this.processCpuTime/1000000;              // milliseconds
            currentCpuTime = (this.processCpuTime - prev.processCpuTime);
            long deltaTime = this.getSampleTime() - prev.getSampleTime();   // milliseconds
            currentCpuLoad =  currentCpuTime * 100.0 / ( deltaTime * this.availableProcessors ) ;
        } else {
            this.processCpuTime = this.processCpuTime/1000000;              // milliseconds
            currentCpuTime = 0;
            currentCpuLoad = 0;
        }
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintHeader()
     */
    protected String getInnerPrintHeader() {
        return header.replaceAll(";", getSeparator());
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintData()
     */
    protected String getInnerPrintData() {
        return  decimalFormat.format(getProcessCpuTime()) + getSeparator()
              + decimalFormat.format(getCurrentCpuTime()) + getSeparator()
              + decimalFormat.format(getCurrentCpuLoad());
    }

    /**
     * @return  Total time used by this process.
     */
    public double getProcessCpuTime() {
        return processCpuTime;
    }

    /**
     * @param processCpuTime  Total time used by this process.
     */
    public void setProcessCpuTime(final long processCpuTime) {
        this.processCpuTime = processCpuTime;
    }

    /**
     * @return  Number of available processors.
     */
    public double getAvailableProcessors() {
        return availableProcessors;
    }

    /**
     * @param availableProcessors  Number of available processors.
     */
    public void setAvailableProcessors(final long availableProcessors) {
        this.availableProcessors = availableProcessors;
    }

    /**
     * @return  Current CPU time.
     */
    public double getCurrentCpuTime() {
        return currentCpuTime;
    }

    /**
     * @param currentCpuTime  Current CPU time.
     */
    public void setCurrentCpuTime(final long currentCpuTime) {
        this.currentCpuTime = currentCpuTime;
    }

    /**
     * @return  Current CPU load.
     */
    public double getCurrentCpuLoad() {
        return currentCpuLoad;
    }

    /**
     * @param currentCpuLoad  Current CPU load.
     */
    public void setCurrentCpuLoad(final long currentCpuLoad) {
        this.currentCpuLoad = currentCpuLoad;
    }

    /**
     * Total time used by this process.
     */
    private long processCpuTime = 0;

    /**
     * Number of available processors.
     */
    private long availableProcessors = 0;

    /**
     * Current CPU time.
     */
    private long currentCpuTime = 0;

    /**
     * Current CPU load.
     */
    private double currentCpuLoad = 0;

    /**
     * Default output header.
     */
    private static final String header = "processCpuTime;currentCpuTime;currentCpuLoad";

    @Override
    public Map<String, ?> getValueOfAttributes() {
        HashMap<String, Object> valueOfAttributes = new HashMap<String, Object>();
        valueOfAttributes.put("processCpuTime", processCpuTime);
        valueOfAttributes.put("currentCpuTime", currentCpuTime);
        valueOfAttributes.put("currentCpuLoad", currentCpuLoad);
        return valueOfAttributes;
    }
}
