/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServletData.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.util.HashMap;
import java.util.Map;


/**
 * Data retrieved from servlets.
 */
public class ServletData extends SampleData {
    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#compute(SampleData)
     */
    public void compute(final SampleData needs_cast) {
        ServletData prev = (ServletData) needs_cast;
        if (prev != null) {
        	double sampleTime = this.computeSamplePeriod(prev);
            // requestRate
            int deltaReq = this.requestCount - prev.requestCount;
            this.setRequestRate(deltaReq / sampleTime);

            // errorRate
            int deltaErr = this.errorCount - prev.errorCount;
            this.setErrorRate(deltaErr / sampleTime);

            // processing time per request
            long deltaTime = this.processingTime - prev.processingTime;
            if (deltaReq > 0) {
                this.setProcTimeReq(1.0 * deltaTime / deltaReq);
            } else {
                this.setProcTimeReq(0.0);
            }
        }

    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintHeader()
     */
    protected String getInnerPrintHeader() {
        return header.replaceAll(";", getSeparator());
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintData()
     */
    protected String getInnerPrintData() {
        return  this.getRequestCount() + getSeparator()
              + decimalFormat.format(this.getRequestRate()) + getSeparator()
              + this.getErrorCount() + getSeparator()
              + decimalFormat.format(this.getErrorRate()) + getSeparator()
              + decimalFormat.format(this.getProcTimeReq()) + getSeparator()
              + this.getMaxTime() + getSeparator()
              + this.getMinTime();
    }

    @Override
    public Map<String, ?> getValueOfAttributes() {
        HashMap<String, Object> valueOfAttributes = new HashMap<String, Object>();
        valueOfAttributes.put("requestCount", requestCount);
        valueOfAttributes.put("requestRate", requestRate);
        valueOfAttributes.put("errorCount", errorCount);
        valueOfAttributes.put("errorRate", errorRate);
        valueOfAttributes.put("procTimeReq", procTimeReq);
        valueOfAttributes.put("maxTime", maxTime);
        valueOfAttributes.put("minTime", minTime);
        return valueOfAttributes;
    }

    /**
     * @return  Request count.
     */
    public int getRequestCount() {
        return requestCount;
    }

    /**
     * @param requestCount  Request count.
     */
    public void setRequestCount(final int requestCount) {
        this.requestCount = requestCount;
    }

    /**
     * @return  Error count.
     */
    public int getErrorCount() {
        return errorCount;
    }

    /**
     * @param errorCount  Error count.
     */
    public void setErrorCount(final int errorCount) {
        this.errorCount = errorCount;
    }

    /**
     * @return  Processing time.
     */
    public long getProcessingTime() {
        return processingTime;
    }

    /**
     * @param processingTime  Processing time.
     */
    public void setProcessingTime(final long processingTime) {
        this.processingTime = processingTime;
    }

    /**
     * @return  Maximal processing time.
     */
    public long getMaxTime() {
        return maxTime;
    }

    /**
     * @param maxTime  Maximal processing time.
     */
    public void setMaxTime(final long maxTime) {
        this.maxTime = maxTime;
    }

    /**
     * @return  Minimal processing time.
     */
    public long getMinTime() {
        return minTime;
    }

    /**
     * @param minTime  Minimal processing time.
     */
    public void setMinTime(final long minTime) {
        this.minTime = minTime;
    }

    /**
     * @return  Request rate (computed).
     */
    public double getRequestRate() {
        return requestRate;
    }

    /**
     * @param requestRate  Request rate (computed).
     */
    protected void setRequestRate(final double requestRate) {
        this.requestRate = requestRate;
    }

    /**
     * @return  Error rate (computed).
     */
    public double getErrorRate() {
        return errorRate;
    }

    /**
     * @param errorRate  Error rate (computed).
     */
    protected void setErrorRate(final double errorRate) {
        this.errorRate = errorRate;
    }

    /**
     * @return  Process time per request (computed).
     */
    public double getProcTimeReq() {
        return procTimeReq;
    }

    /**
     * @param procTimeReq  Process time per request (computed).
     */
    protected void setProcTimeReq(final double procTimeReq) {
        this.procTimeReq = procTimeReq;
    }

    /**
     * Request count.
     */
    private int requestCount;

    /**
     * Error count.
     */
    private int errorCount;

    /**
     * Processing time.
     */
    private long processingTime;

    /**
     * Maximal processing time.
     */
    private long maxTime;

    /**
     * Minimal processing time.
     */
    private long minTime;

    /**
     * Request rate (computed).
     */
    private double requestRate;

    /**
     * Error rate (computed).
     */
    private double errorRate;

    /**
     * Process time per request (computed).
     */
    private double procTimeReq;

    /**
     * Default output header.
     */
    private static final String header = "requestCount;requestRate;errorCount;errorRate;procTimeReq;maxTime;minTime";
}
