/**
 * JASMINe
 * Copyright (C) 2009-2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxHelper.java 8077 2011-04-27 14:55:19Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Unmarshaller;

import org.ow2.jasmine.monitoring.mbeancmd.generated.ObjectFactory;
import org.ow2.jasmine.monitoring.mbeancmd.generated.ProbeConfig;
import org.ow2.jasmine.monitoring.mbeancmd.generated.Target;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

public class JmxHelper {

    /**
     * Logger
     */
    private static Log logger = LogFactory.getLog(JmxHelper.class);

    /**
     * Default JMX URL.
     */
    public static final String DEFAULT_URL = "service:jmx:rmi:///jndi/rmi://localhost:1099/jrmpconnector_jonas";

    /**
     * Default name of the properties file.
     */
    public static final String DEFAULT_FILE = "probe-config.xml";
    public static final String DEFAULT_CONFIG_FILE = "probe-config.xml";

    /**
     * Map of JMX targets.
     */
    private static Map<String, String> jmxTargets = new TreeMap<String, String>();

    /**
     * Map of JMX user names.
     */
    private static Map<String, String> jmxUsers = new TreeMap<String, String>();

    /**
     * Map of JMX passwords.
     */
    private static Map<String, String> jmxPasswords = new TreeMap<String, String>();

    /**
     * Map of protocol provider package list. The provider packages are read
     * from the jasmine.jmx.protocol.providers.<target> property Useful for non
     * standard protocols such as t3 Example: if the JMX URL is:
     * service:jmx:t3:/
     * /frecb000650:7001/jndi/weblogic.management.mbeanservers.runtime then the
     * protocol provider is: weblogic.management.remote see also:
     * jmx.remote.protocol.provider.pkgs property in JMXConnectorFactory
     * definition
     *
     */
    private static Map<String, String> jmxProtocolProviderPackages = new TreeMap<String, String>();

    /**
     * Default JMX target.
     */
    private static String defaultTarget = null;

    /**
     * Naming context that is used by the JmxAps.
     */
    private static Context ctx = null;

    public static Context getContext() {
        if (ctx != null) {
            return ctx;
        }
        try {
            ctx = new InitialContext();
        } catch (NamingException e) {
            logger.error("Error while creating initialContext : {0}. An unmanaged JMX connection will be used instead.", e.getMessage());
        }
        return ctx;
    }

    /**
     * The JMX URL of the default target.
     * @return JMX URL of the default target
     */
    public static String getDefaultJmxTarget() {
        if (defaultTarget != null) {
            return jmxTargets.get(defaultTarget);
        }
        return null;
    }

    /**
     * @return Default target.
     */
    public static String getDefaultTarget() {
        return defaultTarget;
    }

    /**
     * @param jmxUrl A given jmx URL
     * @return The user name associated to this JMX URL
     */
    public static String getJmxUser(String jmxUrl) {
        return jmxUsers.get(jmxUrl);
    }

    /**
     * @param jmxUrl A given jmx URL
     * @return The password associated to this JMX URL
     */
    public static String getJmxPassword(String jmxUrl) {
        return jmxPasswords.get(jmxUrl);
    }

    /**
     *
     * @param jmxUrl jmxUrl A given jmx URL
     * @return The protocol provider package associated to this JMX URL
     */
    public static String getJmxProtocolProviderPackage(String jmxUrl) {
        return jmxProtocolProviderPackages.get(jmxUrl);
    }

    /**
     * Gets the JMX URL for a given target.
     *
     * @param name Name of the target.
     *
     * @return JMX URL corresponding to that target.
     */
    public static String getJmxUrl(final String name) {
        if (name == null) {
            return jmxTargets.get(defaultTarget);
        } else if (name.startsWith("service:jmx:rmi")) {
            return name;
        } else {
            return jmxTargets.get(name);
        }
    }

    /**
     * @return An array containing the names of all current JMX targets.
     */
    public static String[] getJmxTargets() {

        String[] targets = null;
        if (!jmxTargets.isEmpty()) {
            targets = new String[jmxTargets.size()];
            targets = jmxTargets.keySet().toArray(targets);
        }
        return targets;
    }

    /**
     * @param jmxUrl A jmx url for a target that maybe defined in the configuration file.
     * @return If there is a target having its url equal to the given jmxUrl, return its name.
     * Otherwise return null.
     */
    public static String getTargetName(String jmxUrl) {
        String name = null;
        if (!jmxTargets.isEmpty()) {
            for (Iterator<String> keysIt = jmxTargets.keySet().iterator(); keysIt.hasNext(); ) {
                String targetName = keysIt.next();
                String targetUrl = jmxTargets.get(targetName);
                if (targetUrl != null && targetUrl.equals(jmxUrl)) {
                    // Found the target whose url is the given jmxUrl
                    name = targetName;
                    break;
                }
            }
        }
        return name;
    }

    /**
     * Gets the names of all JMX targets matching a certain criteria.
     *
     * @param proposed Criteria.
     *
     * @return An array containing the names of all current JMX targets.
     */
    public static String[] getJmxTargets(final String[] proposed) {
        boolean all = false;
        String[] targetsArray = null;
        List<String> targetsList = new LinkedList<String>();

        // Look for defined target
        if (proposed != null) {
            for (int i = 0; i < proposed.length; i++) {
                if ("all".equalsIgnoreCase(proposed[i])) {
                    all = true;
                    break;
                }
                String url = getJmxUrl(proposed[i]);
                if (url != null) {
                    if (url.equals(proposed[i])) {
                        // proposed[i] corresponds to a jmxUrl
                        // check for a target having this url
                        String targetName = getTargetName(url);
                        if (targetName != null) {
                            targetsList.add(targetName);
                        } else {
                            // proposed[i] corresponds to an un-defined target.
                            targetsList.add(proposed[i]);
                        }
                    } else {
                        // proposed[i] corresponds to a defined target
                        targetsList.add(proposed[i]);
                    }
                } else {
                    logger.info("Target {0} is not defined in configuration file. Please check the probe-config.xml.", proposed[i]);
                }
            }
        } else {
            if (defaultTarget != null) {
                targetsList.add(defaultTarget);
            }
        }

        if (all) {
            targetsArray = getJmxTargets();
        } else {
            targetsArray = new String[targetsList.size()];
            for (int i = 0; i < targetsList.size(); i++) {
                targetsArray[i] = targetsList.get(i);
            }
        }
        return targetsArray;
    }

    /**
     * Loads all JMX jmxConnection properties from a file.
     */
    private static void load() {

        Log logger = LogFactory.getLog(JmxHelper.class);

        // Default target is derived from the jasmine.jmx.url system property.
        String defaultUrl = System.getProperty("jasmine.jmx.url");
        if (defaultUrl != null) {
            String defaultName = null;
            if (defaultUrl.matches(".*connector_.*")) {
                int ix = defaultUrl.lastIndexOf("connector_");
                defaultName = defaultUrl.substring(ix + "connector_".length());
            } else {
                defaultName = defaultUrl;
            }
            defaultTarget = defaultName;
        }

        // Retrieve the probe config file
        ClassLoader loader = ClassLoader.getSystemClassLoader();
        InputStream resource = null;
        try {
            resource = loader.getResourceAsStream(DEFAULT_CONFIG_FILE);
            if (resource == null) {
                resource = new FileInputStream(new File(DEFAULT_CONFIG_FILE));
            }
        } catch (Exception e) {
            if (defaultUrl == null) {
                logger.error("No default target defined with jasmine.jmx.url property, and cannot find probe-config.xml");
            } else {
                jmxTargets.put(defaultTarget, defaultUrl);
            }
            return;
        }

        logger.debug("Parsing probe-config.xml");

        ProbeConfig probeConfig = null;
        // Read targets configuration from probes configuration file
        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
            Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
            probeConfig = (ProbeConfig) unmarshaller.unmarshal(resource);
            for (Target target : probeConfig.getTarget()) {
                String name = target.getId();
                String url = target.getUrl();
                jmxTargets.put(name, url);
                String user = target.getUser();
                jmxUsers.put(url, user);
                String password = target.getPassword();
                jmxPasswords.put(url, password);
                /*
                 * TODO
                String protocolProvider = ...
                jmxProtocolProviderPackages.put(url, protocolProvider);
                */
            }
        } catch (Exception e) {
            e.printStackTrace();
            //throw e;
        } finally {
            try {
                resource.close();
            } catch (IOException ignore) {
            }
        }

        // If the default target is also defined in the configuration file,
        // jasmine.jmx.url takes precedence.
        if (defaultUrl != null) {
            jmxTargets.put(defaultTarget, defaultUrl);
        }

    }

    /**
     * @param propertyFile File name of the properties file.
     *
     * @return Parsed version of the propertyFile file.
     *
     * @throws IOException If propertyFile cannot be read.
     */
    private static Properties getProperties(final File propertyFile) throws IOException {
        Properties properties = new Properties();

        FileInputStream in = new FileInputStream(propertyFile);
        properties.load(in);
        in.close();

        return properties;
    }


    static {
        load();
    }
}
