/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 *
 */
package org.ow2.jasmine.monitoring.mbeancmd.audit;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.audit.util.MetricMath;

/**
 * @author waeselyf
 * 
 */
public class FileCollector {

    /**
     * Constructor from an input stream (e.g. System.in) 
     * @param in
     */
    public FileCollector(final InputStream in) {
        this.input_source = new LineNumberReader(new InputStreamReader(in));
    }

    /**
     * Constructor from a file
     * 
     * @param file
     * @throws FileNotFoundException
     */
    public FileCollector(final File file) throws FileNotFoundException {
        this.input_source = new LineNumberReader(new FileReader(file));
    }

    /*
    public static void main(String[] args) {
        String file = "toto.log";
        if (args.length > 0) {
            file = args[0];
        }
        String[] atts = new String[] { "requestCount", "procTimeReq" };

        FileCollector fc = null;
        try {
            fc = new FileCollector(new File(file));
            fc.setAttributes(atts);
            fc.scan();
            MetricSorter sorter = new MetricSorter();
            TreeSet<Metric> tset = sorter.sort(fc.getPoll(), "requestCount");
            MetricSorter.printMetrics(System.out, tset, "requestCount");
            System.out.println();
            sorter.setBaseline(fc.getBaseline());
            tset = sorter.sort(fc.getPoll(), "requestCount");
            MetricSorter.printMetrics(System.out, tset, "requestCount");

        } catch (FileNotFoundException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        System.out.println(fc.getPoll().size());
    }
    */

    /**
     * Scans the file or input stream.
     * - builds the baseline from initial values.
     * - builds the most recent metrics
     * - if on, computes statistics on a per attribute basis.  
     *
     */
    public void scan() {
        boolean goOn = true;

        HashMap<String, Metric> metricMap = new HashMap<String, Metric>();
        while (goOn) {
            try {
                String ln = this.input_source.readLine();
                if (ln == null) {
                    goOn = false;
                    continue;
                }
                if (header == null) {
                    header = ln;
                    processHeader();
                } else {
                    Metric metric = parseMetric(ln);
                    if (!baseline.containsKey(metric.getRadical())) {
                        baseline.put(metric.getRadical(), metric);
                    }
                    metricMap.put(metric.getRadical(), metric);
                    if (isStatistics) {
                        updateStatistics(metric);
                    }
                }
            } catch (IOException e) {
                e.printStackTrace(System.err);
                goOn = false;
            } catch (InvalidHeaderException e) {
                e.printStackTrace(System.err);
                goOn = false;
            } catch (ParseException e) {
                e.printStackTrace(System.err);
                goOn = false;
            }
        }
        for (Metric m : metricMap.values()) {
            metrics.add(m);
        }

        try {
            input_source.close();
        } catch (IOException e) {
            // Nothing to do
        }
    }

    /**
     * 
     * @return
     */
    public LinkedList<Metric> getPoll() {
        return metrics;
    }

    /**
     * 
     */
    public HashMap<String, Metric> getBaseline() {
        return baseline;
    }

    /**
     * 
     * Builds a LinkedList of metrics from the computed statistics
     * Each metrics as a list of attributes derived from attr through the scan
     * - average: mean value for the full scan
     * - averageNonZero: mean value, considering only non null values
     * - minNotzero: the minimal not null value
     * - max: maximum value.
     * - timeInterval: the time interval of the full scan
     * - timeNonZero: cumulated time whete attr has a non null value.
     * - ratioNonZero: timeNonZero/timeInterval 
     * 
     * @param attr
     * @return
     */
    public LinkedList<Metric> getStats(String attr) {

        LinkedList<Metric> ret = new LinkedList<Metric>();
        HashMap<String, Statistic> stat = null;
        for (int i = 0; i < atts.length ; i++) {
            if (atts[i].equals(attr)) {
                stat = stats.get(i);
                break;
            }
        }
        if (stat != null) {
            for (String key : stat.keySet()) {
                
                try {
                    Metric m = Metric.newInstance(key);
                    Statistic s = stat.get(key);
                    m.getAttributes().add(new Attribute(attr, new Double(s.average)));
                    m.getAttributes().add(new Attribute("averageNotZero", new Double(s.averageNonZero)));
                    m.getAttributes().add(new Attribute("minNotZero", new Double(s.min)));
                    m.getAttributes().add(new Attribute("max", new Double(s.max)));
                    m.getAttributes().add(new Attribute("timeInterval", new Long(s.time)));
                    m.getAttributes().add(new Attribute("timeNonZero", new Long(s.timeNonZero)));
                    m.getAttributes().add(new Attribute("ratioNonZero", new Double(s.ratioNZ)));
                    ret.add(m);
                } catch (InvalidMetricRadical e) {
                    // skip it
                }
                
            }
        }
        
        return ret;
    }

    /**
     * Set the attributes to scan 
     * 
     * @param attributes
     */
    public void setAttributes(String[] attributes) {
        atts = attributes.clone();
        attIndex = new int[atts.length];
        if (isStatistics) {
            initStatistics();
        }
    }

    /**
     * @return the separator
     */
    public String getSeparator() {
        return separator;
    }

    /**
     * @param separator
     *            the separator to set
     */
    public void setSeparator(String separator) {
        this.separator = separator;
    }

    /**
     * Enable statistics. Statistics will be computed only y this method is called.
     *
     */
    public void enableStatistics() {
        isStatistics = true;
        initStatistics();
    }
    
    /**
     * Initialises statistics data
     *
     */
    private void initStatistics() {

        if (atts != null) {
            // stats = new HashMap<String,Statistic>[atts.length];
            stats = new ArrayList<HashMap<String, Statistic>>(atts.length);
            for (int i = 0; i < atts.length; i++) {
                stats.add(new HashMap<String, Statistic>());
            }
        }
    }

    /**
     * Update statistics data from the current metric.
     * 
     * @param m
     */
    private void updateStatistics(Metric m) {
        for (int i = 0; i < atts.length; i++) {
            
            // look for the statistic, create it if needed
            String mId = m.getRadical();
            Statistic s = stats.get(i).get(mId);
            if (s == null) {
                s = new Statistic();
                stats.get(i).put(mId, s);
            }
            
            // retrieve attribute value from metric
            Attribute a = m.getAttribute(atts[i]);
            double d = MetricMath.toDouble(a);
            long t = m.getTimestamp();
                        
            if (s.count == 0) { // new  statistics
                s.t0 = t;
                s.time = 0;
                s.timeNonZero =0;
                s.ratioNZ = 0;
                s.sum = 0;
                s.average = d;
                s.averageNonZero = d;
                s.min = d;
                s.max = d;
            } else { // update existing statistics
                s.time = t - s.t0;
                s.sum += d * (t - s.prevTStamp);
                s.average = s.sum / s.time;
                if (d != 0) {
                    s.timeNonZero += t - s.prevTStamp;
                    s.averageNonZero = s.sum / s.timeNonZero;
                }
                if (s.time != 0) {
                    s.ratioNZ = 1.0 * s.timeNonZero / s.time;
                }
                if ( (s.min == 0) && (d > 0)) {
                    s.min = d;
                }
                if ((s.min != 0) && (d > 0) && (d < s.min)) {
                    s.min =  d;
                }
                
                if (d > s.max) {
                    s.max = d;
                }
            }
            
            // some update before next loop
            s.count++;
            s.prevTStamp = t;            
        }
    }

    /**
     * Retrieves the time, mbean, source and attributes fields
     * 
     * @throws InvalidHeaderException
     * 
     */
    private void processHeader() throws InvalidHeaderException {
        String[] fields = header.split(separator);

        // look for time field
        timeIndex = fieldIndex(fields, timeField);
        mbeanIndex = fieldIndex(fields, mbeanField);
        sourceIndex = fieldIndex(fields, sourceField);

        for (int i = 0; i < atts.length; i++) {
            attIndex[i] = fieldIndex(fields, atts[i]);
        }
    }

    /**
     * 
     * @author waeselyf
     *
     */
    private class InvalidHeaderException extends Exception {
        /**
         * 
         */
        private static final long serialVersionUID = 1L;

        public InvalidHeaderException(String msg) {
            super(msg);
        }
    }

    /**
     * Check the presence of an expected field within the header 
     * 
     * @param s
     * @param key
     * @return
     * @throws InvalidHeaderException
     */
    private int fieldIndex(String[] s, String key) throws InvalidHeaderException {
        int ret = -1;
        for (int i = 0; i < s.length; i++) {
            if (key.equals(s[i])) {
                ret = i;
                break;
            }
        }
        if (ret == -1) {
            throw new InvalidHeaderException("Expected field not found: " + key);
        }
        return ret;
    }

    /**
     * 
     * @param line
     * @throws ParseException
     */
    private Metric parseMetric(String line) throws ParseException {

        String[] tokens = line.split(separator);

        // timestamp
        long timestamp = parseTime(tokens[timeIndex]);

        // source
        String source = tokens[sourceIndex];

        // ObjectName
        ObjectName on = null;
        try {
            on = ObjectName.getInstance(tokens[mbeanIndex]);
        } catch (MalformedObjectNameException e) {
            try {
                on = ObjectName.getInstance("nodomain:type=void");
            } catch (MalformedObjectNameException e1) {
                // Cannot happen
            }
        }

        // Attribute list
        AttributeList attl = new AttributeList();
        for (int i = 0; i < atts.length; i++) {
            double d = 0.0;
            try {
                d = Double.parseDouble(tokens[attIndex[i]]);
            } catch (NumberFormatException e) {
                // nothing to do
            }
            attl.add(new Attribute(atts[i], Double.valueOf(d)));

        }
        Metric metric = new Metric(timestamp, source, on, attl);
        return metric;
    }

    private long parseTime(String time) throws ParseException {
        long ret = 0;
        if (isTimeAsLong) {
            ret = Long.parseLong(time);
        } else {
            ret = simpleDateFormat.parse(time).getTime();
        }
        return ret;
    }

    private class Statistic {

        double max = 0.0;

        double min = 0.0;

        double average = 0.0;

        double averageNonZero = 0.0;

        double sum = 0.0;

        int count = 0;

        long time = 0;

        long timeNonZero = 0;
        
        double ratioNZ = 0.0;
        
        long prevTStamp = 0;
        
        long t0 = 0;
    }

    /**
     * Lines input line per line.
     */
    private LineNumberReader input_source = null;

    /**
     * separator
     */
    private String separator = ";";

    /**
     * MBean field
     */
    private String mbeanField = "mbean";

    private int mbeanIndex = -1;

    /**
     * Source field
     */
    private String sourceField = "sname";

    private int sourceIndex = -1;

    /**
     * 
     */

    /**
     * Time field
     */
    private String timeField = "time";

    private int timeIndex = -1;

    /**
     * TimeFormat
     */
    private boolean isTimeAsLong = true;

    /**
     * Time format used to parse the boundaries of the time interval: - fromTime -
     * toTime It is fixed to the following human readable format: "yyyy/MM/dd
     * HH:mm:ss".
     */
    private SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

    /**
     * Header
     */
    private String header = null;

    /**
     * attributes
     */
    private String[] atts = null;

    private int[] attIndex = null;

    /**
     * Metrics maps
     */
    private HashMap<String, Metric> baseline = new HashMap<String, Metric>();

    private LinkedList<Metric> metrics = new LinkedList<Metric>();

    /**
     * are statistics on ?
     */
    private boolean isStatistics = false;

    /**
     * Array of HashMap: one per attribute The array must be allocated when
     * attributes are set and isStatistics = true
     */
    private ArrayList<HashMap<String, Statistic>> stats = null;

}
