/**
 * JASMINe
 * Copyright (C) 2005-2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Stat.java 9339 2011-11-24 17:31:42Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.File;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeSet;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.management.openmbean.CompositeData;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAp;
import org.ow2.jasmine.monitoring.mbeancmd.JmxHelper;
import org.ow2.jasmine.monitoring.mbeancmd.Outer;
import org.ow2.jasmine.monitoring.mbeancmd.context.StatContext;
import org.ow2.jasmine.monitoring.mbeancmd.graph.Grapher;
import org.ow2.jasmine.monitoring.mbeancmd.graph.conf.Configurator;
import org.ow2.jasmine.monitoring.mbeancmd.jasmine.JasmineConnector;

/**
 * Periodically polls attributes of one or more mbeans on one or more J2EE
 * servers.
 * Extended to support several mbean names identified with aliases, and attributes
 * prefixed with alias names (FTO).
 */
public class Stat extends AbstractCommand {

    protected boolean goOn;

    /**
     */
    public Stat() {
        setOptions();
    }

    /**
     * Tests the Stat implementation.
     *
     * @param args
     *            Arguments to pass to Stat#setArgs.
     */
    public static void main(final String[] args) {
        Stat p = new Stat();
        p.setArgs("stat", args);
        p.exec(null);
    }

    public void stop() {
        goOn = false;
    }

    /**
     * Implementation of inherited abstract method.
     *
     * Will never return except in the case of failure.
     *
     * @see AbstractCommand#exec()
     */
    public int exec(final CommandDispatcher cmdDispatcher) {

        this.cmdDispatcher = cmdDispatcher;

        try {
            parseCommandLine(arguments);
        } catch (Exception e) {
            System.out.println("Options format exception in stat command : " + e.getMessage());
            cmdDispatcher.setFailed(e.getMessage());
            return 1;
        }

        /*
         * out (or pout) is : - the output of the sampler - the input for the
         * Outer to print into System.out or into a file
         *
         * sink is: - the sink of the Outer - the input for the graph, if any
         */
        PipedOutputStream out = new PipedOutputStream();
        pout = new PrintStream(out);

        Outer outer = null;
        try {
            if (isConsoleOption) {
                // output on console forced
                outer = new Outer(out);
                outers.add(outer);
            }

            if (this.outputFilePath != null) {
                // create outer for a file
                if (outer == null) {
                    outer = new Outer(out, new File(outputFilePath));
                    outers.add(outer);
                } else {
                    // output on both file and console
                    Outer consoleOuter = outer;
                    PipedOutputStream sink = new PipedOutputStream();
                    consoleOuter.setSink(sink);
                    outer = new Outer(sink, new File(outputFilePath));
                    outers.add(outer);
                    new Thread(consoleOuter).start();
                }
            } else if (this.jasmineURI == null && outer == null) {
                outer = new Outer(out);
                outers.add(outer);
            }

            if (this.jasmineURI != null) {
                if (outer == null) {
                    outer = JasmineConnector.connect(out, this.jasmineURI);
                } else {
                    Outer oldOuter = outer;
                    outer = JasmineConnector.connect(outer, this.jasmineURI);
                    new Thread(oldOuter).start();
                }
                outers.add(outer);
            }
        } catch (IOException e) {
            logger.error("Cannot set up file or jasmine output :  {0}", e.getMessage(), e);
        }

        if (this.graphDef != null) {
            try {
                PipedOutputStream sink = new PipedOutputStream();
                outer.setSink(sink);
                Grapher gr = new Grapher(sink, this.graphDef);
                gr.start();
            } catch (IOException e) {
                logger.error("Cannot set up graph output :  {0}", e.getMessage(), e);
            }
        }
        new Thread(outer).start();

        try {
            process();
            cmdDispatcher.setStopped();
        } catch (Exception e) {
            cmdDispatcher.setFailed(e.getMessage());
        }
        return 0;
    }

    /**
     * Calls {@link AbstractCommand#help()} and prints the DTD for the graph.
     */
    public void help() {
        super.help();

        System.out.println("\n" + "DTD for graph definitions:\n" + "--------------------------\n\n");
        LineNumberReader ln = Configurator.getDTD();
        if (ln != null) {
            try {
                String line = null;
                while ((line = ln.readLine()) != null) {
                    System.out.println(line);
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            logger.error("Error: DTD not found !");
        }
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see AbstractCommand#summary()
     */
    public String summary() {
        return "Periodically probes MBeans in one or multiple J2EE servers";
    }

    /**
     * Parses the command line arguments into {@link Stat#commandLine}.
     *
     * @param args
     *            Arguments to parse.
     *
     * @throws ParseException
     *             If parsing fails.
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    public void parseCommandLine(final String[] args) throws ParseException, MalformedObjectNameException {
        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);

        this.nameArgs = commandLine.getOptionValues("name");
        if (nameArgs.length == 1) {
            // no alias, set objectName
            setObjectName();
        } else if (nameArgs.length > 1) {
            if (!getAliases()) {
                // Aliases format wrong
                throw new ParseException("-name aliases format exception");
            }
            // Check that mbeanNames are well formed.
            // If not, MalformedObjectNameException is thrown.
            for (String mbeanName : aliases.values()) {
                ObjectName on = ObjectName.getInstance(mbeanName);
            }
        } else {
            throw new ParseException("-name option is empty");
        }

        if (commandLine.hasOption("p")) {
            setPeriod();
        }
        if (commandLine.hasOption("r")) {
            setRefreshPeriod();
        }

        if (commandLine.hasOption("cmdid")) {
            this.cmdid = commandLine.getOptionValue("cmdid");
        }

        if (commandLine.hasOption("graph")) {
            this.graphDef = commandLine.getOptionValue("graph");
        }
        if (commandLine.hasOption("f")) {
            this.outputFilePath = commandLine.getOptionValue("f");
        }
        if (commandLine.hasOption("jasmine")) {
            this.jasmineURI = commandLine.getOptionValue("jasmine");
        }

        this.isConsoleOption = commandLine.hasOption("console");

        if (commandLine.hasOption("s")) {
            this.separator = commandLine.getOptionValue("s");
            Outer.setSeparator(separator);
        }

        if (commandLine.hasOption("rate")) {
            this.rateOption = true;
            this.attsRate = commandLine.getOptionValues("rate");
            if (aliases()) {
                logger.warn("-rate option is ignored because aliases are defined");
            }
        }

        if (commandLine.hasOption("delta")) {
            this.deltaOption = true;
            this.attsDelta = commandLine.getOptionValues("delta");
            if (aliases()) {
                logger.warn("-delta option is ignored because aliases are defined");
            }
        }
        if (commandLine.hasOption("slope")) {
            this.slopeOption = true;
            String[] slopeOptionValues = commandLine.getOptionValues("slope");
            if ((slopeOptionValues.length % 2) != 0) {
                /* option = slope -> number of attributes should be pair */
                throw new MalformedObjectNameException("Command failed at: slope option. Number of attributes should be pair.");
            }
            this.attsSlope = slopeOptionValues;
            if (aliases()) {
                logger.warn("-slope option is ignored because aliases are defined");
            }
        }

        if (commandLine.hasOption("a")) {
            this.attsSimple = commandLine.getOptionValues("a");
            if (this.attsSimple != null) {
                // -a option is used with explicit attribute name arguments
                this.aOption = true;
                if (aliases()) {
                    // process arguments to determine attribute names for each alias,
                    // and subsequently, for each aliased mbean name
                    if (!getMultiAttsSimple()) {
                        throw new ParseException("-a aliase:attName format exception");
                    }
                }
            } else {
                this.allOption = true;
            }
        } else {
            // if neither -a, nor -rate, -delta, -slope, that means all the attributes must be polled
            if (!this.deltaOption && !this.rateOption && !this.slopeOption) {
                this.allOption = true;
            }
        }

        // Get attribute names for rate, delta, slope computing
        if (this.deltaOption || this.rateOption || this.slopeOption) {
            this.oldValues = true;
            // Determine all the attributes concerned by rate, delta, slope computing
            attsRateDelatSlope = new ArrayList<String>();
            if (attsRate != null) {
                for (String att : attsRate) {
                    if (!attsRateDelatSlope.contains(att)) {
                        attsRateDelatSlope.add(att);
                    }
                }
            }
            if (attsDelta != null) {
                for (String att : attsDelta) {
                    if (!attsRateDelatSlope.contains(att)) {
                        attsRateDelatSlope.add(att);
                    }
                }
            }
            if (attsSlope != null) {
                for (String att : attsSlope) {
                    if (!attsRateDelatSlope.contains(att)) {
                        attsRateDelatSlope.add(att);
                    }
                }
            }
        }

        // Get attribute names to poll if not allOption and if aliases are not used
        // (don't change the way to determine the attributes to poll if aliases are not used).
        if (!this.allOption && !aliases()) {
            if (aOption) {
                for (int i = 0; i < attsSimple.length; i++) {
                    this.attsTotal.add(attsSimple[i]);
                }
            }
            if (this.oldValues) {
                for (String att : attsRateDelatSlope) {
                    if (!attsTotal.contains(att)) {
                        this.attsTotal.add(att);
                    }
                }
            }
        }
    }

    /**
     * Parse -a arguments from <code>attsSimple</code> and determine the attributes list for each aliased mbean name.
     */
    private boolean getMultiAttsSimple() {
        for (String aliasAtt : attsSimple) {
            if (",".equals(aliasAtt)) {
                // ignore a , separator
                continue;
            }
            if (!aliasAtt.contains(":") || aliasAtt.indexOf(":") == aliasAtt.length() -1) {
                logger.error("-a argument must have format alias:attName");
                return false;
            }
            String alias = aliasAtt.substring(0, aliasAtt.indexOf(":"));
            String attName = aliasAtt.substring(aliasAtt.indexOf(":") + 1, aliasAtt.length());
            if (!aliases.containsKey(alias)) {
                logger.error("-a argument must have format alias:attName but no alias {0} exists ", alias);
                return false;
            }
            // Get rid of , ending an att name
            if (attName.endsWith(",")) {
                attName = attName.substring(0, attName.length() -1);
            }
            logger.debug("attribut alias : {0} for name {1}", alias, attName);
            String mbeanNamePattern = aliases.get(alias);
            if (!multiAttsSimple.containsKey(mbeanNamePattern)) {
                multiAttsSimple.put(mbeanNamePattern, new ArrayList<String>());
                // insert the alias as the first element of attribute names list
                multiAttsSimple.get(mbeanNamePattern).add(alias);
            }
            multiAttsSimple.get(mbeanNamePattern).add(attName);
        }
        return true;
    }

    /**
     * Initialize <code>aliases</code> map using nameArgs.
     * Verify that format is :
     *     "mbeanName" "as" "alias"
     * @return true if mbean name aliases could be get, false if wrong aliases format
     */
    private boolean getAliases() {
        int i = 0;
        while (i < nameArgs.length) {
            // expecting mbeanName1 as alias1, .., mbeanNameN as aliasN
            String mbeanName = nameArgs[i];
            // get rid of ,
            if (",".equals(mbeanName)) {
                i = i + 1;
                continue;
            }
            try {
                String as = nameArgs[i + 1];
                if (!"as".equals(as)) {
                    // unexpected format
                    return false;
                }
                String alias = nameArgs[i + 2];
                // Get rid of , ending an alias name
                if (alias.endsWith(",")) {
                    alias = alias.substring(0, alias.length() -1);
                }
                if (aliases.containsKey(alias)) {
                    // unexpected format
                    logger.error("-name arguments must contain distinct aliases");
                    return false;
                }
                if (aliases.containsValue(mbeanName)) {
                    // unexpected format
                    logger.error("-name arguments must contain distinct mbean names");
                    return false;
                }
                logger.debug("Got alias {0} for mbean name {1}", alias, mbeanName);
                aliases.put(alias, mbeanName);
            } catch (IndexOutOfBoundsException iobe) {
                // unexpected format
                logger.error("-name arguments count must contain triplets (mbeanName as alias)");
                return false;
            }
            i = i + 3;
        }
        return true;
    }


    /**
     * Sets the object name based on the "name" argument in the command line.
     *
     * @throws MalformedObjectNameException
     *             Object name given in the command line is invalid.
     */
    private void setObjectName() throws MalformedObjectNameException {
        String mbean = commandLine.getOptionValue("name");

        if (mbean.contains("\"")) {

            StringBuffer quotedMbean = new StringBuffer();
            StringTokenizer st = new StringTokenizer(mbean, "\"");
            quotedMbean.append(st.nextToken());
            while (st.hasMoreTokens()) {

                quotedMbean.append(ObjectName.quote(st.nextToken()));

                if (st.hasMoreTokens()) {
                    quotedMbean.append(st.nextToken());
                }
            }
            objectName = new ObjectName(quotedMbean.toString());
        } else {
            objectName = new ObjectName(mbean);
        }
    }



    /**
     * Sets the period if the command line has the "p" option.
     *
     * @throws NumberFormatException
     *             Number after the "p" option is invalid.
     */
    private void setPeriod() throws NumberFormatException {
        period = Long.parseLong(commandLine.getOptionValue("p"));
    }

    /**
     * Sets the refresh period if the command line has the "r" option.
     *
     * @throws NumberFormatException
     *             Number after the "r" option is invalid, note that "never" is
     *             a valid number.
     */
    private void setRefreshPeriod() throws NumberFormatException {
        String val = commandLine.getOptionValue("r");
        if ("never".equalsIgnoreCase(val)) {
            isRefreshable = false;
        } else {
            refreshPeriod = Long.parseLong(val);
        }
    }

    /**
     * Polls the MBeans.
     */
    private void mbeanPoll() throws Exception {

        // Poll in every target the MBeans found by the refresh method (updateOnames)
        goOn = (contexts != null ? true : false);
        boolean refreshNow = false;
        while (goOn) {
            // create one thread per polled server
            for (int i = 0; i < contexts.length; i++) {
                // The polled target is represented by a 'context'
                StatContext context = contexts[i];

                MBeanServerConnection cnx = context.getJmxap().getMBeanServerConnection();
                long now = System.currentTimeMillis();
                refreshNow = isRefreshable && (now >= context.getRefreshDeadLine());
                if (refreshNow) {
                    try {
                        if (objectName != null) {
                            // Default mode, only one mbean name
                            context.updateOnames(objectName);
                        } else {
                            // FTO: aliases used
                            for (String mbeanName : aliases.values()) {
                                context.updateOnames(ObjectName.getInstance(mbeanName));
                            }
                        }
                        context.setRefreshDeadLine(now + this.refreshPeriod * 1000);
                    } catch (Exception e) {
                        // was: continue; !
                        throw e;
                    }
                }
                Iterator<?> it = context.getOnames().iterator();
                int nbOns = context.getOnames().size();
                int itCounter = 0;
                while (it.hasNext()) {
                    long t = System.currentTimeMillis();
                    // The polled object (MBean) is represented by 'onGot'
                    ObjectName onGot = (ObjectName) it.next();
                    try {
                        /*
                         * Workaround for WLS
                         * attl = cnx.getAttributes(onGot, atts);
                         */
                        // A 'currentValues' map is created for each object (onGot) of each target (context)
                        // It gives the value of all the attributes to poll corresponding to the different attribute options.
                        //HashMap<String, Object> currentValues = getAttributeValues(cnx, onGot, context.getName());

                        AttributeList attl = new AttributeList();
                        if (aliases()) {
                            // FTO
                            List<String> atts = null;
                            String key = null;
                            if (multiAttsSimple.containsKey(onGot.toString())) {
                                // onGot OBJECT_NAME was provided by -name option (not a pattern)
                                key = onGot.toString();
                                atts = multiAttsSimple.get(key);
                            } else {
                                // onGot corresponds to a pattern; find this pattern
                                for (String onPattern : multiAttsSimple.keySet()) {
                                    ObjectName on = ObjectName.getInstance(onPattern);
                                    Set<ObjectName> ons = context.getObjectNames(on);
                                    if (ons.contains(onGot)) {
                                        key = onPattern;
                                        atts = multiAttsSimple.get(key);
                                        break;
                                    }
                                }
                            }

                            if (atts == null) {
                                logger.warn("Could not determine attributes to poll for MBean {0} (or no attributes defined by the -a option for this MBean).", onGot.toString());
                            } else {

                                // Get alias and remove it from the attribute names list
                                List<String> copyAtts = new ArrayList<String>(atts);
                                String currentAlias = atts.remove(0);
                                // Get Attributes
                                AttributeList attList = getAttributeValues(cnx, onGot, context.getName(), atts.toArray(new String[atts.size()]));
                                // Restore atts in multiAttsSimple
                                multiAttsSimple.put(key, copyAtts);

                                // Get StatMBeanResult data structure
                                StatMBeanResult res = new StatMBeanResult(currentAlias, onGot, attList);

                                // First iteration over mbeans, construct results
                                if (results != null) {
                                    results.add(res);
                                    if (results.size() == nbOns) {
                                        // all the mbeans treated, set resultsComplete in order to
                                        // construct header
                                        resultsComplete = results;
                                        results = null;
                                    }
                                }

                                printMBeanPollMulti(t, context, attl, onGot, res);
                            }

                        } else {
                            // Default mode, only one mbean name, change nothing !!!
                            // Process options like rate, etc....
                            HashMap<String, Object> currentValues = getAttributeValues(cnx, onGot, context.getName());

                            if (this.aOption) {
                                for (int j = 0; j < attsSimple.length; j++) {
                                    Attribute attribute = new Attribute(attsSimple[j], currentValues.get(attsSimple[j]));
                                    attl.add(attribute);
                                }
                            }
                            if (this.rateOption) {
                                try {
                                    AttributeList attributesRate = getAttributesRate(context, onGot, attsRate, currentValues);
                                    attl.addAll(attributesRate);
                                } catch (MalformedObjectNameException ex) {
                                    ex.printStackTrace();
                                    throw ex;
                                }
                            }
                            if (this.deltaOption) {
                                try {
                                    AttributeList attributesDelta = getAttributesDelta(context, onGot, attsDelta, currentValues);
                                    attl.addAll(attributesDelta);
                                } catch (MalformedObjectNameException ex) {
                                    ex.printStackTrace();
                                    throw ex;
                                }
                            }
                            if (this.slopeOption) {
                                try {
                                    AttributeList attributesSlope = getAttributesSlope(context, onGot, attsSlope, currentValues);
                                    attl.addAll(attributesSlope);
                                } catch (MalformedObjectNameException ex) {
                                    ex.printStackTrace();
                                    throw ex;
                                }
                            }
                            if (this.allOption) {
                                Iterator<String> itAttNames = this.attsTotal.iterator();
                                while (itAttNames.hasNext()) {
                                    String attName = itAttNames.next();
                                    Object attValue = currentValues.get(attName);
                                    Attribute attribute = new Attribute(attName, attValue);
                                    attl.add(attribute);
                                }
                            }

                            if (oldValues) {
                                updateOldValues(onGot, context, currentValues);
                            }
                            currentValues = null;
                            printMBeanPoll(t, context, attl, onGot);
                        }

                        if (!context.isStarted()) {
                            context.setStarted(true);
                        }
                    } catch (Exception e) {
                        logger.error("Error on mbean {0} for {1} - {2}", onGot, context.getName(), e.getMessage());
                        throw e;
                    }
                }
            }

            // Sleep for a while
            try {
                Thread.sleep(period * 1000);
            } catch (InterruptedException e) {
                goOn = false;
            }
        }
        // stop outers
        Iterator<Outer> itOuters = outers.iterator();
        while (itOuters.hasNext()) {
            Outer outer = itOuters.next();
            outer.stop();
        }
        pout.close();
    }

    /**
     * FTO. Get the list of Attributes corresponding to an array of attribute names.
     * @param on the MBean on which attributes are read
     * @param target the name of the target
     * @param atts the attribute names
     * @return the corresponding Attributes list
     * @throws IOException the target not reachable
     * @throws ReflectionException
     * @throws InstanceNotFoundException
     * @throws AttributeNotFoundException
     * @throws MBeanException
     */
    private AttributeList getAttributeValues(final MBeanServerConnection cnx, final ObjectName on, final String target, final String[] atts)
    throws IOException, ReflectionException, InstanceNotFoundException, AttributeNotFoundException, MBeanException {

         // Try to get all attributes in one shot
        try {
            return cnx.getAttributes(on, atts);
        } catch (ArrayIndexOutOfBoundsException e) {
            /*
             * workaround for WLS, if attributes cannot be read in one shot
             * due to IOP trouble which raise an ArrayIndexOutOfBoundsException exception.
             * A warning is printed every day
             */
            if ((warningTrigger % (86400 / period)) == 0) {
                logger.warn("MBeanCmd stat : cannot get attributes in one shot, trying one by one ({0}) for {1}", on, target);
                warningTrigger++;
            }
        }
        AttributeList result = new AttributeList();

        // Try one by one
        for (String attName : atts) {
            Object attValue = cnx.getAttribute(on, attName);
            Attribute att = new Attribute(attName, attValue);
            result.add(att);
        }

        return result;
    }

    /**
     * Construct a Map containing mapping for each attribute name in a working names list, the attribute's value.
     * The working attribute names list is constructed as follows: in the case where all the attributes are polled (allAttributes == true)
     * the list of attributes name is constructed from the polled MBeans metadata. Otherwise, the work list is the given attsName.
     * @param cnx connection for the polled target
     * @param on polled MBean's name
     * @param attsName list of attribute names
     * @param target target's name
     * @return  a map of attributes name to attributes values.
     * @throws AttributeNotFoundException
     * @throws InstanceNotFoundException
     * @throws MBeanException
     * @throws ReflectionException
     * @throws IOException
     */
    private HashMap<String, Object> getAttributeValues(final MBeanServerConnection cnx, final ObjectName on, final String target)
        throws AttributeNotFoundException, InstanceNotFoundException, MBeanException, ReflectionException, IntrospectionException, IOException {

        // If attsTotal is empty, try to initialize it
        if (this.attsTotal.isEmpty()) {
            List<String> allAttsName = new ArrayList<String>();
            try {
                MBeanInfo infos = cnx.getMBeanInfo(on);
                MBeanAttributeInfo[] attInfos =  infos.getAttributes();
                for (int i = 0; i < attInfos.length; i++) {
                    MBeanAttributeInfo attInfo = attInfos[i];
                    String attName = attInfo.getName();
                    allAttsName.add(attName);
                }
            } catch (InstanceNotFoundException e) {
                this.logger.info("Cannot get info about the {0} MBean in target {1}", on, target);
                throw e;
            } catch (IntrospectionException e) {
                this.logger.info("Cannot get info about the {0} MBean in target {1}", on, target);
                throw e;
            } catch (ReflectionException e) {
                this.logger.info("Cannot get info about the {0} MBean in target {1}", on, target);
                throw e;
            } catch (IOException e) {
                this.logger.info("Cannot get info about the {0} MBean in target {1}", on, target);
                throw e;
            }
            this.attsTotal = allAttsName;
            this.logger.info("Attributes list to poll was constructed using the {0} MBean in target {1}", on, target);
        }

        HashMap<String, Object> newValueOfAttributes = new HashMap<String, Object>();
        AttributeList attributes = getAttributeValues(cnx, on, target, toArray(this.attsTotal));
        for (Iterator<Object> it = attributes.iterator(); it.hasNext(); ) {
            Attribute attribute = (Attribute) it.next();
            String attributeName = attribute.getName();
            Object attributeValue = attribute.getValue();
            newValueOfAttributes.put(attributeName, attributeValue);
        }

        return newValueOfAttributes;
    }

    /**
     * Construct a list of javax.management.Attribute objects corresponding to the attributes for which the rate must be calculated.
     * @param context the command's context
     * @param on polled MBean's name
     * @param attsRate attributes name for which the rate must be calculated
     * @param currentValues last polled values of all the attributes
     * @return Each Attribute object has a name = rate_attributeName and the value of the rate is calculated using current value - previous value / period.
     * If there is no previous value (first poll), the value is null.
     * @throws MalformedObjectNameException
     */
    private AttributeList getAttributesRate(final StatContext context, final ObjectName on, final String[] attsRate, final HashMap<String, Object> currentValues) throws MalformedObjectNameException {

        AttributeList result = new AttributeList();

        /*
         * Calculate the rate of each attribute
         */
        for (int i = 0; i < attsRate.length; i++) {
            String attName = attsRate[i];
            Object currentValue = currentValues.get(attName);

            boolean hasNumeriqueValue = currentValue instanceof Integer || currentValue instanceof Double || currentValue instanceof Long || currentValue instanceof Float;
            if (hasNumeriqueValue) {
                String name = PREFIX_RATE + attsRate[i];

                Double value = null;

                Object oldValue = context.getOldValue(on, attName);
                if (oldValue != null) {
                    Double dX = new Double(currentValue.toString()) - new Double(oldValue.toString());
                    Double dT = new Double(this.period);
                    if (dT == 0) {
                        value = Double.MAX_VALUE;
                    } else {
                        /* Rounding dX/dY double with "numberDecimalPlaces" elements after the comma */
                        value = Stat.floor(dX / dT, this.numberDecimalPlaces);
                    }
                }

                Attribute attribute = new Attribute(name, value);
                result.add(attribute);
            } else {
                throw new MalformedObjectNameException("Command failed at: All the attributes of rate option should be numeric. Problem with " + attName + " of MBean " + on.toString());
            }
        }
        return result;
    }

    private void updateOldValues(final ObjectName on, final StatContext context, final HashMap<String, Object> currentValues) {
        for (String attName : attsRateDelatSlope) {
            Object currentValue = currentValues.get(attName);
            // current value becomes old
            context.setOldValue(on, attName, currentValue);
        }
    }

    /**
     * Construct a list of javax.management.Attribute objects corresponding to the attributes for which the delta must be calculated.
     * @param context the command's context
     * @param on polled MBean's name
     * @param attsRate attributes names for which the delta must be calculated
     * @param currentValues last polled values of all the attributes
     * @return Each Attribute object has a name = delta_attributeName and the value =  current value - previous value.
     * If there is no previous value (first poll), the value is null.
     * @throws MalformedObjectNameException
     */
    private AttributeList getAttributesDelta(final StatContext context, final ObjectName on, final String[] attsDelta, final HashMap<String, Object> currentValues) throws MalformedObjectNameException {

        AttributeList result = new AttributeList();

        /*
         * Calculate delta
         */
        for (int i = 0; i < attsDelta.length; i++) {
            String attName = attsDelta[i];
            Object currentValue = currentValues.get(attName);

            boolean hasNumeriqueValue = currentValue instanceof Integer || currentValue instanceof Double || currentValue instanceof Long || currentValue instanceof Float;
            if (hasNumeriqueValue) {
                String name = PREFIX_DELTA + attsDelta[i];
                Object oldValue = context.getOldValue(on, attName);
                Double dX = null;
                if (oldValue != null) {
                    dX = new Double(currentValue.toString()) - new Double(oldValue.toString());
                }
                Attribute attribute = new Attribute(name, dX);
                result.add(attribute);
            } else {
                throw new MalformedObjectNameException("Command failed at: All the attributes of delta option should be numeric : " + attName + ".");
            }
        }
        return result;
    }

    /**
     * Construct a list of javax.management.Attribute objects corresponding to the pair of attributes for which the slope must be calculated.
     * @param context the command's context
     * @param on polled MBean's name
     * @param attsRate attribute name pairs (attName1, attName2) for which the delta must be calculated
     * @param currentValues last polled values of all the attributes
     * @return Each Attribute object has a name = attName1_per_attName2 and the value =  delta(attName1) / delta(attName2).
     * If there is no previous value (first poll), the value is null.
     * @throws MalformedObjectNameException
     */
    private AttributeList getAttributesSlope(final StatContext context, final ObjectName on, final String[] attsSlope, final HashMap<String, Object> currentValues) throws MalformedObjectNameException {

        AttributeList result = new AttributeList();

        /*
         * Calculate the slope
         */
        for (int i = 0; i < (attsSlope.length - 1); i += 2) {
            String attName1 = attsSlope[i];
            String attName2 = attsSlope[i + 1];
            Object currentValue1 = currentValues.get(attName1);
            Object currentValue2 = currentValues.get(attName2);

            boolean hasNumeriqueValue1 = currentValue1 instanceof Integer || currentValue1 instanceof Double || currentValue1 instanceof Long || currentValue1 instanceof Float;
            boolean hasNumeriqueValue2 = currentValue2 instanceof Integer || currentValue2 instanceof Double || currentValue2 instanceof Long || currentValue2 instanceof Float;

            if (hasNumeriqueValue1 && hasNumeriqueValue2) {
                String name = attName1 + INFIX_SLOPE + attName2;

                Object oldValue1 = context.getOldValue(on, attName1);
                Object oldValue2 = context.getOldValue(on, attName2);
                Double value = null;
                if (oldValue1 != null && oldValue2 != null) {
                    Double dX = new Double(currentValue1 + "") - new Double(oldValue1 + "");
                    Double dY = new Double(currentValue2 + "") - new Double(oldValue2 + "");
                    if (dX == 0) {
                        value = new Double(0);
                    } else if (dY == 0) {
                        value = Double.MAX_VALUE;
                    } else {
                        /* Rounding dX/dY double with "numberDecimalPlaces" elements after the comma */
                        value = Stat.floor(dX / dY, this.numberDecimalPlaces);
                    }
                }
                Attribute attribute = new Attribute(name, value);
                result.add(attribute);
            } else {
                if (!hasNumeriqueValue1 && !hasNumeriqueValue2) {
                    throw new MalformedObjectNameException("Command failed at: All the attributes of slope option should be numeric : " + attName1 + "," + attName2 + ".");
                } else if (!hasNumeriqueValue1) {
                    throw new MalformedObjectNameException("Command failed at: All the attributes of slope option should be numeric : " + attName1 + ".");
                } else {
                    throw new MalformedObjectNameException("Command failed at: All the attributes of slope option should be numeric : " + attName2 + ".");
                }
            }
        }
        return result;
    }

    /**
     * Try to guess attributes of MBean(s).
     * @param context the target on which we are searching
     * @param on the name or the pattern of the MBean(s) we are looking for
     */
    private String[] getAttsToPoll(final StatContext context, final ObjectName on) throws IOException {
        String[] atts = null;
        ObjectName onGot = null;
        Iterator<?> it = null;
        try {
            it = context.getMBeanServerConnection().queryNames(on, null).iterator();
        } catch (IOException e) {
            logger.error("Cannot get MBeans using {0} in target {1}", on.toString(), context.getName());
            throw e;
        }
        while (it.hasNext()) {
            // PhD Take only the first one ?
            onGot = (ObjectName) it.next();
            MBeanAttributeInfo[] info;
            try {
                info = context.getMBeanServerConnection().getMBeanInfo(onGot).getAttributes();
                atts = new String[info.length];
                for (int i = 0; i < atts.length; i++) {
                    atts[i] = info[i].getName();
                }
                break;
            } catch (Exception e) {
                logger.error("Cannot get attributes for MBean {0} in target {1}", onGot.toString(), context.getName());
            }
        }
        if (atts == null) {
            logger.warn("No attribute found for MBeans using {0} in target {1}", on.toString(), context.getName());
            atts = new String[0];
        }
        return atts;
    }

    /**
     * Prints a given header.
     *
     * @param attl
     *            Header attributes.
     *
     * @return Printed header.
     */
    private String getMBeanPollHeader(final AttributeList attl) {
        if (header == null) {
            String headerBase = getHeaderBase();
            // Get the names in the header
            StringBuffer buf = new StringBuffer(512);
            Iterator<?> it = attl.iterator();
            while (it.hasNext()) {
                buf.append(separator);
                Attribute att = (Attribute) it.next();
                if ((att.getValue() != null) && att.getValue().getClass().isArray()) {
                    putArrayIntoHeader(buf, att);
                } else if ((att.getValue() != null) && (att.getValue() instanceof Map)) {
                    putMapIntoHeader(buf, att);
                } else if ((att.getValue() != null) && (att.getValue() instanceof CompositeData)) {
                    putCompositeDataIntoHeader(buf, att);
                } else {
                    buf.append(att.getName());
                }
            }
            // append names to header base
            header = headerBase + buf.toString();
        }
        return header;
    }

    /**
     * Construct first part of the header.
     * @return the header base
     */
    private String getHeaderBase() {
        StringBuffer buf = new StringBuffer(512);
        buf.append("date");
        buf.append(separator);
        buf.append("time");
        buf.append(separator);
        buf.append("sname");
        buf.append(separator);
        buf.append("server");
        buf.append(separator);
        buf.append("domain");
        buf.append(separator);
        buf.append("mbean");
        buf.append(separator);
        buf.append("cmdid");
        return buf.toString();
    }
    /**
     * Puts a list of attributes into a buffer.
     *
     * @param buf
     *            String buffer to print into.
     * @param att
     *            Attribute list.
     */

    private void putArrayIntoHeader(final StringBuffer buf, final Attribute att) {
        Object obj = att.getValue();
        int length = Array.getLength(obj);
        for (int i = 0; i < length; i++) {
            buf.append((i == 0 ? "" : separator));
            buf.append(att.getName());
            buf.append("_");
            buf.append(i);
        }
    }

    /**
     * Puts a map of attributes into a buffer.
     *
     * @param buf
     *            String buffer to print into.
     * @param att
     *            Attribute map.
     */
    private void putMapIntoHeader(final StringBuffer buf, final Attribute att) {
        Map<?, ?> map = (Map<?, ?>) att.getValue();
        TreeSet<?> ts = new TreeSet<Object>(map.keySet());
        Iterator<?> it = ts.iterator();
        String sep = "";
        while (it.hasNext()) {
            buf.append(sep);
            sep = separator;
            buf.append(att.getName());
            buf.append("_");
            buf.append(it.next());
        }
    }

    /**
     * Puts composite data into a buffer.
     *
     * @param buf
     *            String buffer to print into.
     * @param att
     *            Attribute data.
     */
    private void putCompositeDataIntoHeader(final StringBuffer buf, final Attribute att) {
        CompositeData cdata = (CompositeData) att.getValue();
        Set<?> itemNames = cdata.getCompositeType().keySet();
        Iterator<?> it = itemNames.iterator();
        String sep = "";
        while (it.hasNext()) {
            buf.append(sep);
            sep = separator;
            buf.append(att.getName());
            buf.append("_");
            buf.append(it.next());
        }
    }

    /**
     * FTO. Construct header using a list of StatMBeanResult objects.
     * Each StatMBeanResult corresponds to attributes read for one of the MBeans
     * found based on the patterns (-name option). The MBeans are determined by
     * connecting to the first target.
     * The StatMBeanResults are used to construct a list of HeaderAtt instances
     * that are used when printing each result line, in order to determined
     * for to which element in the header the result value corresponds.
     * @return header
     */
    private String getMultiHeader() {

        List<StatMBeanResult> onceAliasResults = new ArrayList<StatMBeanResult>();
        for (String alias : aliases.keySet()) {
            // get a result with current alias
            StatMBeanResult currentRes = null;
            for (StatMBeanResult res : resultsComplete) {
                if (res.getAlias().equals(alias)) {
                    currentRes = res;
                    break;
                }
            }
            if (currentRes != null) {
                onceAliasResults.add(currentRes);
            }
        }

        for (StatMBeanResult res : onceAliasResults) {
            String alias = res.getAlias();
            AttributeList al = res.getAttl();
            for (int i = 0; i < al.size(); i++) {
                Attribute a = (Attribute) al.get(i);
                Object val = a.getValue();
                String attName = null;
                if ((val != null) && val.getClass().isArray()) {
                    int length = Array.getLength(val);
                    for (int j = 0; j < length; j++) {
                        attName = a.getName() + "_" + j;
                        HeaderAtt ha = new HeaderAtt(alias, attName);
                        hatts.add(ha);
                    }
                } else if ((val != null) && (val instanceof Map)) {
                    Map<?, ?> map = (Map<?, ?>) val;
                    TreeSet<?> ts = new TreeSet<Object>(map.keySet());
                    Iterator<?> it = ts.iterator();
                    while (it.hasNext()) {
                        attName = a.getName() + "_" + it.next();
                        HeaderAtt ha = new HeaderAtt(alias, attName);
                        hatts.add(ha);
                    }
                } else if ((val != null) && (val instanceof CompositeData)) {
                    CompositeData cdata = (CompositeData) val;
                    Set<?> itemNames = cdata.getCompositeType().keySet();
                    Iterator<?> it = itemNames.iterator();
                    String sep = "";
                    while (it.hasNext()) {
                        attName = a.getName() + "_" + it.next();
                        HeaderAtt ha = new HeaderAtt(alias, attName);
                        hatts.add(ha);
                    }
                } else {
                    attName = a.getName();
                    HeaderAtt ha = new HeaderAtt(alias, attName);
                    hatts.add(ha);
                }
            }
        }

        // Get the names in the header
        StringBuffer buf = new StringBuffer(512);
        for (HeaderAtt ha : hatts) {
            buf.append(separator);
            buf.append(ha.getAlias());
            buf.append(":");
            buf.append(ha.getAttName());
        }

        String headerBase = getHeaderBase();
        // append names to header base
        header = headerBase + buf.toString();
        return header;
    }

    /**
     * Prints out the poll data based on a header.
     *
     * @param t
     *            Time.
     * @param context
     *            Statistics context.
     * @param attl
     *            Attribute list.
     * @param oname
     *            Object to print as data.
     */
    private void printMBeanPollMulti(final long t, final StatContext context, final AttributeList attl, final ObjectName oname, final StatMBeanResult result) {
        if (header == null) {
            // check that the first iteration is completed
            if (resultsComplete != null) {
                // call getMultiHeader() to construct the header
                // and print it out.
                pout.println(getMultiHeader());

                // print a line for each result in resultsComplete
                for (StatMBeanResult res : resultsComplete) {
                    printResult(t, context, res);
                }
            }
        } else {
            printResult(t, context, result);
            // get rid of resultsComplete, not needed anymore
            resultsComplete = null;

        }
    }

    /**
     * Print a result line with values in a StatMBeanResult instance.
     * Use <code>hatts</code> list of HeaderAtt instances to identify
     * make correspondence between the header names and the values from res.
     * @param t
     * @param context
     * @param res
     */
    private void printResult(final long t, final StatContext context, final StatMBeanResult res) {
        // Construct the array of values
        Object[] vals = new Object[hatts.size()];
        int k = 0;
        for (HeaderAtt ha : this.hatts) {
            if (ha.getAlias().equals(res.getAlias())) {
                // attribute name to look for
                String attName = ha.getAttName();
                AttributeList al = res.getAttl();
                boolean found = false;
                for (int i = 0; i < al.size(); i++) {
                    Attribute a = (Attribute) al.get(i);
                    String aName = a.getName();
                    Object aValue = a.getValue();
                    if (aName.equals(attName)) {
                        vals[k] = aValue;
                        found = true;
                        break;
                    } else {
                        // check that aValue has a complex type
                        if ((aValue != null) && aValue.getClass().isArray()) {
                            int length = Array.getLength(aValue);
                            boolean foundInArray = false;
                            for (int j = 0; j < length; j++) {
                                Object aValueElem = Array.get(aValue, j);
                                String aNameElem = aName +  "_" + j;
                                if (aNameElem.equals(attName)) {
                                    vals[k] = aValueElem;
                                    foundInArray = true;
                                    break;
                                }
                            }
                            if (foundInArray) {
                                found = true;
                                break;
                            }
                        } else if ((aValue != null) && (aValue instanceof Map)) {
                            boolean foundInMap = false;
                            Map<?, ?> map = (Map<?, ?>) aValue;
                            TreeSet<?> ts = new TreeSet<Object>(map.keySet());
                            Iterator<?> it = ts.iterator();
                            while (it.hasNext()) {
                                String key = (String) it.next();
                                Object aValueElem = map.get(key);
                                String aNameElem = aName +  "_" + key;
                                if (aNameElem.equals(attName)) {
                                    vals[k] = aValueElem;
                                    foundInMap = true;
                                    break;
                                }
                            }
                            if (foundInMap) {
                                found = true;
                                break;
                            }
                        } else if ((aValue != null) && (aValue instanceof CompositeData)) {
                            boolean foundInComp = false;
                            CompositeData cdata = (CompositeData) aValue;
                            Set<?> itemNames = cdata.getCompositeType().keySet();
                            Iterator<?> it = itemNames.iterator();
                            while (it.hasNext()) {
                                String key = (String) it.next();
                                Object aValueElem = cdata.get(key);
                                String aNameElem = aName +  "_" + key;
                                if (aNameElem.equals(attName)) {
                                    vals[k] = aValueElem;
                                    foundInComp = true;
                                    break;
                                }
                            }
                            if (foundInComp) {
                                found = true;
                                break;
                            }
                        }
                    }
                }
                if (!found) {
                    vals[k] = null;
                }
            } else {
                vals[k] = null;
            }
            k++;
        }

        Date d = new Date(t);
        pout.print(SIMPLEDATEFORMAT.format(d));
        pout.print(separator);
        pout.print(t);
        pout.print(separator);
        pout.print(context.getJmxUrl());
        pout.print(separator);
        pout.print(context.getServer());
        pout.print(separator);
        pout.print(context.getDomain());
        pout.print(separator);
        pout.print(res.getOn());
        pout.print(separator);
        pout.print(cmdid);
        for (Object val : vals) {
            pout.print(separator);
            if (val != null) {
                pout.print(val);
            }
            // if value null, nothing to be printed out
        }
        pout.println();
    }
    /**
     * Prints out the poll data based on a header.
     *
     * @param t
     *            Time.
     * @param context
     *            Statistics context.
     * @param attl
     *            Attribute list.
     * @param oname
     *            Object to print as data.
     */
    private void printMBeanPoll(final long t, final StatContext context, final AttributeList attl, final ObjectName oname) {
        if (header == null) {
            pout.println(this.getMBeanPollHeader(attl));
        }

        Date d = new Date(t);
        pout.print(SIMPLEDATEFORMAT.format(d));
        pout.print(separator);
        pout.print(t);
        pout.print(separator);
        pout.print(context.getJmxUrl());
        pout.print(separator);
        pout.print(context.getServer());
        pout.print(separator);
        pout.print(context.getDomain());
        pout.print(separator);
        pout.print(oname);
        pout.print(separator);
        pout.print(cmdid);
        Iterator<?> it = attl.iterator();
        while (it.hasNext()) {
            pout.print(separator);
            Attribute att = (Attribute) it.next();
            if ((att.getValue() != null) && att.getValue().getClass().isArray()) {
                Object obj = att.getValue();
                int length = Array.getLength(obj);
                for (int i = 0; i < length; i++) {
                    pout.print((i == 0 ? "" : separator) + Array.get(obj, i));
                }
            } else if ((att.getValue() != null) && (att.getValue() instanceof Map)) {
                Map<?, ?> map = (Map<?, ?>) att.getValue();
                TreeSet<?> ts = new TreeSet<Object>(map.keySet());
                String sep = "";
                Iterator<?> it2 = ts.iterator();
                while (it2.hasNext()) {
                    pout.print(sep);
                    sep = separator;
                    pout.print(map.get(it2.next()));
                }
            } else if ((att.getValue() != null) && (att.getValue() instanceof CompositeData)) {
                CompositeData cdata = (CompositeData) att.getValue();
                Set<?> itemNames = cdata.getCompositeType().keySet();
                Iterator<?> it2 = itemNames.iterator();
                String sep = "";
                while (it2.hasNext()) {
                    pout.print(sep);
                    sep = separator;
                    pout.print(cdata.get((String) it2.next()));
                }
            } else if (att.getValue() != null) {
                pout.print(att.getValue());
            } else {
                // value null, nothing to be printed out
            }
        }
        pout.println();
    }

    /**
     * Sets all options that are parseable from the command line.
     *
     * @see Stat#options
     */
    private void setOptions() {
        options = new Options();
        options.addOption("v", "verbose", false, "Verbose mode");

        // ObjectName
        Option name = new Option("name", "objectName", true, "An ObjectName passed to every method");
        name.setRequired(true);
        name.setArgName("mbean name");
        //name.setArgs(1);
        name.setArgs(Option.UNLIMITED_VALUES);
        options.addOption(name);

        // Optional attribute list
        Option atts = new Option("a", "atts", false, "attributes to poll");
        atts.setRequired(false);
        atts.setOptionalArg(true);
        atts.setArgs(Option.UNLIMITED_VALUES);
        atts.setArgName("attributes");
        options.addOption(atts);

        // Period
        Option period = new Option("p", "period", true, "Polling period");
        period.setRequired(false);
        period.setArgName("period");
        period.setArgs(1);
        options.addOption(period);

        // MBean list refresh period
        Option refresh = new Option("r", "refresh", true, "Period for rebuilding the mbean list");
        refresh.setRequired(false);
        refresh.setArgName("refreshPeriod");
        refresh.setArgs(1);
        options.addOption(refresh);

        // graph output
        Option graph = new Option("graph", "graph", true, "Enable graphical output");
        graph.setRequired(false);
        graph.setArgName("graphDefinition");
        graph.setArgs(1);
        options.addOption(graph);

        // console output
        Option console = new Option("console", "console", false, "Send output to stdout");
        console.setRequired(false);
        options.addOption(console);

        // file output
        Option file = new Option("f", "file", true, "Send output to file");
        file.setRequired(false);
        file.setArgName("path");
        file.setArgs(1);
        options.addOption(file);

        // jasmine connector output
        Option jasmine = new Option("jasmine", "jasmine", true, "Output logged data to jasmine (will disable stdout)");
        jasmine.setRequired(false);
        jasmine.setArgName("jasmineURI");
        jasmine.setArgs(1);
        options.addOption(jasmine);

        // Field separator
        Option separator = new Option("s", "separator", true, "Set the column separator");
        separator.setRequired(false);
        separator.setArgName("separator");
        separator.setArgs(1);
        options.addOption(separator);

        // cmd id
        Option cmdid = new Option("cmdid", "cmdid", true, "Cmd Identifier");
        cmdid.setRequired(false);
        cmdid.setArgName("cmdid");
        cmdid.setArgs(1);
        options.addOption(cmdid);

        // Optional attribute target
        Option target = new Option("target", "target", true, "instances to poll");
        target.setRequired(false);
        target.setOptionalArg(true);
        target.setArgs(Option.UNLIMITED_VALUES);
        target.setArgName("instances");
        options.addOption(target);

        // Optional attribute rate list
        Option rate = new Option("rate", "rate", false, "attributes rate to poll");
        rate.setRequired(false);
        rate.setOptionalArg(true);
        rate.setArgs(Option.UNLIMITED_VALUES);
        rate.setArgName("rate");
        options.addOption(rate);

        // Optional attribute delta list
        Option delta = new Option("delta", "delta", false, "attributes delta to poll");
        delta.setRequired(false);
        delta.setOptionalArg(true);
        delta.setArgs(Option.UNLIMITED_VALUES);
        delta.setArgName("delta");
        options.addOption(delta);

        // Optional attribute slope list
        Option slope = new Option("slope", "slope", false, "attributes slope to poll");
        slope.setRequired(false);
        slope.setOptionalArg(true);
        slope.setArgs(Option.UNLIMITED_VALUES);
        slope.setArgName("slope");
        options.addOption(slope);
    }

    /**
     * Initializes data polling and calls {@link Stat#mbeanPoll()}.
     */
    private void process() throws Exception {
        String[] targetOption = commandLine.getOptionValues("target");
        logger.debug("Process -target option");

        String[] targets = JmxHelper.getJmxTargets(targetOption);

        if (targets == null) {
            String defaultTarget = JmxHelper.getDefaultTarget();
            if (defaultTarget != null) {
                targets = new String[1];
                targets[0] = defaultTarget;
            }
        }

        if (targets.length > 0) {
            // Construct target contexts
            contexts = new StatContext[targets.length];
            boolean targetReached = true;
            for (int i = 0; i < contexts.length; i++) {
                // If oldValues=true, old attribute values are managed in StatContext
                contexts[i] = new StatContext(oldValues);
                String targetName = targets[i];
                contexts[i].setName(targetName);
                String jmxUrl = JmxHelper.getJmxUrl(targetName);
                contexts[i].setJmxUrl(jmxUrl);
                contexts[i].setJmxap(new JmxAp(jmxUrl, this.cmdDispatcher));
                contexts[i].setDomain("unknown_domain");
                contexts[i].setServer("unknown_server_name");
                try {
                    ObjectName j2eeServer = ObjectName.getInstance("*:j2eeType=J2EEServer,*");
                    ObjectName j2eeDomain = ObjectName.getInstance("*:j2eeType=J2EEDomain,*");
                    MBeanServerConnection mbscnx = contexts[i].getJmxap().getMBeanServerConnection();
                    Iterator<?> onames = mbscnx.queryNames(j2eeServer, null).iterator();
                    boolean isJ2eeServer = false;
                    while (onames.hasNext()) {
                        ObjectName on = (ObjectName) onames.next();
                        contexts[i].setServer(on.getKeyProperty("name"));
                        contexts[i].setDomain(on.getDomain());
                        isJ2eeServer = true;
                        break;
                    }
                    if (!isJ2eeServer) {
                        // try with J2EEDomain MBean
                        onames = mbscnx.queryNames(j2eeDomain, null).iterator();
                        while (onames.hasNext()) {
                            ObjectName on = (ObjectName) onames.next();
                            contexts[i].setDomain(on.getDomain());
                        }
                    }
                    // In order to use server name, isAdminNaming must be set to false
                    // Currently this choice is made only when using a target name = jmx url
                    if (jmxUrl.equals(targetName)) {
                        contexts[i].setAdminNaming(false);
                    }
                } catch (Exception e) {
                    if (e instanceof IOException) {
                        logger.warn("Cannot reach target:" + e);
                        targetReached = false;
                    }
                } finally {
                    contexts[i].getJmxap().releaseMBeanServerConnection();
                }
                // Determine attributes to poll in case they must be 'guessed'
                if (aOption) {
                    if (aliases()) {
                        // FTO
                        for (String mbeanNamePattern : multiAttsSimple.keySet()) {
                            List<String> attNames = multiAttsSimple.get(mbeanNamePattern);
                            if (attNames.contains("*")) {
                                // at least one "*", this means all the attributes must be read
                                // need to 'guess' the attributes
                                String[] foundAtts = getAttsToPoll(contexts[i], ObjectName.getInstance(mbeanNamePattern));
                                // recreate att names list
                                List<String> newAttNames = new ArrayList<String>();
                                // add alias
                                newAttNames.add(attNames.get(0));
                                for (int j = 0; j < foundAtts.length; j++) {
                                    newAttNames.add(foundAtts[j]);
                                }
                                // replace old list by new list
                                multiAttsSimple.put(mbeanNamePattern, newAttNames);
                                attNames = null;
                            }
                        }
                    } else {
                        // No aliases defined, don't change anything...
                        if (aOption && (attsSimple == null || attsSimple.length == 0)) {
                            attsSimple = getAttsToPoll(contexts[i], objectName);
                        }
                    }
                }

                logger.info("Target {0} - {1}", contexts[i].getName(), contexts[i].getJmxUrl());
            }
            if (targetReached) {
                cmdDispatcher.setRunning();
            }
            mbeanPoll();
        } else {
            logger.error("Could not found targets. Use probe-config.xml file to define targets, or define jasmine.jmx.url for default target url definition.");
            throw new Exception("No target defined");
        }
    }

    /**
     * Rounding a double with n elements after the comma.
     * @param a The value to convert.
     * @param n The number of decimal places to retain.
     * @return Value rounded to n decimal places.
     */
    private static double floor(double a, int n) {
        double p = Math.pow(10.0, n);
        return Math.floor((a * p) + 0.5) / p;
    }

    /**
     * Convert ArrayList of String to Array of String
     * @param arrayList
     * @return array
     */
    private static String[] toArray(List<String> arrayList) {
        int size = arrayList.size();
        String[] array = new String[size];
        for (int i = 0; i < size; i++) {
            array[i] = arrayList.get(i);
        }
        return array;
    }
    /**
     * Statistics contexts.
     */
    private StatContext[] contexts = null;
    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;
    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    /**
     * ObjectName to use to determine MBeans to poll.
     * FTO. Rest equal to null if aliases are used.
     */
    private ObjectName objectName = null;

    /**
     * Polling period in seconds.
     */
    private long period = 10;
    /**
     * Refresh period, in seconds. A value of 0 means refresh at each poll
     */
    private long refreshPeriod = 300;
    /**
     * Is content refresheable ?
     */
    private boolean isRefreshable = true;
    /**
     * Graph definitions, null for no graph output.
     */
    private String graphDef = null;
    /**
     * Ident of the cmd
     */
    private String cmdid = null;
    /**
     * Output file path, null for no file output.
     */
    private String outputFilePath = null;
    /**
     * URI of the JASMINe event switch, null for no JASMINe output.
     */
    private String jasmineURI = null;
    /**
     * Output stream.
     */
    private PrintStream pout = null;
    /**
     * If true, the output to console is forced.
     */
    private boolean isConsoleOption = false;
    /**
     * Separator to use.
     */
    private String separator = ";";
    /**
     * Header to use.
     */
    private String header = null;
    /**
     * Command dispatcher instance.
     */
    private CommandDispatcher cmdDispatcher = null;
    /**
     * IOP trouble workarround : Warning counter
     */
    private long warningTrigger = 0;

    /**
     * Becomes true if -a option is used, and in this case the attsSimple is initialized with
     * the list of the attributes to poll and to return their value as it is.
     */
    private boolean aOption = false;

    /**
     * Becomes true if no -a option, and neither -rate, -delta, -slope options are used.
     * Also true if -a option with no value, with or without on of -rate, -delta, -slope.
     */
    private boolean allOption = false;
    /**
     * Becomes true if -rate option is used.
     * To calculate number of "task" per second
     */
    private boolean rateOption = false;
    /**
     * Becomes true if -delta option is used.
     * To calculate the number of "task" per period
     */
    private boolean deltaOption = false;
    /**
     *
     * Becomes true if -slope option is used.
     * To calculate the change in the X coordinate divided by the corresponding change in the Y coordinate
     * dX/dY
     */
    private boolean slopeOption = false;

    /**
     * Becomes true if one of rateOption, deltaOption or slopeOption is true.
     * Otherwise is false, that means we don't have to keep old attribute values.
     */
    private boolean oldValues = false;

    /**
     * The attributes to poll specified by -a option, but also -rate, -delta and -slope options.
     * It may be 'guessed' from the MBean (based on objectName) in some cases (for example, if all attributes need to be polled).
     * (this 'guess' approach its not perfect when several targets are used (more then one context in <code>contexts</code>)
     * and when MBean name is a pattern).
     */
    private String[] attsSimple = null;

    /**
     * Maps a mbean pattern to the attributes names to be read from mbeans having their
     * ObjectName compatible with the pattern.
     * The first element in the String list is the alias corresponding to the pattern.
     * Then, there is one element equal to *, if all attributes have to be read,
     * or there are several elements, each element corresponding to an alias:element in the -a option.
     */
    private Map<String, List<String>> multiAttsSimple = new HashMap<String, List<String>>();

    private List<StatMBeanResult> results = new ArrayList<StatMBeanResult>();

    private List<StatMBeanResult> resultsComplete = null;

    List<HeaderAtt> hatts = new ArrayList<HeaderAtt>();

    /**
     * The value or values of -name option.
     * FTO : replaces <code>name</code>.
     */
    private String[] nameArgs = null;

    /**
     * Maps alias names on mbean names provided by command arguments.
     */
    private Map<String, String> aliases = new HashMap<String, String>();

    /**
     * Allows to know if aliases are used (multiple mbean names are used).
     * @return true if aliases are defined
     */
    private boolean aliases() {
        if (aliases.isEmpty()) {
            return false;
        }
        return true;
    }

    /** The attributes name for which rate is to be calculated
     */
    private String[] attsRate = null;

    /** The attributes name for which delta is to be calculated
     */
    private String[] attsDelta = null;

    /** The attributes name for which slope is to be calculated
     */
    private String[] attsSlope = null;

    /**
     * Attribute names for which one of the delta, rate or slope is to be calculated.
     */
    private List<String> attsRateDelatSlope = null;

    /**
     * Total list of attribute names to poll. This list is used to construct the header message.
     * This list can't be null. If no options are used (or -a without value), then this list must be constructed based
     * on one or several MBeans attributes (depending on the chosen strategy).
     * Currently, it is based on the first MBean found in the first target. Should be calculated for each target, and kept in the context,
     * but this is not compatible with the outer current implementation, which supports only one possible header.
     */
    private List<String> attsTotal = new ArrayList<String>();

    /**
     * List ouf outers for the command.
     */
    private List<Outer> outers = new ArrayList<Outer>();

    /**
     * The number of decimal places to retain
     */
    private int numberDecimalPlaces = 16;

    /**
     * Prefix used to construct names for indicators corresponding to the -rate option
     */
    private static final String PREFIX_RATE = "rate_";

    /**
     * Prefix used to construct names for indicators corresponding to the -delta option
     */
    private static final String PREFIX_DELTA = "delta_";


    /**
     * Used to construct names for indicators corresponding to the -slope att1 att2 option.
     * The indicators name is att1_per_att2.
     */
    private static final String INFIX_SLOPE = "_per_";
}
