/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Configurator.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.graph.conf;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;

import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.dom4j.util.XMLErrorHandler;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Reads the configuration from the XML configuration file.
 */
public class Configurator extends Constants implements Configuration {
    /**
     * Tests the Configurator implementation.
     *
     * @param args  Ignored.
     */
    public static void main(final String[] args) {
        Configuration cf = new Configurator();
        try {
            cf.loadConfig("./src/xml/graph2.xml");
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        String[] s = (String[]) cf.getSerieIds();
        String[] g = (String[]) cf.getGraphIds();
        GraphConfig gc = cf.getGraphConfig("g1");
        s = gc.getSerieIds();
        System.out.println(g[0]);
        System.out.println(gc.getLegend(s[1]));
        System.out.println(cf.getSerieConfig(s[1]).getTitle());
        System.out.println("Absciss=" + cf.getAbscissColumn());
        System.out.println("separator=" + cf.getSeparator());
        System.out.println("dateFormat=" + cf.getDateFormat().toPattern());
    }

    /**
     * Implementation of inherited method.
     *
     * @see Configuration#getSerieConfig
     */
    public SerieConfig getSerieConfig(final String id) {
        return (SerieConfig) series.get(id);
    }

    /**
     * Implementation of inherited method.
     *
     * @see Configuration#getSerieIds
     */
    public String[] getSerieIds() {
        String[] s = new String[1];
        return (String[]) series.keySet().toArray(s);
    }

    /**
     * Implementation of inherited method.
     *
     * @see Configuration#getGraphConfig
     */
    public GraphConfig getGraphConfig(final String id) {
        return (GraphConfig) graphs.get(id);
    }

    /**
     * Implementation of inherited method.
     *
     * @see Configuration#getGraphIds
     */
    public String[] getGraphIds() {
        String[] s = new String[1];
        return (String[]) graphs.keySet().toArray(s);
    }

    /**
     * Implementation of inherited method.
     *
     * @see Configuration#loadConfig
     */
    public void loadConfig(final String conf) throws IOException {
        String configuration;
        if (conf == null) {
            configuration = System.getProperty(Constants.GRAPH_CONFIG_PATH);
        } else {
            configuration = conf;
        }

        SAXReader reader = new SAXReader();
        reader.setValidation(true);
        reader.setStripWhitespaceText(false);
        reader.setEntityResolver(new MyEntityResolver());

        // add error handler which turns any errors into XML
        reader.setErrorHandler(new XMLErrorHandler());

        // parse the document
        try {
            doc = reader.read(configuration);
        } catch (DocumentException e) {
            if (e.getNestedException() instanceof java.io.FileNotFoundException) {
                System.out.println("FileNotFoundException for configuration file: " + configuration);
                throw (java.io.FileNotFoundException) e.getNestedException();
            }
            e.printStackTrace();
        }

        // build configuration data
        parseParent();
        parseSeries();
        parseGraphs();
    }

    /**
     * @return  The line number reader for the XML DTD.
     */
    public static LineNumberReader getDTD() {
        InputStream dtd= Thread.currentThread().getContextClassLoader().getResourceAsStream("xml/graph.dtd");
        return new LineNumberReader(new InputStreamReader(dtd));
    }

    /**
     * Implementation of inherited method.
     *
     * @see Configuration#getSeparator
     */
    public String getSeparator() {
        return separator;
    }

    /**
     * Implementation of inherited method.
     *
     * @see Configuration#getAbscissColumn
     */
    public String getAbscissColumn() {
        return this.abscissColumn;
    }

    /**
     * Implementation of inherited method.
     *
     * @see Configuration#getDateFormat
     */
    public SimpleDateFormat getDateFormat() {
        return simpleDateFormat;
    }

    /**
     * Entity resolver.
     */
    private class MyEntityResolver implements EntityResolver {
        /**
         * Implementation of inherited method.
         *
         * @see EntityResolver#resolveEntity
         */
        public InputSource resolveEntity(final String publicId, final String systemId) throws SAXException, IOException {
            InputSource source = null;
            if (publicId.equals("-//OW2//DTD JASMINe MBeanCmd 1.0//EN")) {
                URL dtd= Thread.currentThread().getContextClassLoader().getResource("xml/graph.dtd");
                source = new InputSource(dtd.toString());
            }
            return source;
        }
    }

    /**
     * Parses the series based on the XML configuration.
     */
    private void parseSeries() {
        Iterator it = doc.selectNodes("//serie-def").iterator();
        while (it.hasNext()) {
            addSerie((Node) it.next());
        }
    }

    /**
     * Adds a series.
     *
     * @param node  XML node describing the series.
     */
    private void addSerie(final Node node) {
        SerieConfig sc = new SerieConfig(this);

        // ID
        String id = node.selectSingleNode("@id").getText();
        sc.setId(id);

        // title
        sc.setTitle(node.selectSingleNode("./name").getText());

        // description
        Node nd = node.selectSingleNode("./description");
        if (nd != null) {
            sc.setDescription(nd.getText());
        }

        // filter pattern
        nd = node.selectSingleNode("./pattern");
        if (nd != null) {
            sc.setMbeanPattern(nd.getText());
        }

        // x axis
        try {
            sc.setXAxis(node.selectSingleNode("./x/@col").getText());
            sc.setXType(Constants.getType(node.selectSingleNode("./x/@type").getText()));
        } catch(Exception e) {
            // If nothing set, use defaults
            sc.setXAxis(abscissColumn);
            sc.setXType(TIME);
        }

        // y axis
        try {
            sc.setYAxis(node.selectSingleNode("./y/@col").getText());
            sc.setYType(Constants.getType(node.selectSingleNode("./y/@type").getText()));
        } catch(Exception e) {
            // Nothing
        }

        series.put(id, sc);
    }

    /**
     * Parses the graph based on the XML configuration.
     */
    private void parseGraphs() {
        Iterator it = doc.selectNodes("//graph").iterator();
        while (it.hasNext()) {
            addGraph((Node) it.next());
        }
    }

    /**
     * Adds a graph.
     *
     * @param node  XML node describing the graph.
     */
    private void addGraph(final Node node) {
        GraphConfig gc = new GraphConfig(this);

        // ID
        String id = node.selectSingleNode("@id").getText();
        gc.setId(id);

        // title
        gc.setTitle(node.selectSingleNode("./title").getText());

        // series
        Iterator it = node.selectNodes("./serie").iterator();
        while (it.hasNext()) {
            Node nd = (Node) it.next();
            String sid = nd.selectSingleNode("@ref").getText();
            String legend = parseSerieLegend(nd);
            if (legend == null) {
                return;
            }
            gc.addSerie(sid, legend);
        }

        graphs.put(id, gc);
    }

    /**
     * Parses a legend.
     *
     * @param serie  XML node describing the legend.
     *
     * @return Parsed legend.
     */
    private String parseSerieLegend(final Node serie) {
        String serieId = serie.selectSingleNode("@ref").getText();
        String val = "";

        if (!serie.hasContent()) {
            // the legend is the name of the series
            Node node = doc.selectSingleNode("//serie-def[@id='" + serieId +"']/name");
            if (node == null) {
                System.err.println("Didn't found serie definition for : " + serieId);
                return null;
            }
            val = doc.selectSingleNode("//serie-def[@id='" + serieId +"']/name").getText();
        } else {
            // Agregate evaluated content
            Element element = (Element) serie;
            for (int i = 0, size = element.nodeCount(); i < size; i++) {
                Node n = element.node(i);
                String type = n.getName();
                if ("i".equals(type)) {
                    //n = n.selectSingleNode("./@link");
                    val += evalINode(n, serieId);
                } else {
                    val += n.getText();
                }
            }
        }

        // Remove extra blank characters
        val = val.replaceAll("[\\t\\n\\f\\r]", " ");
        val = val.trim();
        val = val.replaceAll("[ ]{2,}", " ");

        return val;
    }

    /**
     * Evaluate a node. It may contain:
     *
     *      - A ref: then it evaluates to the value associated to the ref.
     *      - A link: it is a XPath link, it evaluates to the value of the
     *                pointed to element.
     *      - A text: it evaluates to the contents of the element in
     *                the serie-def.
     *
     * If several attributes are present, the evaluation order is:
     *      1. text
     *      2. ref
     *      3. link
     *
     * @param inode     XML node.
     * @param serieId  Series ID corresponding to the node.
     *
     * @return Evaluated inode.
     */
    private String evalINode(final Node inode, final String serieId) {
        String value = "";
        boolean evaluated = false;

        // First look for the text attribute
        Node node = inode.selectSingleNode("./@text");
        if (node != null) {
            String elementName = node.getStringValue();
            Node n = doc.selectSingleNode("//serie-def[@id='" + serieId +"']/"+ elementName);
            if (n != null) {
                value = n.getText();
                evaluated = true;
            }
        }

        // Then look for the ref attribute
        node = inode.selectSingleNode("./@ref");
        if ((!evaluated) && (node != null)) {
            Node n = doc.selectSingleNode("//*[@id='"+ node.getStringValue()+"']");
            if (n != null) {
                value = n.getText();
                evaluated = true;
            }
        }

        // Finally look for the link attribute
        node = inode.selectSingleNode("./@link");
        if ((!evaluated) && (node != null)) {
            Node n = doc.selectSingleNode(node.getStringValue());
            if (node != null) {
                value = n.getText();
                evaluated = true;
            }
        }

        return value;
    }

    /**
     * Parses topmost properties.
     */
    private void parseParent() {
        Node node = doc.getRootElement();

        Node nd = node.selectSingleNode("./@separator");
        if (nd != null) {
            separator = nd.getText();
        }

        nd = node.selectSingleNode("./@absciss");
        if (nd != null) {
            abscissColumn = nd.getText();
        }

        nd = node.selectSingleNode("./mbean");
        if (nd != null) {
            mbeanColumn = nd.getText();
        }

        nd = node.selectSingleNode("./@timeformat");
        if (nd != null) {
            String df = nd.getText();
            if (df.toLowerCase(Locale.ENGLISH).equals("long")) {
                simpleDateFormat = null;
            } else {
                simpleDateFormat = new SimpleDateFormat(nd.getText());
            }
        } else {
            simpleDateFormat = new SimpleDateFormat(DEFAULT_DATE_FORMAT);
        }
    }

    /**
     * Map of series.
     */
    private Map series = new TreeMap();

    /**
     * Map of graphs.
     */
    private Map graphs = new TreeMap();

    /**
     * XML document.
     */
    private Document doc = null;

    /**
     * Field separator.
     */
    private String separator = DEFAULT_SEPARATOR;

    /**
     * Absciss column.
     */
    private String abscissColumn = DEFAULT_ABSCISS_COLUMN;

    /**
     * MBean column.
     */

    private String mbeanColumn = DEFAULT_MBEAN_COLUMN;

    /**
     * Date format.
     */
    private SimpleDateFormat simpleDateFormat = null;
}
